/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Class: WPS.Control.ObjectVisualizer
 *
 * The Object visualizer gives access to the legato map client. It is possible
 * to visualize GeoJSON objects with styles and set the visible area of the map via a bounding box
 * or a center point with a minimum and maximum scale.
 *
 * Inherits from: - <WPS.Control>
 *
 */
WPS.Control.ObjectVisualizer = OpenLayers.Class( WPS.Control, {

  /*
   * {<OpenLayers.Map>} the current used map
   */
  map: null,

  currentLayerId : 0,

  factory: null,

  /**
   * Function: initialize
   * Basic constructor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control
   * The following keys are supported
   * - map {<OpenLayers.Map>} The current map
   */
  initialize : function(options) {
    Legato.Util.Ensure.ensureExists(options.map, "map must be defined");
    OpenLayers.Control.prototype.initialize.apply(this, arguments);
    this.factory = new WPS.Result.Factory(options.map, null);
  },

  /**
   * Function: destroy
   *
   * See Also:
   * <WPS.Control>
   */
  destroy : function() {
    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /**
   * Function: setBoundingBox
   *
   * Sets the map extent to the bounding box with given coordinates.
   *
   * Parameters:
   * left - {Number} Minimum horizontal coordinate.
   * bottom - {Number} Minimum vertical coordinate.
   * right - {Number} Maximum horizontal coordinate.
   * top - {Number} Maximum vertical coordinate.
   */
  setBoundingBox : function(left, bottom, right, top) {
    Legato.Util.Ensure.ensureNumber(left, "The parameter for the minimum horizontal coordinate must be a number.");
    Legato.Util.Ensure.ensureNumber(bottom, "The parameter for the minimum vertical coordinate must be a number.");
    Legato.Util.Ensure.ensureNumber(right, "The parameter for the maximum horizontal coordinate must be a number.");
    Legato.Util.Ensure.ensureNumber(top, "The parameter for the maximum vertical coordinate must be a number.");
    var bbox = new OpenLayers.Bounds(left, bottom, right, top);
    this.map.zoomToExtent(bbox);
  },

  /**
   * Function: setCenterAndScale
   *
   * Sets the map extent based on the center point with given coordinates and the scale.
   *
   * Parameters:
   * x - {Number} Horizontal coordinate of the center point.
   * y - {Number} Vertical coordinate of the center point.
   * scale - {Number} Denominator value of scale. If null, the minimum possible scale will be used.
   */
  setCenterAndScale : function(x, y, scale) {
    Legato.Util.Ensure.ensureNumber(x, "The parameter for the horizontal coordinate must be a number.");
    Legato.Util.Ensure.ensureNumber(y, "The parameter for the vertical coordinate must be a number.");

    if (scale !== null) {
      Legato.Util.Ensure.ensureNumber(scale, "The parameter for the scale must be a number.");
    }

    //zoom map
    this.map.zoomToScale(scale);
    this.map.setCenter(new OpenLayers.LonLat(x, y));
  },

  /**
   * Function: visualize
   *
   * Shows the given objects with styles and map tips on the map. All the given objects will be in the
   * same layer. After adding the objects, the map client zooms to the newly added objects.
   *
   * Parameters:
   * layerName - {String} The name of the layer where the given geometry objects should be added. If this parameter
   * is null, then a new layer will be created and its name will be generated automatically. Otherwise,
   * the method looks for the layer with the given name. If the layer is found, then the objects will
   * be in that layer. Otherwise, a new layer with the given name is created.
   * objects - {Array} An array of geometry objects to add.
   * The following keys are supported:
   * - geometry {GeoJSON} Mandatory. A GeoJSON Object representing the geometry for visualizing in the map client.
   * - mapTip {String} Optional. If exists, the visualized object will have a map tip that is shown
   * when the mouse is over the geometry object.
   * - style {Object} Optional. This object contains styling attributes corresponding to SVG definition.
   * The following SVG styling attributes are supported:
   * - fill
   * - stroke
   * - stroke-width
   */
  visualize : function(layerName, objects) {
    var layer = this.getLayer(layerName);
    var zoomBounds = {
        left: Number.MAX_VALUE,
        bottom : Number.MAX_VALUE,
        right : Number.MIN_VALUE,
        top : Number.MIN_VALUE
    };
    for(var index = 0; index < objects.length; index++) {
      var geoJsonObject = objects[index];
      Legato.Util.Ensure.ensureExists(geoJsonObject.geometry, "geometry must be defined");
      var resultObject = this.createBasicResultObject(geoJsonObject.geometry);
      var symbolizerPart = this.setAndReturnBasicGraphicsInformation(resultObject, geoJsonObject.geometry);
      if (Legato.Lang.ObjectUtils.exists(geoJsonObject.style)) {
        this.setStyleInformation(geoJsonObject.style, symbolizerPart);
      }
      if (Legato.Lang.ObjectUtils.exists(geoJsonObject.mapTip)) {
        this.setMapTipInformation(geoJsonObject.mapTip, resultObject);
      }
      var marker = this.factory.createResult(resultObject);
      var executionContext = new WPS.Result.ExecutionContext(this.map);
      var feature = marker.createAndAddMarkerFeature(executionContext);

      var bounds = feature.geometry.getBounds();
      zoomBounds.left = Math.min(zoomBounds.left, bounds.left);
      zoomBounds.bottom = Math.min(zoomBounds.bottom, bounds.bottom);
      zoomBounds.right = Math.max(zoomBounds.right, bounds.right);
      zoomBounds.top = Math.max(zoomBounds.top, bounds.top);
    }

    if (objects.length > 0) {
      this.map.zoomToExtent(new OpenLayers.Bounds(zoomBounds.left, zoomBounds.bottom, zoomBounds.right, zoomBounds.top));
    }

    for(index = 0; index < this.map.layers.length; index++) {
      if (this.map.layers[index] == layer) {
          return;
      }
    }
    this.map.addLayer(layer);
  },

  getLayer : function (layerName) {
    if (!Legato.Lang.ObjectUtils.isNull(layerName)) {
      var layers = this.map.layers;
      for (var index = 0; index < layers.length; index++) {
        var layer = layers[index];
        if (layerName == layer.name) {
          return layer;
        }
      }
    }
    var layerOptions = {
      displayInLayerSwitcher : true,
      calculateInRange : function() {
        return true;
      },
      style : WPS.Feature.Style.ACTIVE
    };
    if (!Legato.Lang.ObjectUtils.isNull(layerName)) {
      return new OpenLayers.Layer.Vector(layerName, layerOptions);
    }
    else {
      this.currentLayerId++;
      return new OpenLayers.Layer.Vector("Result Marker Layer #" + this.currentLayerId, layerOptions);
    }
  },

  setStyleInformation : function(style, symbolizerPart) {
    fillParameters = ["fill"];
    strokeParameters = ["stroke", "stroke-width"];
    symbolizerPart.fill = {
        svgParameter : []
    };
    symbolizerPart.stroke = {
        svgParameter : []
    };
    this.setStyleParameters(style, symbolizerPart.fill, fillParameters);
    this.setStyleParameters(style, symbolizerPart.stroke, strokeParameters);
  },

  setStyleParameters : function(source, destination, parameters) {
    for (var index = 0; index < parameters.length; index++) {
      var paramName = parameters[index];
      if (Legato.Lang.ObjectUtils.exists(source[paramName])) {
        destination.svgParameter.push({
          name : paramName,
          content : [ source[paramName] ]
        });
      }
    }
  },

  setMapTipInformation : function (mapTip, resultObject) {
    resultObject.processOutputs.output[0].data.complexData.content[0].value.message = {
      value : {
        content : mapTip
      }
    };
  },

  setAndReturnBasicGraphicsInformation : function (resultObject, geoJsonObject) {
    var graphicsInformation = resultObject.processOutputs.output[0].data.complexData.content[0].value.geometry.value.symbolizer.value;
    switch (geoJsonObject.type) {
      case "Point":
        resultObject.processOutputs.output[0].data.complexData.content[0].value.geometry.value.symbolizer.value = {
          graphic : {
            externalGraphicOrMark : [ {} ]
          }
        };
        return resultObject.processOutputs.output[0].data.complexData.content[0].value.geometry.value.symbolizer.value.graphic.externalGraphicOrMark[0];
      case "LineString": case "Polygon":
        return graphicsInformation;
    }
  },

  createBasicResultObject : function(geoJsonObject) {
    var markerSchema = WPSPD.Result.Marker.ComplexFactory.prototype.schema;
    var basicResultObject = {
        processOutputs : {
          output : [{
            data : {
              complexData : {
                schema : markerSchema,
                content : [{
                  value : {
                    geometry : {
                      value : {
                        geometry : {
                          value : geoJsonObject
                        },
                        symbolizer : {
                          value : {

                          }
                        }
                      }
                    }
                  }
                }]
              }
            }
          }]
        }
      };
      return basicResultObject;
  },

  CLASS_NAME : "WPS.Control.ObjectVisualizer"
});

/**
 * Structure: lwpsc:ObjectVisualizer
 * XML based configuration for a <WPS.Control.ObjectVisualizer>.
 *
 * A valid configuration example for a ObjectVisualizer would be:
 * (start code)
 * <lwpsc:ObjectVisualizer>
 *   <map>
 *     <lb:Reference target="map" />
 *   </map>
 * </lwpsc:ObjectVisualizer>
 * (end)
 */
WPS.Control.ObjectVisualizer.Bean = WPS.Control.Bean.Extend('WPS.Control.ObjectVisualizer',
  WPS.Control.QName('ObjectVisualizer'),
  {
    _constructor : WPS.Control.ObjectVisualizer,
    options : {
      map : OpenLayers.Map
    }
  }
);
