/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

WPS.Component.AbstractGeometry = OpenLayers
    .Class(
        WPS.Component.AbstractInput,
        {
          inputPanelDiv : null,
          inputPanel : null,
          geoInputControl : null,
          geoModifyControl : null,
          featureSelector : null,
          map : null,
          layer : null,
          featureGeometrySelectionController : null,
          geometryType : null,
          initialize : function(id, model, map, options) {
            WPS.Component.AbstractInput.prototype.initialize.apply(this, arguments);
            this.geoInputControl.events.on({
              "activate" : this.geoInputControlActivated,
              "deactivate" : this.geoInputControlDeactivated,
              scope : this
            });
            this.geoModifyControl.events.on({
              "activate" : this.geoModifyControlActivated,
              scope : this
            });
            if (this.featureSelector) {
              this.featureSelector.events.on({
                "activate" : this.featureSelectorActivated,
                scope : this
              });
            }
          },
          destroy : function() {
            this.geoInputControl.events.un({
              "activate" : this.controlActivated,
              "deactivate" : this.controlDeactivated,
              scope : this
            });
            this.geoModifyControl.events.un({
              "activate" : this.geoModifyControlActivated,
              scope : this
            });
            if (this.featureSelector) {
              this.featureSelector.events.un({
                "activate" : this.featureSelectorActivated,
                scope : this
              });
            }
            this.unregisterLayerEvents();
            if (Legato.Lang.ObjectUtils.exists(this.map) && Legato.Lang.ObjectUtils.exists(this.layer)) {
              this.map.removeLayer(this.layer);
            }
            WPS.Component.AbstractInput.prototype.destroy.apply(this, arguments);
          },
          checkedChanged : function(event) {
            WPS.Component.AbstractInput.prototype.checkedChanged.call(this, event);
            var checked = (event.target || event.srcElement).checked;
            if (!checked) {
              this.geoInputControl.deactivate();
              this.geoModifyControl.deactivate();
              if (this.featureSelector) {
                this.featureSelector.deactivate();
              }
            }
          },

          geoModifyControlActivated : function() {
            this.geoInputControl.deactivate();
            if (this.featureSelector) {
              this.featureSelector.deactivate();
            }
            this.onActivation(true);
          },

          geoInputControlActivated : function() {
            this.geoModifyControl.deactivate();
            if (this.featureSelector) {
              this.featureSelector.deactivate();
            }
            this.onActivation(true);
          },

          geoInputControlDeactivated : function() {
            this.onDeactivation(true);
          },

          featureSelectorActivated : function() {
            this.geoModifyControl.deactivate();
            this.geoInputControl.deactivate();
            this.onActivation(true);
          },

          initializeInputs : function() {
            var layerOptions = {
              displayInLayerSwitcher : false,
              legatoIgnore : true,
              // indicate that the temp vector layer will never be out of range
              // without this, resolution properties must be specified at the
              // map-level for this temporary layer to init its resolutions
              // correctly
              calculateInRange : function() {
                return true;
              },
              isFeatureLayer : true
            };
            this.layer = new OpenLayers.Layer.Vector(this.model.identifier + 'Input', layerOptions);
            this.layer.style = this.getActiveStyle();
            this.map.addLayer(this.layer);

            // Always ensure that the draw layer will be on top
            this.layer.setZIndex(this.map.Z_INDEX_BASE.Feature - 1);

            this.registerLayerEvents();

            this.inputPanelDiv = document.createElement('div');
            OpenLayers.Element.addClass(this.inputPanelDiv, 'olControlPanel');
            this.container.appendChild(this.inputPanelDiv);

            var panelOptions = {
              div : this.inputPanelDiv
            };

            this.inputPanel = new OpenLayers.Control.Panel(panelOptions);
            this.map.addControl(this.inputPanel);
            this.initializeGeoInputControl();
            this.initializeGeoModifyControl();
            this.initializeFeatureSelector();
            return this.inputPanelDiv;
          },

          onGeometryChange : function(event) {
            switch (event.type) {
            case "featuresadded":
              this.updateModelGeometries();
              break;
            case "featuresremoved":
              this.updateModelGeometries();
              break;
            case "featuremodified":
              this.updateModelGeometries();
              break;
            default:
              break;
            // do nothing
            }
          },

          activate : function() {
            if (!(this.geoInputControl.active || this.geoModifyControl.active || (this.featureSelector && this.featureSelector.active))) {
              if(this.featureSelector
                  && Legato.OpenLayers.Map.Util.getSelectedLayers(this.map.layers)
                  && Legato.OpenLayers.Map.Util.getSelectedLayers(this.map.layers).length > 0){
                this.featureSelector.activate();
              } else {
                this.geoInputControl.activate();
              }

              this.onActivation();
            }
          },

          deactivate : function() {
            this.geoInputControl.deactivate();
            this.geoModifyControl.deactivate();
            if (this.featureSelector) {
              this.featureSelector.deactivate();
            }
            this.onDeactivation();
          },

          initializeGeoInputControl : function() {
            var control = this.createGeoInputControl();
            this.inputPanel.addControls([ control ]);
            this.geoInputControl = control;
          },

          initializeGeoModifyControl : function() {
            var control = this.createGeoModifyControl();
            this.inputPanel.addControls([ control ]);
            this.geoModifyControl = control;
          },

          initializeFeatureSelector : function() {

            if (Legato.Lang.ObjectUtils.exists(this.featureGeometrySelectionController)) {

              var that = this;

              var geometryType = that.geometryType;

              var layerPredicate = new Legato.Predicate.And([ new Legato.Layer.Predicate.SelectedLayer(), new Legato.Layer.Predicate.LayerWithAttribute({
                name : 'featureLayer',
                value : 'true'
              }), new Legato.Layer.Predicate.LayerWithAttribute({
                name : 'geometryType',
                predicate : new Legato.Predicate.Contains(geometryType)
              }) ]);

              var featureGeometrySelectionController = that.featureGeometrySelectionController;
              var format = new OpenLayers.Format.GeoJSON();
              var model = that.model;

              var featureSelectionController = function(layers, selectionGeometry, mode, callbacks) {
                // Create array of layer ids
                var layerIds = [];
                for ( var index = 0; index < layers.length; index++) {
                  layerIds.push(Legato.Lang.ObjectUtils.exists(layers[index].layerId) ? layers[index].layerId : layers[index].name);
                }

                var geometry = format.extract.geometry.call(format, selectionGeometry);
                var existingGeometry = format.extract.geometry.call(format, model.getValue());

                var eventCtrlDown = mode === OpenLayers.Handler.MOD_CTRL ? true : false;
                var eventShiftDown = mode === OpenLayers.Handler.MOD_SHIFT ? true : false;

                featureGeometrySelectionController(layerIds, geometryType, geometry, existingGeometry, eventCtrlDown, eventShiftDown, callbacks);
              };

              var callback = function(geometries) {
                if(!Legato.Lang.ObjectUtils.isNullOrUndefined(geometries) && geometries.type === geometryType){
                  model.setValue(format.parseGeometry(geometries));
                } else {
                  if (geometries.type.substring(0, 5) === 'Multi') {
                    throw new Legato.Lang.Exception(
                      WPS.I18n.getMessage('WPS.AbstractGeometry.InvalidMultiGeometryType')
                    );
                  } else {
                    throw new Legato.Lang.Exception(
                      WPS.I18n.getMessage('WPS.AbstractGeometry.InvalidSelectionGeometryType', {invalidGeometryType : geometries.type, validGeometryType : geometryType})
                    );
                  }
                }
              };

              var featureSelector = new Legato.Control.FeatureSelector({
                layerPredicate : layerPredicate,
                controller : featureSelectionController,
                callback : callback,
                title : WPS.I18n.getMessage('WPS.AbstractGeometry.Select')
              });
              this.inputPanel.addControls([ featureSelector ]);
              this.featureSelector = featureSelector;

            } else {
              this.featureSelector = null;
            }
          },

          createGeoInputControl : function() {
            throw new Legato.Lang.Exception(this.CLASS_NAME + ' must implement the createGeoInputControl function from from abstract class '
                + WPS.Component.AbstractGeometry.CLASS_NAME);
          },

          createGeoModifyControl : function() {
            throw new Legato.Lang.Exception(this.CLASS_NAME + ' must implement the createGeoModifyControl function from from abstract class '
                + WPS.Component.AbstractGeometry.CLASS_NAME);
          },

          registerLayerEvents : function() {
            this.layer.events.on({
              'featuresadded' : this.onGeometryChange,
              'featuresremoved' : this.onGeometryChange,
              'featuremodified' : this.onGeometryChange,
              scope : this
            });
          },

          unregisterLayerEvents : function() {
            this.layer.events.un({
              'featuresadded' : this.onGeometryChange,
              'featuresremoved' : this.onGeometryChange,
              'featuremodified' : this.onGeometryChange,
              scope : this
            });
          },

          getLayerGeometry : function() {
            var geometries = this.getLayerGeometries();
            if (Legato.Lang.ObjectUtils.isArray(geometries)) {
              if (geometries.length > 0) {
                return geometries[0];
              } else {
                return null;
              }
            } else {
              return null;
            }
          },

          getLayerGeometries : function() {
            var features = this.layer.features;
            if (Legato.Lang.ObjectUtils.isArray(features)) {
              var geometries = [];
              for ( var index = 0; index < features.length; index++) {
                if (Legato.Lang.ObjectUtils.exists(features[index].geometry)) {
                  var geometry = features[index].geometry;
                  // If it is a multi geometry, push its components
                  if ((geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiPoint' || geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiLineString' || geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiPolygon')
                      && Legato.Lang.ObjectUtils.isArray(geometry.components)) {
                    for ( var jndex = 0; jndex < geometry.components.length; jndex++) {
                      if (Legato.Lang.ObjectUtils.exists(geometry.components[jndex])) {

                        geometries.push(geometry.components[jndex]);
                      }
                    }
                  }
                  // If it is a singular geometry, push it as is
                  else {
                    geometries.push(geometry);
                  }
                }
              }
              return geometries;
            } else {
              return null;
            }
          },

          setLayerGeometry : function(geometry) {
            if (Legato.Lang.ObjectUtils.exists(geometry)) {
              this.setLayerGeometries([ geometry ]);
            } else {
              this.setLayerGeometries(null);
            }
          },

          setLayerGeometries : function(geometries) {
            if (Legato.Lang.ObjectUtils.isArray(geometries)) {
              var oldFeatures = this.layer.features.slice();
              this.layer.destroyFeatures(null, {
                silent : true
              });
              var newFeatures = [];
              for ( var index = 0; index < geometries.length; index++) {
                var feature;
                if (index < oldFeatures.length) {
                  // Reuse an old feature object
                  // It was like this before, I'm afraid to change it
                  feature = oldFeatures[index];
                  feature.geometry = geometries[index];
                  feature.id = OpenLayers.Util.createUniqueID(feature.CLASS_NAME + "_");
                } else {
                  // Create a new feature object
                  feature = new OpenLayers.Feature.Vector(geometries[index]);
                }
                newFeatures.push(feature);
              }
              this.layer.addFeatures(newFeatures, {
                silent : true
              });
            } else {
              // Destroy old features
              this.layer.destroyFeatures(null, {
                silent : true
              });
            }
          },
          updateLayerGeometries : function()
          {
            var geometry = this.model.getValue();
            this.layer.geometry = geometry;
            if (geometry === null)
            {
              this.setLayerGeometries(null);
            }
            else
            {
              if (geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiPoint' || geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiLineString' || geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiPolygon')
              {
                this.setLayerGeometries(geometry.components);
              }
              else
              {
                this.setLayerGeometries([geometry]);
              }
            }
          },
          updateModelGeometries : function()
          {
            var geometry = this.layer.geometry;
            if (this.layer.geometry === undefined || this.layer.geometry === null)
            {
              this.layer.geometry = this.getLayerGeometry();
            }
            else
            {
              if (geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiPoint' || geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiLineString' || geometry.CLASS_NAME === 'OpenLayers.Geometry.MultiPolygon')
              {
                geometry.removeComponents(geometry.components);
                geometry.addComponents(this.getLayerGeometries());
              }
              else
              {
                this.layer.geometry = this.getLayerGeometry();
              }
            }
            this.model.setValue(this.layer.geometry);
          },
          onModelValueChange : function(event) {
            this.updateLayerGeometries();
          },

          redrawFeaturesWithStyle : function(style) {
            var features = this.layer.features;
            if (features) {
              for ( var i = 0; i < features.length; i++) {
                this.layer.drawFeature(features[i], style);
              }
            }
          },

          onHoverIn : function() {
            if (WPS.Component.STATUS_INACTIVE === this.status) {
              this.redrawFeaturesWithStyle(this.getInactiveHoverStyle());
            } else {
              this.redrawFeaturesWithStyle(this.getActiveHoverStyle());
            }
            WPS.Component.AbstractInput.prototype.onHoverIn.call(this);
          },

          onHoverOut : function() {
            if (WPS.Component.STATUS_ACTIVE_HOVER === this.status) {
              this.redrawFeaturesWithStyle(this.getActiveStyle());
            } else {
              this.redrawFeaturesWithStyle(this.getInactiveStyle());
            }
            WPS.Component.AbstractInput.prototype.onHoverOut.call(this);
          },

          onActivation : function(triggerEvent) {
            this.redrawFeaturesWithStyle(this.getActiveStyle());
            WPS.Component.AbstractInput.prototype.onActivation.call(this, triggerEvent);
          },

          onDeactivation : function(triggerEvent) {
            WPS.Component.AbstractInput.prototype.onDeactivation.call(this, triggerEvent);
            this.redrawFeaturesWithStyle(this.getInactiveStyle());
          },

          getNamedStyle : function(name) {
            if (!Legato.Lang.ObjectUtils.exists(this.styles)) {
             return null;
            }
            for(var i = 0; i < this.styles.length; i++){
              if(this.styles[i].name === name){
                return this.styles[i].styles;
              }
            }
            return null;
          },

          getDefaultStyle : function(){
            return null;
          },

          getActiveStyle : function(){
            var style = OpenLayers.Util.extend({}, WPS.Feature.Style.ACTIVE);
            return OpenLayers.Util.extend(style, this.getDefaultStyle());
          },

          getInactiveStyle : function(){
            var style = OpenLayers.Util.extend({}, WPS.Feature.Style.INACTIVE);
            return OpenLayers.Util.extend(style, this.getDefaultStyle());
          },

          getInactiveHoverStyle : function(){
            var style = OpenLayers.Util.extend({}, WPS.Feature.Style.INACTIVE_HOVER);
            return OpenLayers.Util.extend(style, this.getDefaultStyle());
          },

          getActiveHoverStyle : function(){
            var style = OpenLayers.Util.extend({}, WPS.Feature.Style.ACTIVE_HOVER);
            return OpenLayers.Util.extend(style, this.getDefaultStyle());
          },

          CLASS_NAME : 'WPS.Component.AbstractGeometry'
        });