/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

WPS.Component.AbstractGeometry = OpenLayers.Class(WPS.Component, {
  inputPanelDiv : null,
  inputPanel : null,
  geoInputControl : null,
  geoModifyControl : null,
  featureSelector : null,
  map : null,
  layer : null,
  featureGeometrySelectionController : null,
  geometryType : null,
  initialize : function(id, model, map, options) {
    WPS.Component.prototype.initialize.apply(this, arguments);
    this.geoInputControl.events.on({
      "activate" : this.geoInputControlActivated,
      "deactivate" : this.geoInputControlDeactivated,
      scope : this
    });
    this.geoModifyControl.events.on({
      "activate" : this.geoModifyControlActivated,
      scope : this
    });
    if (this.featureSelector) {
      this.featureSelector.events.on({
        "activate" : this.featureSelectorActivated,
        scope : this
      });
    }
  },
  destroy : function() {
    this.geoInputControl.events.un({
      "activate" : this.controlActivated,
      "deactivate" : this.controlDeactivated,
      scope : this
    });
    this.geoModifyControl.events.un({
      "activate" : this.geoModifyControlActivated,
      scope : this
    });
    if (this.featureSelector) {
      this.featureSelector.events.un({
        "activate" : this.featureSelectorActivated,
        scope : this
      });
    }
    this.unregisterLayerEvents();
    if (Legato.Lang.ObjectUtils.exists(this.map) && Legato.Lang.ObjectUtils.exists(this.layer)) {
      this.map.removeLayer(this.layer);
    }
    WPS.Component.prototype.destroy.apply(this, arguments);
  },
  checkedChanged : function(event) {
    WPS.Component.prototype.checkedChanged.call(this, event);
    var checked = (event.target || event.srcElement).checked;
    if (!checked) {
      this.geoInputControl.deactivate();
      this.geoModifyControl.deactivate();
      if (this.featureSelector) {
        this.featureSelector.deactivate();
      }
    }
  },

  geoModifyControlActivated : function() {
    this.geoInputControl.deactivate();
    if (this.featureSelector) {
      this.featureSelector.deactivate();
    }
    this.onActivation(true);
  },

  geoInputControlActivated : function() {
    this.geoModifyControl.deactivate();
    if (this.featureSelector) {
      this.featureSelector.deactivate();
    }
    this.onActivation(true);
  },

  geoInputControlDeactivated : function() {
    this.onDeactivation(true);
  },

  featureSelectorActivated : function() {
    this.geoModifyControl.deactivate();
    this.geoInputControl.deactivate();
    this.onActivation(true);
  },

  initializeInputs : function() {
    var layerOptions = {
      displayInLayerSwitcher : false,
      // indicate that the temp vector layer will never be out of range
      // without this, resolution properties must be specified at the
      // map-level for this temporary layer to init its resolutions
      // correctly
      calculateInRange : function() {
        return true;
      },
      isFeatureLayer : true
    };
    this.layer = new OpenLayers.Layer.Vector(this.model.identifier + 'Input', layerOptions);
    this.layer.style = WPS.Feature.Style.ACTIVE;
    this.map.addLayer(this.layer);

    // Always ensure that the draw layer will be on top
    this.layer.setZIndex(this.map.Z_INDEX_BASE.Feature - 1);

    this.registerLayerEvents();

    this.inputPanelDiv = document.createElement('div');
    OpenLayers.Element.addClass(this.inputPanelDiv, 'olControlPanel');
    this.container.appendChild(this.inputPanelDiv);

    var panelOptions = {
      div : this.inputPanelDiv
    };

    this.inputPanel = new OpenLayers.Control.Panel(panelOptions);
    this.map.addControl(this.inputPanel);
    this.initializeGeoInputControl();
    this.initializeGeoModifyControl();
    this.initializeFeatureSelector();
    return this.inputPanelDiv;
  },

  onGeometryChange : function(event) {
    var feature;
    switch (event.type) {
    case "featuresadded":
      feature = event.features[0];
      this.model.setValue(feature.geometry);
      break;
    case "featuresremoved":
      this.model.setValue(null);
      break;
    case "featuremodified":
      feature = event.feature;
      this.model.setValue(feature.geometry);
      break;
    default:
      break;
      // do nothing
    }
  },

  activate : function() {
    if (!(this.geoInputControl.active || this.geoModifyControl.active || (this.featureSelector && this.featureSelector.active))) {
      this.geoInputControl.activate();
      this.onActivation();
    }
  },

  deactivate : function() {
    this.geoInputControl.deactivate();
    this.geoModifyControl.deactivate();
    if (this.featureSelector) {
      this.featureSelector.deactivate();
    }
    this.onDeactivation();
  },

  initializeGeoInputControl : function() {
    var controlOptions = {
      mode : Legato.Control.DrawFeature.MODE_REPLACE_FEATURE,
      type : OpenLayers.Control.TYPE_TOGGLE
    };
    var control = this.createGeoInputControl(controlOptions);
    this.inputPanel.addControls([ control ]);
    this.geoInputControl = control;
  },

  initializeGeoModifyControl : function() {
    var controlOptions = {
      mode : Legato.Control.ModifyFeature.MODE_RESHAPE
    };
    var control = this.createGeoModifyControl(controlOptions);
    this.inputPanel.addControls([ control ]);
    this.geoModifyControl = control;
  },

  initializeFeatureSelector : function() {

    if (Legato.Lang.ObjectUtils.exists(this.featureGeometrySelectionController)) {

      var that = this;

      var geometryType = that.geometryType;

      var layerPredicate = new Legato.Predicate.And([ new Legato.Layer.Predicate.SelectedLayer(), new Legato.Layer.Predicate.LayerWithAttribute({
        name : 'featureLayer',
        value : 'true'
      }), new Legato.Layer.Predicate.LayerWithAttribute({
        name : 'geometryType',
        predicate : new Legato.Predicate.Contains(geometryType)
      }) ]);

      var featureGeometrySelectionController = that.featureGeometrySelectionController;
      var format = new OpenLayers.Format.GeoJSON();

      var featureSelectionController = function(layers, selectionGeometry, mode, callbacks) {
        // Create array of layer ids
        var layerIds = [];
        for ( var index = 0; index < layers.length; index++) {
          layerIds.push(Legato.Lang.ObjectUtils.exists(layers[index].layerId) ? layers[index].layerId : layers[index].name);
        }

        var geometry = format.extract.geometry.call(format, selectionGeometry);
        featureGeometrySelectionController(layerIds, geometryType, 1, geometry, callbacks);
      };

      var model = that.model;

      var callback = function(geometries) {
        if (geometries && geometries.length >= 0 && geometries[0]) {
          // REVIEW AV Format neu erstellen
          var geometry = format.parseGeometry(geometries[0]);
          model.setValue(geometry);
        }
      };

      var featureSelector = new Legato.Control.FeatureSelector({
        layerPredicate : layerPredicate,
        controller : featureSelectionController,
        callback : callback,
        //i18n
        title: 'Zur Auswahl von Geometrien. Bitte vorher das gewünschte Thema markieren.'
      });
      this.inputPanel.addControls([ featureSelector ]);
      this.featureSelector = featureSelector;

    } else {
      this.featureSelector = null;
    }
  },

  createGeoInputControl : function(controlOptions) {
    throw new Legato.Lang.Exception(this.CLASS_NAME +
      ' must implement the createGeoInputControl function from from abstract class ' +
      WPS.Component.AbstractGeometry.CLASS_NAME);
  },

  createGeoModifyControl : function(controlOptions) {
    throw new Legato.Lang.Exception(this.CLASS_NAME +
        ' must implement the createGeoModifyControl function from from abstract class ' +
        WPS.Component.AbstractGeometry.CLASS_NAME);
  },

  registerLayerEvents : function() {
    this.layer.events.on({
      'featuresadded' : this.onGeometryChange,
      'featuresremoved' : this.onGeometryChange,
      'featuremodified' : this.onGeometryChange,
      scope : this
    });
  },

  unregisterLayerEvents : function() {
    this.layer.events.un({
      'featuresadded' : this.onGeometryChange,
      'featuresremoved' : this.onGeometryChange,
      'featuremodified' : this.onGeometryChange,
      scope : this
    });
  },

  onModelValueChange : function(event) {
    var silentOption = {
      silent : true
    };
    var layerFeatures = this.layer.features;
    var feature = null;
    if (layerFeatures && layerFeatures.length > 0) {
      feature = layerFeatures[0];
    }
    this.layer.destroyFeatures(null, silentOption);
    if (this.model.getValue() !== null) {
      if (feature !== null) {
        feature.geometry = this.model.getValue();
      } else {
        feature = new OpenLayers.Feature.Vector(this.model.getValue());
      }
      this.layer.addFeatures(feature, silentOption);
    }
  },

  redrawFeaturesWithStyle : function(style) {
    var features = this.layer.features;
    if (features) {
      for ( var i = 0; i < features.length; i++) {
        this.layer.drawFeature(features[i], style);
      }
    }
  },

  onHoverIn : function() {
    if (WPS.Component.STATUS_INACTIVE === this.status) {
      this.redrawFeaturesWithStyle(WPS.Feature.Style.INACTIVE_HOVER);
    } else {
      this.redrawFeaturesWithStyle(WPS.Feature.Style.ACTIVE_HOVER);
    }
    WPS.Component.prototype.onHoverIn.call(this);
  },

  onHoverOut : function() {
    if (WPS.Component.STATUS_ACTIVE_HOVER === this.status) {
      this.redrawFeaturesWithStyle(WPS.Feature.Style.ACTIVE);
    } else {
      this.redrawFeaturesWithStyle(WPS.Feature.Style.INACTIVE);
    }
    WPS.Component.prototype.onHoverOut.call(this);
  },

  onActivation : function(triggerEvent) {
    this.redrawFeaturesWithStyle(WPS.Feature.Style.ACTIVE_HOVER);
    WPS.Component.prototype.onActivation.call(this, triggerEvent);
  },

  onDeactivation : function(triggerEvent) {
    WPS.Component.prototype.onDeactivation.call(this, triggerEvent);
    this.redrawFeaturesWithStyle(WPS.Feature.Style.INACTIVE);
  },

  CLASS_NAME : 'WPS.Component.AbstractGeometry'
});