/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

Legato.Control.SketchLayerEditor = OpenLayers
    .Class(
        Legato.Control,
        {
          /**
           * Property: map {OpenLayers.Map} the current used map
           */
          map : null,
          /**
           * Property: layer {OpenLayers.Layer.Vector}
           */
          layer : null,
          /**
           * Property: inputPanel {Legato.Style} a list of named style
           * definitions
           */
          styles : null,
          /**
           * Property: div {DOMElement} parent element
           */
          div : null,
          /**
           * Property: container {DOMElement} where this sketch layer editor will paint
           * its controls
           */
          container : null,
          /**
           * Property: controlsPanel {OpenLayers.Control.Panel} the main control
           * panel for all input controls
           */
          controlsPanel : null,
          executeCallback : null,
          cancelCallback : null,

          /*
           * Function: initialize
           *
           * See Also: <OpenLayer.Control>
           */
          initialize : function(options) {
            OpenLayers.Control.prototype.initialize.apply(this, arguments);
          },

          /*
           * Function: destroy
           *
           * See Also: <OpenLayer.Control>
           */
          destroy : function() {
            OpenLayers.Control.prototype.destroy.apply(this, arguments);
          },

          draw : function() {
            OpenLayers.Control.prototype.draw.apply(this, arguments);
            return this.div;
          },
          /*
           * Returns a merged style map of configured style definitions and
           * default styles.
           */
          getStyles : function() {
            var defaultSketchStyles = Legato.Feature.Sketch.style['default'];

            if (this.styles) {
              for ( var i = 0; i < this.styles.length; i++) {
                var configuredStyle = this.styles[i];
                var configuredStyleName = configuredStyle.name.toUpperCase();
                if(defaultSketchStyles[configuredStyleName]){
                  OpenLayers.Util.extend(defaultSketchStyles[configuredStyleName], configuredStyle.styles);
                }
              }
            }

            return defaultSketchStyles;
          },

          execute : function(onExecute, onCancel) {
            this.executeCallback = onExecute;
            this.cancelCallback = onCancel;

            if (this.layer !== null) {
              return;
            }
            var layerOptions = {
              calculateInRange : function() {
                return true;
              },
              legatoIgnore : true,
              isFeatureLayer : true
            };
            var layerName = Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.LayerName');
            this.layer = new OpenLayers.Layer.Vector(layerName, layerOptions);
            this.map.addLayer(this.layer);
            this.createForm(onExecute, onCancel);
          },

          createForm : function() {
            if (this.container === null) {
              this.container = document.createElement('div');
              this.container.className = 'sketchLayerEditorContainer';
            }
            var titleElement = document.createElement('div');
            titleElement.className = 'sketchLayerEditorTitle';
            titleElement.innerHTML = Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Title');
            this.container.appendChild(titleElement);

            var abstractElement = document.createElement('div');
            abstractElement.className = 'sketchLayerEditorAbstract';
            abstractElement.innerHTML = Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Abstract');
            this.container.appendChild(abstractElement);

            this.controls = [];
            var controlsContainer = document.createElement('div');
            controlsContainer.className = 'sketchLayerEditorControlsContainer';
            this.container.appendChild(controlsContainer);

            var controlsPanelTitleElement = document.createElement('div');
            controlsPanelTitleElement.className = 'sketchLayerEditorControlsPanelTitle';
            controlsPanelTitleElement.innerHTML = Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.ControlsPanel.Title');
            controlsContainer.appendChild(controlsPanelTitleElement);

            var controlsPanelDiv = document.createElement('div');
            OpenLayers.Element.addClass(controlsPanelDiv, 'olControlPanel');
            controlsContainer.appendChild(controlsPanelDiv);

            var drawText = this.createDrawControl('TEXT', Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Text'),
                Legato.Control.DrawText);
            var drawPoint = this.createDrawControl('POINT',
                Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Point'),
                Legato.Control.DrawPoint);
            var drawMultiLineString = this.createDrawControl('LINE',
                Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Line'),
                Legato.Control.DrawMultiLineString);
            var drawPolygon = this.createDrawControl('POLYGON',
                Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Polygon'),
                Legato.Control.DrawPolygon);
            var modifyFeature = new Legato.Control.ModifyMultiPolygon(
                this.layer, {
                  title : Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Edit'),
                  mode : Legato.Control.ModifyFeature.MODE_RESHAPE,
                  type : OpenLayers.Control.TYPE_TOOL
                });
            var removeFeature = new OpenLayers.Control.DeleteFeature(
                this.layer,
                this.onDeleteFeatureSelect,
                this.onDeleteFeatureUnselect,
                {
                  title : Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Delete'),
                  type : OpenLayers.Control.TYPE_TOOL
                });

            var panelOptions = {
              div : controlsPanelDiv,
              defaultControl : drawPoint
            };

            this.controlsPanel = new OpenLayers.Control.Panel(panelOptions);
            this.controlsPanel
                .addControls([ drawText, drawPoint, drawMultiLineString,
                    drawPolygon, modifyFeature, removeFeature ]);

            for ( var i = 0; i < this.controlsPanel.controls.length; i++) {
              this.controlsPanel.controls[i].events.on({
                'activate' : this.onActivate,
                'deactivate' : this.onDeactivate,
                scope : this
              });
            }
            this.map.addControl(this.controlsPanel);

            var buttonsElement = document.createElement('div');
            buttonsElement.className = 'sketchLayerEditorButtons';
            this.container.appendChild(buttonsElement);

            var executeButton = document.createElement('input');
            executeButton.type = 'button';
            executeButton.value = Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Execute');
            executeButton.onclick = OpenLayers.Function.bindAsEventListener(
                this.executeButtonClicked, this);
            buttonsElement.appendChild(executeButton);

            var cancelButton = document.createElement('input');
            cancelButton.type = 'button';
            cancelButton.value = Legato.I18n.getMessage('Legato.Control.SketchLayerEditor.Cancel');
            cancelButton.onclick = OpenLayers.Function.bindAsEventListener(
                this.cancelButtonClicked, this);
            buttonsElement.appendChild(cancelButton);

            this.div.appendChild(this.container);
          },

          createDrawControl : function(type, title, typeClass) {
            var featureStyles = this.getStyles();
            var style = featureStyles[type];
            var attributes = this
                .createAttributesMap(type);
            return new typeClass(this.layer, {
              title : title,
              type : OpenLayers.Control.TYPE_TOOL,
              style : style,
              attributes : attributes
            });
          },

          createAttributesMap : function(type) {
            var attributeList = Legato.Feature.Sketch.Attributes[type];
            var attributesMap = {"x-type" : type};
            for ( var i = 0; i < attributeList.length; i++) {
              attributesMap[attributeList[i]] = null;
            }
            return attributesMap;
          },

          onDeactivate : function(event) {
            this.events.triggerEvent('deactivate', {
              component : this
            });
          },

          onActivate : function(event) {
            this.events.triggerEvent('activate', {
              component : this
            });
          },

          activate : function() {
            if (this.controlsPanel) {
              this.controlsPanel.activate();
            }
          },

          deactivate : function() {
            if (this.controlsPanel) {
              for ( var i = 0; i < this.controlsPanel.controls.length; i++) {
                this.controlsPanel.controls[i].deactivate();
              }
              this.controlsPanel.redraw();
            }
          },
          executeButtonClicked : function() {
            var sketchLayerAsGeoJSON = this.getSketchLayerAsGeoJSON();
            this.executeCallback(sketchLayerAsGeoJSON);
          },

          cancelButtonClicked : function() {
            this.cancelCallback();
          },

          onDeleteFeatureSelect: function(feature) {
            var type = feature.attributes['x-type'];
            var selectStyles = Legato.Feature.Sketch.style.select;
            OpenLayers.Util.extend(feature.style, selectStyles[type]);
            this.layer.redraw();
          },

          onDeleteFeatureUnselect: function(feature) {
              var type = feature.attributes['x-type'];
              var selectStyles = Legato.Feature.Sketch.style['default'];
              OpenLayers.Util.extend(feature.style, selectStyles[type]);
              this.layer.redraw();
          },

          getSketchLayerAsGeoJSON : function() {
            var writer = new OpenLayers.Format.GeoJSON();
            for ( var i = 0; i < this.layer.features.length; i++) {
              var feature = this.layer.features[i];
              var style = feature.style;
              var featureAttributes = feature.attributes;
              /* jslint forin:true */
              for ( var attribute in featureAttributes) {
                if (featureAttributes.hasOwnProperty(attribute)) {
                  var styleAttribute = Legato.Feature.Sketch.AttributeStyleMap[attribute];
                  featureAttributes[attribute] = style[styleAttribute];
               }
              }
              /* jslint forin:false */
            }
            return writer.write(this.layer.features, true);
          },

          reset : function() {
            if (!Legato.Lang.ObjectUtils.isNullOrUndefined(this.layer)
                && !Legato.Lang.ObjectUtils.isNullOrUndefined(this.container)) {
              this.deactivate();
              this.div.removeChild(this.container);
              this.map.removeLayer(this.layer);
              this.map.removeControl(this.controlsPanel);
              this.controlsPanel.destroy();
              this.controlsPanel = null;
              this.layer = null;
              this.container = null;
            }
          },

          CLASS_NAME : "Legato.Control.SketchLayerEditor"
        });

Legato.Control.SketchLayerEditor.Bean = Legato.Control.Bean.Extend(
    'Legato.Control.SketchLayerEditor', Legato.Control
        .QName('SketchLayerEditor'), {
      _constructor : Legato.Control.SketchLayerEditor,
      options : {
        map : OpenLayers.Map,
        div : Legato.Lang.Element,
        styles : Legato.Beans.Object.List()
      }
    });