/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @requires OpenLayers/Renderer/Elements.js
 */

/**
 * Class: OpenLayers.Renderer.SVG
 *
 * Inherits:
 *  - <OpenLayers.Renderer.Elements>
 */
OpenLayers.Renderer.Div = OpenLayers.Class(OpenLayers.Renderer.Elements, {

    rootNodeId: 'legato:renderer:emptyDiv:root',


    /**
     * Constructor: OpenLayers.Renderer.SVG
     *
     * Parameters:
     * containerID - {String}
     */
    initialize: function(containerID) {
        if (!this.supported()) {
            return;
        }
        OpenLayers.Renderer.Elements.prototype.initialize.apply(this,
                                                                arguments);
    },

    /**
     * APIMethod: destroy
     */
    destroy: function() {
        OpenLayers.Renderer.Elements.prototype.destroy.apply(this, arguments);
    },

    /**
     * APIMethod: supported
     *
     * Returns:
     * {Boolean} Whether or not the browser supports the SVG renderer
     */
    supported: function() {
        return true;
    },

    /**
     * Method: setExtent
     *
     * Parameters:
     * extent - {<OpenLayers.Bounds>}
     * resolutionChanged - {Boolean}
     *
     * Returns:
     * {Boolean} true to notify the layer that the new extent does not exceed
     *     the coordinate range, and the features will not need to be redrawn.
     *     False otherwise.
     */
    setExtent: function(extent, resolutionChanged) {
      OpenLayers.Renderer.Elements.prototype.setExtent.apply(this, arguments);
      return true;
    },

    /**
     * Method: getNodeType
     *
     * Parameters:
     * geometry - {<OpenLayers.Geometry>}
     * style - {Object}
     *
     * Returns:
     * {String} The corresponding node type for the specified geometry
     */
    getNodeType: function(geometry, style) {
        return geometry.CLASS_NAME;
    },

    /**
     * Method: setStyle
     * Use to set all the style attributes to a SVG node.
     *
     * Takes care to adjust stroke width and point radius to be
     * resolution-relative
     *
     * Parameters:
     * node - {SVGDomElement} An SVG element to decorate
     * style - {Object}
     * options - {Object} Currently supported options include
     *                              'isFilled' {Boolean} and
     *                              'isStroked' {Boolean}
     */
    setStyle: function(node, style, options) {
      return node;
    },

    /**
     * Method: dashStyle
     *
     * Parameters:
     * style - {Object}
     * widthFactor - {Number}
     *
     * Returns:
     * {String} A SVG compliant 'stroke-dasharray' value
     */
    dashStyle: function(style, widthFactor) {
      return 'none';
    },

    /**
     * Method: createNode
     *
     * Parameters:
     * type - {String} Kind of node to draw
     * id - {String} Id for node
     *
     * Returns:
     * {DOMElement} A new node of the given type and id
     */
    createNode: function(type, id) {
      var element       = document.createElement('div');
      element.id        = id;
      element.className = type;
      return element;
    },

    /**
     * Method: createRenderRoot
     *
     * Returns:
     * {DOMElement} The specific render engine's root element
     */
    createRenderRoot: function() {
      var element   = document.createElement('div');
      element.id    = this.rootNodeId;
      return element;
    },

    /**
     * Method: createRoot
     *
     * Returns:
     * {DOMElement} The main root element to which we'll add vectors
     */
    createRoot: function() {
      var element   = document.createElement('div');
      element.id    = this.rootNodeId;
      return element;
    },

    /**
     * Method: createDefs
     *
     * Returns:
     * {DOMElement} The element to which we'll add the symbol definitions
     */
    createDefs: function() {
      var element   = document.createElement('div');
      element.id        = 'legato:renderer:emptyDiv:defs';
      return element;
    },

    /**************************************
     *                                    *
     *     GEOMETRY DRAWING FUNCTIONS     *
     *                                    *
     **************************************/

    /**
     * Method: drawPoint
     * This method is only called by the renderer itself.
     *
     * Parameters:
     * node - {DOMElement}
     * geometry - {<OpenLayers.Geometry>}
     *
     * Returns:
     * {DOMElement} or false if the renderer could not draw the point
     */
    drawPoint: function(node, geometry) {
      this.addGeometryAttributes(node, geometry);
      return node;
    },

    /**
     * Method: drawCircle
     * This method is only called by the renderer itself.
     *
     * Parameters:
     * node - {DOMElement}
     * geometry - {<OpenLayers.Geometry>}
     * radius - {Float}
     *
     * Returns:
     * {DOMElement} or false if the renderer could not draw the circle
     */
    drawCircle: function(node, geometry, radius) {
      this.addGeometryAttributes(node, geometry);
      return node;
    },

    /**
     * Method: drawLineString
     * This method is only called by the renderer itself.
     *
     * Parameters:
     * node - {DOMElement}
     * geometry - {<OpenLayers.Geometry>}
     *
     * Returns:
     * {DOMElement} or null if the renderer could not draw all components of
     *     the linestring, or false if nothing could be drawn
     */
    drawLineString: function(node, geometry) {
      this.addGeometryAttributes(node, geometry);
      return node;
    },

    /**
     * Method: drawLinearRing
     * This method is only called by the renderer itself.
     *
     * Parameters:
     * node - {DOMElement}
     * geometry - {<OpenLayers.Geometry>}
     *
     * Returns:
     * {DOMElement} or null if the renderer could not draw all components
     *     of the linear ring, or false if nothing could be drawn
     */
    drawLinearRing: function(node, geometry) {
      this.addGeometryAttributes(node, geometry);
      return node;
    },

    /**
     * Method: drawPolygon
     * This method is only called by the renderer itself.
     *
     * Parameters:
     * node - {DOMElement}
     * geometry - {<OpenLayers.Geometry>}
     *
     * Returns:
     * {DOMElement} or null if the renderer could not draw all components
     *     of the polygon, or false if nothing could be drawn
     */
    drawPolygon: function(node, geometry) {
      this.addGeometryAttributes(node, geometry);
      return node;
    },

    /**
     * Method: drawRectangle
     * This method is only called by the renderer itself.
     *
     * Parameters:
     * node - {DOMElement}
     * geometry - {<OpenLayers.Geometry>}
     *
     * Returns:
     * {DOMElement} or false if the renderer could not draw the rectangle
     */
    drawRectangle: function(node, geometry) {
      this.addGeometryAttributes(node, geometry);
      return node;
    },

    /**
     * Method: drawSurface
     * This method is only called by the renderer itself.
     *
     * Parameters:
     * node - {DOMElement}
     * geometry - {<OpenLayers.Geometry>}
     *
     * Returns:
     * {DOMElement} or false if the renderer could not draw the surface
     */
    drawSurface: function(node, geometry) {
      if(this.addGeometryAttributes(node, geometry)){
        return node;
      }

      return false;
    },

    /**
     * Method: importSymbol
     * add a new symbol definition from the rendererer's symbol hash
     *
     * Parameters:
     * graphicName - {String} name of the symbol to import
     *
     * Returns:
     * {String} - id of the imported symbol
     */
    importSymbol: function (graphicName)  {
        cache = {
            path: 'legato',
            extent: 'null'
        };

        this.symbolCache.legato = cache;

        return cache;
    },

    /*
     * Adds to given element (DIV) some pseudo element atttributes
     */
    addGeometryAttributes: function(element, geometry){
     switch (geometry.CLASS_NAME) {
        case "OpenLayers.Geometry.Point":
          attribute       = document.createAttribute('x');
          attribute.value = geometry.x;
          element.attributes.setNamedItem(attribute);
          attribute       = document.createAttribute('y');
          attribute.value = geometry.y;
          element.attributes.setNamedItem(attribute);
          break;
        case "OpenLayers.Geometry.Rectangle":
          attribute       = document.createAttribute('x');
          attribute.value = geometry.x;
          element.attributes.setNamedItem(attribute);
          attribute       = document.createAttribute('y');
          attribute.value = geometry.y;
          element.attributes.setNamedItem(attribute);
          attribute       = document.createAttribute('width');
          attribute.value = geometry.x;
          element.attributes.setNamedItem(attribute);
          attribute       = document.createAttribute('height');
          attribute.value = geometry.y;
          element.attributes.setNamedItem(attribute);
          break;
        case "OpenLayers.Geometry.Polygon":
          attribute       = document.createAttribute('bboxLeft');
          attribute.value = geometry.bounds.left;
          element.attributes.setNamedItem(attribute);
          attribute       = document.createAttribute('bboxTop');
          attribute.value = geometry.bounds.top;
          element.attributes.setNamedItem(attribute);
          attribute       = document.createAttribute('bboxWidth');
          attribute.value = geometry.bounds.getWidth();
          element.attributes.setNamedItem(attribute);
          attribute       = document.createAttribute('bboxHeight');
          attribute.value = geometry.bounds.getHeight();
          element.attributes.setNamedItem(attribute);
          break;
        /*
         * Done because of JSLint
        case "OpenLayers.Geometry.LineString":
        case "OpenLayers.Geometry.LinearRing":
        case "OpenLayers.Geometry.Curve":
        case "OpenLayers.Geometry.Surface":
        */
        default:
          OpenLayers.Console.info('Could not render given geometry ' + geometry.CLASS_NAME);
          break;
      }
    },

    CLASS_NAME: "OpenLayers.Renderer.Div"
});
