/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: leine $
 * @version $Rev: 72369 $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: Legato.Control.WMSGetFeatureInfo
 *
 * This class represents a compositum of the
 * OpenLayers.Control.WMSGetFeatureInfo class and the
 * OpenLayers.Popup.FramedCloud. If the GetFeatureInfo reuest is made the
 * response will been rendered as a popup.
 *
 * Inherits from: - <Legato.Control>
 */
Legato.Control.WMSGetFeatureInfo = OpenLayers.Class(Legato.Control, {
  /**
   * Property: wmsGetFeatureInfo {<OpenLayers.Control.WMSGetFeatureInfo>} the
   * WMSGetFeatureInfo instance which will been used for requesting
   */
  wmsGetFeatureInfo :null,

  /**
   * Property: wmsGetFeatureInfoGml {<OpenLayers.Control.WMSGetFeatureInfo>} the
   * WMSGetFeatureInfo instance which will been used for requesting using the GML Info Format
   */
  wmsGetFeatureInfoGml :null,

  /**
   * Property: popup {<OpenLayers.Popup>} The popup which will render the
   * reponse
   */
  openLayersPopup :null,

  /**
   * Property: popup {<Legato.Popup>} The popup which will render the
   * reponse
   */
  popup :null,

  initialPopupPosition: null,

  /**
   * Property: popupType {<String>} The popuptype which will been used to
   * render the response. Possible values are Anchored, AnchroedBubble and
   * FramedCloud
   */
  popupType :'',

  /**
   * Property: autoActive {<Boolean>} If set to true the control will allways
   * be enabled and active, meaning that if the user hovers over the map the
   * control will always perform a WMSGetFeatureInfo request.
   */
  autoActive :false,

  /**
   * Property: popupBackgroundColor {<String>} Defines the backgroundColor of the
   * popup. Not used if popuptype is set to AnchroedBubble, cause you can not
   * set the backgroundColor for this type.
   */
  popupBackgroundColor :'',

  /*
   * Property: popupMinSize
   *
   * See Also: <OpenLayers.Popup.minSize>
   */
  popupMinSize : null,

  /*
   * Property: popupMaxSize
   *
   * See Also: <OpenLayers.Popup.maxSize>
   */
  popupMaxSize : null,

  /*
   * Property: popupAutoSize
   *
   * See Also: <OpenLayers.Popup.maxSize>
   */
  popupAutoSize: true,

  /*
   * Property: popupPanMapIfOutOfView
   *
   * See Also: <OpenLayers.Popup.panMapIfOutOfView>
   */
  popupPanMapIfOutOfView: true,

  /*
   * Property: popupKeepInMap
   *
   * See Also: <OpenLayers.Popup.keepInMap>
   */
  popupKeepInMap:  true,

  /*
   * Property: popupCloseOnMove
   *
   * See Also: <OpenLayers.Popup.closeOnMove>
   */
  popupCloseOnMove: false,

  /*
   * Property: popupOpacity
   *
   * See Also: <OpenLayers.Popup.opacity>
   */
  popupOpacity :null,


  /*
   * Property: renderIfNoPopupExists
   *
   * Only render if no other popup exists
   */
  renderIfNoPopupExists: false,


  /*
   * Property: filter
   *
   * A predicate definition which is used to filter the layers for the WMSGetFeatureInfoRequest
   *
   * See also: <Legato.Predicate>
   */
  layerFilter: null,

  /*
   * A LayerTransformer which might be configured for filtering the built WMS GetFeatureInfo request
   *
   * See also: <Legato.Layer.Transformer>
   */
  layerTransformer: null,

  /*
   * Property: highlightGeometry
   *
   * If true the geometry will be hightlightes using GML and SVG, false otherwise.
   */
  highlightGeometry: false,


  /**
   * Constructor: Legato.Control.WMSGetFeatureInfo Create a WMSGetFeatureInfo.
   *
   * Parameters:
   */
  initialize : function(options) {
    if (!options.delay) {
      options.delay = 500;
    }

    if (!options.popupBackgroundColor) {
      options.popupBackgroundColor = '#fafad2';
    }

    options.handlerOptions = {
      click :options.delay,
      hover :options.delay
    };

    Legato.Control.prototype.initialize.apply(this, [ options ]);

    this.wmsGetFeatureInfo = new OpenLayers.Control.WMSGetFeatureInfo(options);

    /*
     * If highlighting is set to true create a second WMS GetFeatureInfo Control
     * which will fire the same request as our first GetFeatureIInfo Control but
     * this time allways exspect that the answer is GML. This returned GML Feature
     * will then been displayed as a normal vector in this current map.
     */
    if(this.highlightGeometry === true){
      options.infoFormat = 'application/vnd.ogc.gml';
      options.legatoIgnore = true;
      this.wmsGetFeatureInfoGml = new OpenLayers.Control.WMSGetFeatureInfo(options);
    }

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.layerFilter)){
      this.layerTransformer = new Legato.Layer.Transformer.SublayerFilteringTransformer(this.layerFilter);
    }
    /*
     * Remember initial position. If set then keep this positon. If null then automatically
     * update the position
     */
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.popup)){
      this.initialPopupPosition = this.popup.getPosition();
    }
  },

  /**
   * Function: draw
   *Legato.LayerTransformer
   * See Also: <OpenLayer.Control>
   */
  draw : function() {
    OpenLayers.Control.prototype.draw.apply(this);
  },

  enableAutoActive : function() {
    if (!this.wmsGetFeatureInfo.active) {
      this.activate();
    }
  },

  /**
   * APIMethod: destroy
   *
   * See Also: <OpenLayer.Control>
   */
  destroy : function() {
    if (this.autoActive) {
      this.map.events.un( {
        'mouseover' :this.enableAutoActive,
        scope :this
      });
    }

    if (this.hover) {
      this.map.events.un( {
        'mousemove' :this.hidePopup,
        scope :this
      });
    }

    this.map.events.un( {
      'legato.selectlayer'    : this.updateSelectedLayers,
      'legato.deselectlayer'  : this.updateSelectedLayers,
      'addlayer'              : this.updateQueryLayers,
      'removelayer'           : this.updateQueryLayers,
      scope :this
    });

    this.wmsGetFeatureInfo.events.un( {
      'getfeatureinfo' :this.renderGetFeatureInfo,
      scope :this
    });
    this.wmsGetFeatureInfo.destroy();

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
      this.wmsGetFeatureInfoGml.events.un( {
        'getfeatureinfo' :this.hightlightFeatureInfoGeometry,
        scope :this
      });
      this.wmsGetFeatureInfoGml.destroy();
    }

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.highlightLayer)){
      this.highlightLayer.destroy();
    }

    if (!Legato.Lang.ObjectUtils.isNullOrUndefined(this.openLayersPopup)) {
      this.map.removePopup(this.openLayersPopup);
      this.openLayersPopup.destroy();
    }

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.popup)){
      this.popup.close();
    }

    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /**
   * APIMethod: activate
   *
   * See Also: <OpenLayer.Control>
   */
  activate : function() {
    OpenLayers.Control.prototype.activate.apply(this, arguments);
    this.updateQueryLayers();
    this.updateSelectedLayers();
    this.wmsGetFeatureInfo.activate();

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
      this.wmsGetFeatureInfoGml.activate();
    }
  },

  /**
   * APIMethod: deactivate
   *
   * See Also: <OpenLayer.Control>
   */
  deactivate : function() {
    this.wmsGetFeatureInfo.deactivate();
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
      this.wmsGetFeatureInfoGml.deactivate();
    }
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.highlightLayer)){
      this.highlightLayer.destroyFeatures();
    }
    this.hidePopup();
    OpenLayers.Control.prototype.deactivate.apply(this, arguments);
  },

  /**
   * Hides the current popup if exists
   */
  hidePopup : function() {
    if (!Legato.Lang.ObjectUtils.isNullOrUndefined(this.openLayersPopup)) {
      this.openLayersPopup.hide();
    }
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.popup)){
      this.popup.close();
    }
  },

  /**
   * Returns all current selectedlayers
   */
  getSelectedLayers: function() {
    return Legato.Lang.CollectionUtils.filter(this.map.layers,
        function(layer) {
          return Legato.OpenLayers.Map.Util.isListedLayer(layer)
              && Legato.Lang.ObjectUtils.isTrue(layer.selected)
              && Legato.Lang.ObjectUtils.isTrue(layer.queryable);
    });
  },

  /**
   * Returns all queryable layers
   */
  getQueryAbleLayers: function(){
    return Legato.Lang.CollectionUtils.filter(this.map.layers,
        function(layer) {
          return Legato.OpenLayers.Map.Util.isListedLayer(layer)
              && Legato.Lang.ObjectUtils.isTrue(layer.queryable);
    });
  },

  /**
   * Returns all layers
   */
  getListedLayers: function(){
    return Legato.Lang.CollectionUtils.filter(this.map.layers,
        function(layer) {
          return Legato.OpenLayers.Map.Util.isListedLayer(layer);
    });
  },

  /**
   * Updates all selected Layers
   */
  updateSelectedLayers: function(){
    if(this.active !== true){
      //No need for updating possible layer. We wait till we're really active
      return;
    }

    var selectedLayers = this.getSelectedLayers();
    var selectedLayerIds = Legato.OpenLayers.Map.Util.getLayerIds(selectedLayers);

    this.wmsGetFeatureInfo.vendorParams = {
        selected_layers: selectedLayerIds
    };

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
      this.wmsGetFeatureInfoGml.vendorParams = {
          selected_layers: selectedLayerIds
      };
    }
  },

  /**
   * Updates all current layers by injecting them into the wrapped
   * WMSGetFeatureInfo control of openlayers.
   */
  updateQueryLayers : function() {

    if(this.active !== true){
      //No need for updating possible layer. We wait till we're really active
      return;
    }

    var queryableLayers = this.getQueryAbleLayers();

    if(Legato.Lang.ObjectUtils.isNullOrUndefined(this.layerTransformer)){
      this.wmsGetFeatureInfo.layers = queryableLayers;
      if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
        this.wmsGetFeatureInfoGml.layers = this.wmsGetFeatureInfo.layers;
      }
    } else {
      var clonedLayers = [];
      for(var i = 0; i < queryableLayers.length; i++){
        var transformedLayer = this.layerTransformer.transform(queryableLayers[i]);
        if(!Legato.Lang.ObjectUtils.isNullOrUndefined(transformedLayer)){
          clonedLayers.push(transformedLayer);
        }
      }

      this.wmsGetFeatureInfo.layers = clonedLayers;
      if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
        this.wmsGetFeatureInfoGml.layers = clonedLayers;
      }
    }
  },

  /*
   * APIMethod: setMap
   *
   * See Also: <OpenLayer.Control>
   */
  setMap : function(map) {
    this.wmsGetFeatureInfo.setMap(map);

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
      this.wmsGetFeatureInfoGml.setMap(map);
    }

    OpenLayers.Control.prototype.setMap.apply(this, arguments);

    if (this.autoActive) {
      this.map.events.on( {
        'mouseover' :this.enableAutoActive,
        scope :this
      });
    }

    if (this.hover) {
      this.map.events.on( {
        'mousemove' :this.hidePopup,
        scope :this
      });
    }

    this.map.events.on( {
      'legato.selectlayer'    : this.updateSelectedLayers,
      'legato.deselectlayer'  : this.updateSelectedLayers,
      'addlayer'              : this.updateQueryLayers,
      'removelayer'           : this.updateQueryLayers,
      scope :this
    });

    this.wmsGetFeatureInfo.events.on( {
      'getfeatureinfo' :this.renderGetFeatureInfo,
      scope :this
    });

    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.wmsGetFeatureInfoGml)){
      this.wmsGetFeatureInfoGml.events.on( {
        'getfeatureinfo' :this.hightlightFeatureInfoGeometry,
        scope :this
      });
    }
  },

  /*
   * Hightlights the GML Geomtry by paintinig it into our highlight layer
   */
  hightlightFeatureInfoGeometry: function(featureInfo){
    if(Legato.Lang.ObjectUtils.isNullOrUndefined(this.highlightLayer)){
      var styleMap = this.getStyleMap();
      this.highlightLayer = new OpenLayers.Layer.Vector("Highlighted Features", {
        displayInLayerSwitcher: false,
        isBaseLayer: false,
        styleMap: styleMap
      });

      this.map.addLayer(this.highlightLayer);
    }

    this.highlightLayer.destroyFeatures();
    if (featureInfo.features && featureInfo.features.length) {
      this.highlightLayer.addFeatures(featureInfo.features);
      this.highlightLayer.redraw();
    }
  },

  /*
   * Get the StyleMap for the Highlight Layer
   */
  getStyleMap : function(){
    var highlightStyle = {
      Point: Legato.Feature.Style.HIGHLIGHT_POINT,
      Line: Legato.Feature.Style.HIGHLIGHT_LINE,
      Polygon: Legato.Feature.Style.HIGHLIGHT_POLYGON
    };

    var style = new OpenLayers.Style();
    style.addRules([
        new OpenLayers.Rule({symbolizer: highlightStyle})
    ]);

    return style;
  },

  /*
   * renders the returned response from the GetFeatureInfo request inside a
   * popup.
   */
  renderGetFeatureInfo : function(featureInfo) {
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.popup)){
      this.popup.close();
    }
    if (!Legato.Lang.ObjectUtils.isNullOrUndefined(this.openLayersPopup)) {
      this.map.removePopup(this.openLayersPopup);
    }

    /*
     * Just render a popup if there is some response
     */
    if (featureInfo.text.length === 0) {
      return;
    }

    /*
     * Abort showing the popup if another popup is allready shown
     * and the config for this control is set to true
     */
    if(this.renderIfNoPopupExists && (this.map.popups.length > 0 || !Legato.Lang.ObjectUtils.isNullOrUndefined(this.popup))){
      return;
    }

    /*
     * Use configured popup control instead of OpenLayers Popup mechnism
     */
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.popup)){
      if(Legato.Lang.ObjectUtils.isNull(this.initialPopupPosition)){
        this.popup.setPosition(featureInfo.xy);
      }
      this.popup.open(featureInfo.text);
      return;
    }

    /*
     * OpenLayers Popup mechanism
     */
    switch (this.popupType.toLowerCase()) {
    case 'anchored':
      this.openLayersPopup = new OpenLayers.Popup.Anchored('getFeatureInfo', this.map
          .getLonLatFromPixel(featureInfo.xy), null, featureInfo.text, null,
          !(this.autoActive || this.hover), OpenLayers.Function.bind(this.onClosePopup, this));
      break;
    case 'anchoredbubble':
      this.openLayersPopup = new OpenLayers.Popup.AnchoredBubble('getFeatureInfo',
          this.map.getLonLatFromPixel(featureInfo.xy), null, featureInfo.text,
          null, !(this.autoActive || this.hover), OpenLayers.Function.bind(this.onClosePopup, this));
      break;
    //case 'framedcloud':
    default:
      this.openLayersPopup = new OpenLayers.Popup.MarginAwareFramedCloud('getFeatureInfo', this.map
          .getLonLatFromPixel(featureInfo.xy), null, featureInfo.text, null,
          !(this.autoActive || this.hover), OpenLayers.Function.bind(this.onClosePopup, this));
      break;
    }

    //apply popup options
    this.openLayersPopup.autoSize = this.popupAutoSize;
    this.openLayersPopup.setBackgroundColor(this.popupBackgroundColor);
    if (this.popupMinSize) {
      this.openLayersPopup.minSize = this.popupMinSize;
    }
    if (this.popupMaxSize) {
      this.openLayersPopup.maxSize = this.popupMaxSize;
    }
    if (this.popupOpacity) {
      this.openLayersPopup.setOpacity(this.popupOpacity);
    }
    this.openLayersPopup.panMapIfOutOfView = this.popupPanMapIfOutOfView;
    this.openLayersPopup.keepInMap = this.popupKeepInMap;
    this.openLayersPopup.closeOnMove = this.popupCloseOnMove;

    this.map.addPopup(this.openLayersPopup);
  },

  /*
   * Callback function. Triggered whenever the user clicks its close button
   */
  onClosePopup: function(event){
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.popup)){
      this.popup.close();
    }
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.highlightLayer)){
      this.highlightLayer.destroyFeatures();
    }

    this.map.removePopup(this.openLayersPopup);
  },

  CLASS_NAME :'Legato.Control.WMSGetFeatureInfo'
});

/**
 * Component: lc:WMSGetFeatureInfo XML based config for a <Legato.Control.WMSGetFeatureInfo>.
 *
 * A valid config example for a WMSGetFeatureInfo would be:
 * (start code)
 * <lc:WMSGetFeatureInfo>
 *   <infoFormat>text/xml</infoFormat>
 *   <hover>true</hover>
 *   <maxFeatures>1</maxFeatures>
 *   <queryVisible>true</queryVisible>
 *   <handlerOptions>
 *   <hover>
 *    <delay>200</delay>
 *  </hover>
 *  </handlerOptions>
 * </lc:WMSGetFeatureInfo>
 * (end)
 *
 * See Also:
 * - <lc> namespace
 */
Legato.Control.WMSGetFeatureInfo.Bean = Legato.Control.Bean.Extend(

'Legato.Control.WMSGetFeatureInfo',

/*
 * Property: QName The qualified name for this control. Needed by XML config to
 * instantiate it.
 */
Legato.Control.QName('WMSGetFeatureInfo'), {
  _constructor :Legato.Control.WMSGetFeatureInfo,
  options : {
    /*
     * Property: hover
     *
     * See Also: <OpenLayers.Control.WMSGetFeatureInfo.hover>
     */
    hover :Legato.Lang.Boolean,

    /*
     * Property: maxFeatures
     *
     * See Also: <OpenLayers.Control.WMSGetFeatureInfo.maxFeatures>
     */
    maxFeatures :Legato.Lang.Integer,

    /*
     * Property: queryVisible
     *
     * See Also: <OpenLayers.Control.WMSGetFeatureInfo.queryVisible>
     */
    queryVisible :Legato.Lang.Boolean,

    /*
     * Property: url
     *
     * See Also: <OpenLayers.Control.WMSGetFeatureInfo.url>
     */
    url :Legato.Lang.String,

    /*
     * Property: infoFormat
     *
     * See Also: <OpenLayers.Control.WMSGetFeatureInfo.infoFormat>
     */
    infoFormat :Legato.Lang.String,

    /*
     * Property: autoActive If set to true the control will allways be enabled
     * and active, meaning that if the user hovers over the map the control will
     * always perform a WMSGetFeatureInfo request.
     */
    autoActive :Legato.Lang.Boolean,

    /*
     * Property: renderIfNoPopupExists If set to true the control will just render itself
     * when there is currently no other popup shown.
     */
    renderIfNoPopupExists :Legato.Lang.Boolean,

    /*
     * Property: delay Defines the delay in ms where a GetFeatureInfoRequest is
     * made. Default is 500 ms.
     */
    delay :Legato.Lang.Integer,

    /*
     * Property: popupType Defines the popuptype which will been used to render
     * the response. Possible values are Anchored, AnchroedBubble and
     * FramedCloud
     */
    popupType :Legato.Lang.String,

    /*
     * Property: popupBackgroundColor Defines the backgroundColor of the popup. Not
     * used if popuptype is set to AnchroedBubble, cause you can not set the
     * backgroundColor for this type.
     */
    popupBackgroundColor :Legato.Lang.String,

    /*
     * Property: popupMinSize
     *
     * See Also: <OpenLayers.Popup.minSize>
     */
    popupMinSize :OpenLayers.Size.Bean,

    /*
     * Property: popupMaxSize
     *
     * See Also: <OpenLayers.Popup.maxSize>
     */
    popupMaxSize :OpenLayers.Size.Bean,

    /*
     * Property: popupAutoSize
     *
     * See Also: <OpenLayers.Popup.maxSize>
     */
    popupAutoSize: Legato.Lang.Boolean,

    /*
     * Property: popupPanMapIfOutOfView
     *
     * See Also: <OpenLayers.Popup.panMapIfOutOfView>
     */
    popupPanMapIfOutOfView: Legato.Lang.Boolean,

    /*
     * Property: popupKeepInMap
     *
     * See Also: <OpenLayers.Popup.keepInMap>
     */
    popupKeepInMap:  Legato.Lang.Boolean,

    /*
     * Property: popupCloseOnMove
     *
     * See Also: <OpenLayers.Popup.closeOnMove>
     */
    popupCloseOnMove: Legato.Lang.Boolean,

    /*
     * Property: popupOpacity
     *
     * See Also: <OpenLayers.Popup.opacity>
     */
    popupOpacity :Legato.Lang.Number,

    /*
     * If set the specified custom popup is used instead the OpenLayers popup mechanism
     *
     * See also: <Legato.Popup>
     */
    popup: Legato.Popup,

    /*
     * A predicate definition which is used to filter the layers for the WMSGetFeatureInfoRequest
     *
     * See also: <Legato.Predicate>
     */
    layerFilter: Legato.Predicate,

    /*
     * Property: highlightGeometry
    *
    * If true the geometry will be highlights using GML and SVG, false otherwise.
    */
    highlightGeometry: Legato.Lang.Boolean
  }
});