/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Class: Legato.Control.OverviewBoundingBox
 *
 * This simple control switches the visibility of an arbitrary div which is given in its "div" property.
 *
 * Inherits from:
 * - <Legato.Control>
 *
 */
Legato.Control.OverviewBoundingBox = OpenLayers.Class(Legato.Control, {

  /**
   * Property: handlers
   * {Object}
   */
  handlers: null,

  /** Property: parentMap
   * {OpenLayers.Map}
   */
  parentMap: null,

  /**
   * Constructor: OpenLayers.Control.OverviewMap
   * Create a new overview map
   *
   * Parameters:
   * object - {Object} Properties of this object will be set on the overview
   * map object.  Note, to set options on the map object contained in this
   * control, set <mapOptions> as one of the options properties.
   */
  initialize: function(options) {
      this.handlers = {};
      //TODO: Check that parentMap is set in options
      Legato.Control.prototype.initialize.apply(this, [options]);
  },

  /**
   * APIMethod: destroy
   * Deconstruct the control
   */
  destroy: function() {
    if (!this.mapDiv) { // we've already been destroyed
        return;
    }
    if (this.handlers.click) {
      this.handlers.click.destroy();
    }
    if (this.handlers.zoom) {
      this.handlers.zoom.destroy();
    }

    this.mapDiv.removeChild(this.extentRectangle);
    this.extentRectangle = null;

    this.mapDiv = null;

    this.map.events.un({
        "moveend": this.update,
        scope: this
    });

    if (this.parentMap && this.parentMap.events) {
      this.parentMap.events.un({
        "moveend": this.update,
        scope: this
      });
    }

    Legato.Control.prototype.destroy.apply(this, arguments);
  },

  /**
   * Method: draw
   * Render the control in the browser.
   */
  draw: function() {
    if (!this.parentMap) {
      throw new Legato.Lang.Exception('parent map is not defined');
    }
    Legato.Control.prototype.draw.apply(this, arguments);

    this.mapDiv = this.map.div;

    this.extentRectangle = document.createElement('div');
    this.extentRectangle.style.position = 'absolute';
    this.extentRectangle.style.zIndex = this.map.Z_INDEX_BASE.Control + 100;  //HACK
    this.extentRectangle.className = this.displayClass+'ExtentRectangle';
    this.mapDiv.appendChild(this.extentRectangle);
    if (!(this.mapDiv.style.position === 'absolute' || this.mapDiv.style.position === 'relative')) {
      this.mapDiv.style.position = 'relative'; //HACK
    }

    // check extent rectangle border width
    this.wComp = parseInt(OpenLayers.Element.getStyle(this.extentRectangle,
                                           'border-left-width'), 10) +
                 parseInt(OpenLayers.Element.getStyle(this.extentRectangle,
                                           'border-right-width'), 10);
    this.wComp = (this.wComp) ? this.wComp : 2;
    this.hComp = parseInt(OpenLayers.Element.getStyle(this.extentRectangle,
                                           'border-top-width'), 10) +
                 parseInt(OpenLayers.Element.getStyle(this.extentRectangle,
                                           'border-bottom-width'), 10);
    this.hComp = (this.hComp) ? this.hComp : 2;

    if(this.parentMap.getExtent()) {
        this.update();
    }

    this.parentMap.events.register('moveend', this, this.update);
    this.map.events.register('moveend', this, this.update);

    this.handlers.click = new OpenLayers.Handler.Click(
        this, {
            "click": this.mapDivClick
        },{
            "single": true, "double": false,
            "stopSingle": true, "stopDouble": true,
            "pixelTolerance": 1,
            map: this.map
        }
    );
    this.handlers.click.activate();

    this.handlers.zoom = new OpenLayers.Handler.Box( this,
        {done: this.handleZoomBox}, {keyMask: this.keyMask} );
    this.handlers.zoom.activate();

    /*if (this.map.getProjection() != this.parentMap.getProjection()) {
      var sourceUnits = this.parentMap.getProjectionObject().getUnits() ||
          this.parentMap.units || this.parentMap.baseLayer.units;
      var targetUnits = this.map.getProjectionObject().getUnits() ||
          this.map.units || this.map.baseLayer.units;
      this.resolutionFactor = sourceUnits && targetUnits ?
          OpenLayers.INCHES_PER_UNIT[sourceUnits] /
          OpenLayers.INCHES_PER_UNIT[targetUnits] : 1;
    }*/

    return this.div;
  },

  /**
   * Method: mapDivClick
   * Handle browser events
   *
   * Parameters:
   * evt - {<OpenLayers.Event>} evt
   */
  mapDivClick: function(evt) {
    var pxCenter = this.rectPxBounds.getCenterPixel();
    var deltaX = evt.xy.x - pxCenter.x;
    var deltaY = evt.xy.y - pxCenter.y;
    var top = this.rectPxBounds.top;
    var left = this.rectPxBounds.left;
    var height = Math.abs(this.rectPxBounds.getHeight());
    var width = this.rectPxBounds.getWidth();
    var newTop = Math.max(0, (top + deltaY));
    newTop = Math.min(newTop, this.map.size.h - height);
    var newLeft = Math.max(0, (left + deltaX));
    newLeft = Math.min(newLeft, this.map.size.w - width);
    this.setRectPxBounds(new OpenLayers.Bounds(newLeft,
                                               newTop + height,
                                               newLeft + width,
                                               newTop));
    this.updateMapToRect(false);
  },

  /**
   * Method: update
   * Update the overview map after layers move.
   */
  update: function() {
    // update extent rectangle
    this.updateRectToMap();
  },

  /**
   * Method: updateRectToMap
   * Updates the extent rectangle position and size to match the map extent
   */
  updateRectToMap: function() {
    // If the projections differ we need to reproject
    var bounds;

    /*
     * The projection for our map or the parent map may be null due we create
     * both of them asynchronous. If this is true we just abort.
     */
    if (Legato.Lang.ObjectUtils.isNullOrUndefined(this.map.getProjection()) ||
        Legato.Lang.ObjectUtils.isNullOrUndefined(this.parentMap.getProjection())) {
      return;
    }

    if (this.map.getProjection() != this.parentMap.getProjection()) {
        bounds = this.parentMap.getExtent().transform(
            this.parentMap.getProjectionObject(),
            this.map.getProjectionObject() );
    } else {
        bounds = this.parentMap.getExtent();
    }

    //Do not paint if bounds is not within your maxExtent
    if(!this.map.getExtent().containsBounds(bounds, true, true)){
      bounds = null;
    }

    var pxBounds = this.getRectBoundsFromMapBounds(bounds);
    this.setRectPxBounds(pxBounds);
  },

  /**
   * Method: updateMapToRect
   * Updates the map extent to match the extent rectangle position and size
   */
  updateMapToRect: function(zoom) {
    var lonLatBounds = this.getMapBoundsFromRectBounds(this.rectPxBounds);

    /*
     * Abort if eather our map or the referenced map does not have a projection
     * yet. This could occur at creation time when maps are created asynchronous.
     */
    if(Legato.Lang.ObjectUtils.isNullOrUndefined(this.map.getProjection()) ||
        Legato.Lang.ObjectUtils.isNullOrUndefined(this.parentMap.getProjection())){
      return;
    }

    if (this.map.getProjection() != this.parentMap.getProjection()) {
      lonLatBounds = lonLatBounds.transform(
        this.map.getProjectionObject(),
        this.parentMap.getProjectionObject() );
    }
    if (!zoom) {
      this.parentMap.panTo(lonLatBounds.getCenterLonLat());
    } else {
      this.parentMap.zoomToExtent(lonLatBounds, true);
    }
  },

  /**
   * Method: setRectPxBounds
   * Set extent rectangle pixel bounds.
   *
   * Parameters:
   * pxBounds - {<OpenLayers.Bounds>}
   */
  setRectPxBounds: function(pxBounds) {
    if (pxBounds) {
      this.extentRectangle.className = this.displayClass +
      'ExtentRectangle';
    } else {
      this.extentRectangle.className = this.displayClass +
      'ExtentRectangleOutside';
      pxBounds = new OpenLayers.Bounds(0,this.map.size.h,this.map.size.w,0);
    }
    var top = Math.max(pxBounds.top, 0);
    var left = Math.max(pxBounds.left, 0);
    var bottom = Math.min(pxBounds.top + Math.abs(pxBounds.getHeight()),
                          this.map.size.h - this.hComp);
    var right = Math.min(pxBounds.left + pxBounds.getWidth(),
                         this.map.size.w - this.wComp);
    var width = Math.max(right - left, 0);
    var height = Math.max(bottom - top, 0);
    this.extentRectangle.style.top = Math.round(top) + 'px';
    this.extentRectangle.style.left = Math.round(left) + 'px';
    this.extentRectangle.style.height = Math.round(height) + 'px';
    this.extentRectangle.style.width = Math.round(width) + 'px';
    this.rectPxBounds = new OpenLayers.Bounds(
        Math.round(left), Math.round(bottom),
        Math.round(right), Math.round(top)
    );
  },

  /**
   * Method: getRectBoundsFromMapBounds
   * Get the rect bounds from the map bounds.
   *
   * Parameters:
   * lonLatBounds - {<OpenLayers.Bounds>}
   *
   * Returns:
   * {<OpenLayers.Bounds>}A bounds which is the passed-in map lon/lat extent
   * translated into pixel bounds for the overview map
   */
  getRectBoundsFromMapBounds: function(lonLatBounds) {
    var bounds = null;
    if (lonLatBounds) {
      var leftBottomLonLat = new OpenLayers.LonLat(lonLatBounds.left,
                                                   lonLatBounds.bottom);
      var rightTopLonLat = new OpenLayers.LonLat(lonLatBounds.right,
                                                 lonLatBounds.top);
      var leftBottomPx = this.getOverviewPxFromLonLat(leftBottomLonLat);
      var rightTopPx = this.getOverviewPxFromLonLat(rightTopLonLat);
      if (leftBottomPx && rightTopPx) {
          bounds = new OpenLayers.Bounds(leftBottomPx.x, leftBottomPx.y,
                                         rightTopPx.x, rightTopPx.y);
      }
    }
    return bounds;
  },

  /**
   * Method: getMapBoundsFromRectBounds
   * Get the map bounds from the rect bounds.
   *
   * Parameters:
   * pxBounds - {<OpenLayers.Bounds>}
   *
   * Returns:
   * {<OpenLayers.Bounds>} Bounds which is the passed-in overview rect bounds
   * translated into lon/lat bounds for the overview map
   */
  getMapBoundsFromRectBounds: function(pxBounds) {
    var leftBottomPx = new OpenLayers.Pixel(pxBounds.left,
                                            pxBounds.bottom);
    var rightTopPx = new OpenLayers.Pixel(pxBounds.right,
                                          pxBounds.top);
    var leftBottomLonLat = this.getLonLatFromOverviewPx(leftBottomPx);
    var rightTopLonLat = this.getLonLatFromOverviewPx(rightTopPx);
    return new OpenLayers.Bounds(leftBottomLonLat.lon, leftBottomLonLat.lat,
                                 rightTopLonLat.lon, rightTopLonLat.lat);
  },

  /**
   * Method: getLonLatFromOverviewPx
   * Get a map location from a pixel location
   *
   * Parameters:
   * overviewMapPx - {<OpenLayers.Pixel>}
   *
   * Returns:
   * {<OpenLayers.LonLat>} Location which is the passed-in overview map
   * OpenLayers.Pixel, translated into lon/lat by the overview map
   */
  getLonLatFromOverviewPx: function(overviewMapPx) {
    var size = this.map.size;
    var fullXPixels = size.w - this.wComp;
    var fullYPixels = size.h - this.hComp;
    var fullWidth = this.map.getExtent().getWidth();
    var fullHeight = this.map.getExtent().getHeight();
    var minX = this.map.getExtent().left;
    var maxY = this.map.getExtent().top;

    var delta_x = (overviewMapPx.x / fullXPixels) * fullWidth;
    var delta_y = (overviewMapPx.y / fullYPixels) * fullHeight;

    var xCoord = minX + delta_x;
    var yCoord = maxY - delta_y;

    return new OpenLayers.LonLat(xCoord, yCoord);
  },

  /**
   * Method: getOverviewPxFromLonLat
   * Get a pixel location from a map location
   *
   * Parameters:
   * lonlat - {<OpenLayers.LonLat>}
   *
   * Returns:
   * {<OpenLayers.Pixel>} Location which is the passed-in OpenLayers.LonLat,
   * translated into overview map pixels
   */
  getOverviewPxFromLonLat: function(lonlat) {
    var res  = this.map.getResolution();
    var extent = this.map.getExtent();
    var px = null;
    if (extent) {
      px = new OpenLayers.Pixel(
        Math.round(1/res * (lonlat.lon - extent.left)),
        Math.round(1/res * (extent.top - lonlat.lat))
      );
    }
    return px;
  },

  /**
   * Method: zoomBox
   *
   * Parameters:
   * position - {<OpenLayers.Bounds>} or {<OpenLayers.Pixel>}
   */
  handleZoomBox: function (position) {
    if (position instanceof OpenLayers.Bounds) {
      this.setRectPxBounds(position);
      this.updateMapToRect(true);

        /*var minXY = this.map.getLonLatFromPixel(
                        new OpenLayers.Pixel(position.left, position.bottom));
        var maxXY = this.map.getLonLatFromPixel(
                        new OpenLayers.Pixel(position.right, position.top));
        var bounds = new OpenLayers.Bounds(minXY.lon, minXY.lat,
                                           maxXY.lon, maxXY.lat);

        // always zoom in/out
        this.map.zoomToExtent(bounds);*/

    } else { // it's a pixel
      //Do nothing, the click handler will deal with it.
    }
  },

  CLASS_NAME :"Legato.Control.OverviewBoundingBox"
});

/**
 * Structure: lc:OverviewBoundingBox
 * XML based config for a <Legato.Control.OverviewBoundingBox>.
 *
 * See Also:
 * - <Legato.Control>
 * - <Legato.Beans.BeanFactory>
 * - <QName>
 *
 * A valid config example for a OverviewBoundingBox would be:
 * (start code)
 * <lc:OverviewBoundingBox div="overviewMap" displayClass="LegatoControlToggleOverviewMap"/>
 * (end)
 */
Legato.Control.OverviewBoundingBox.Bean = OpenLayers.Control.Bean.Extend(
  'Legato.Control.OverviewBoundingBox', Legato.Control.QName('OverviewBoundingBox'), {
    _constructor :Legato.Control.OverviewBoundingBox,
    options:
    {
      parentMap: OpenLayers.Map
    }
  }
);
