/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * @author $Author: valikov $
 * @version $Rev: 91043 $
 * @requires Legato.Format.ExtendedWMC.Model
 */

/**
 * Class: Legato.Format.ExtendedWMC.Parser
 * Parser for ExtendedWMC XML documents
 *
 * Inherits from:
 * - <OpenLayers.Format.XML>
 */
Legato.Format.ExtendedWMC.Parser = OpenLayers.Class(OpenLayers.Format.XML, {

    layerFactories: Legato.Format.ExtendedWMC.DEFAULT_LAYER_FACTORIES,

    /**
     * Mapping of namespace aliases to namespace URIs.
     */
    namespaces: {
        ol: "http://openlayers.org/context",
        l:  "http://www.disy.net/legato/extensions",
        lb:  "http://www.disy.net/legato/beans",
        wmc: "http://www.opengis.net/context",
        sld: "http://www.opengis.net/sld",
        xlink: "http://www.w3.org/1999/xlink",
        xsi: "http://www.w3.org/2001/XMLSchema-instance"
    },

    /**
     * Schema location for a particular minor version.
     */
    schemaLocation: "",

    /**
     * Get the namespace prefix for a given uri from the <namespaces> object.
     *
     * Parameters:
     * uri - {String} namespace uri
     *
     * Returns:
     * {String} A namespace prefix or null if none found.
     */
    getNamespacePrefix: function(uri) {
        var prefix = null;
        if(uri === null) {
            prefix = this.namespaces[this.defaultPrefix];
        } else {
            for(prefix in this.namespaces) {
                if(this.namespaces[prefix] == uri) {
                    break;
                }
            }
        }
        return prefix;
    },

    /**
     * Property: defaultPrefix
     * {String} Default prefix
     */
    defaultPrefix: "wmc",

    /**
     * Property: rootPrefix
     * {String} Prefix on the root node that maps to the context namespace URI.
     */
    rootPrefix: null,

    /**
     * Property: defaultStyleName
     * {String} Style name used if layer has no style param.  Default is "".
     */
    defaultStyleName: "",

    /**
     * Property: defaultStyleTitle
     * {String} Default style title.  Default is "Default".
     */
    defaultStyleTitle: "Default",

    /**
     * Constructor: initialize
     *
     * Parameters:
     *
     * options - {Object} An optional object whose properties will be set on this instance.
     */
    initialize: function(options) {
        OpenLayers.Format.XML.prototype.initialize.apply(this, [options]);
        if (Legato.Lang.ObjectUtils.isObject(options))
        {
          if (Legato.Lang.ObjectUtils.exists(options.layerFactories))
          {
            Legato.Util.Ensure.ensureObject(options.layerFactories);
            this.layerFactories = options.layerFactories;
          }
        }
    },

    /**
     * Property: beanFactory
     * {<Legato.Beans.XMLBeanFactory>} bean factory which will be used to parse extensions.
     */
    beanFactory: null,

    setBeanFactory: function(beanFactory)
    {
      Legato.Util.Ensure.ensureObject(beanFactory);
      this.beanFactory = beanFactory;
    },

    getBeanFactory: function()
    {
      if (this.beanFactory === null)
      {
        this.beanFactory = new Legato.Beans.XMLBeanFactory();
      }
      return this.beanFactory;
    },

    getLayerType: function(service)
    {
      Legato.Util.Ensure.ensureString(service);
      var layerFactory = this.layerFactories[service];
      Legato.Util.Ensure.ensureExists(layerFactory, 'Could not find the layer factory for service [' + service + '].');
      Legato.Util.Ensure.ensureExists(layerFactory.layerType, 'Layer factory for service [' + service + '] does not deliver a layer type.');
      return layerFactory.layerType;
    },

    /**
     * Function: read
     * Read capabilities data from a string, and returns a Legato.Format.ExtendedWMC.Model
     *
     * Parameters:
     * data - {String} or {DOMElement} data to read/parse.
     *
     * Returns:
     * {<Legato.Format.ExtendedWMC.Model>} the parsed model
     */
    read: function(data) {
        if(typeof data == "string") {
          var format = new OpenLayers.Format.XML();
          data = format.read(data);
          //data = OpenLayers.Format.XML.prototype.read.apply(this, [data]);
        }
        var root = data.documentElement;

        if(root === null){
          throw new Legato.Lang.Exception('ViewContext seems to be empty');
        }

        if(root.nodeName !== 'ViewContext'){
          throw new Legato.Lang.Exception('ViewContext seems to be incorrect. First node element must be named ViewContext');
        }

        this.rootPrefix = root.prefix;
        var model = new Legato.Format.ExtendedWMC.Model();
        model.id = root.getAttribute("id");
        model.version = root.getAttribute("version");
        this.runChildNodes(model, root);
        return model;
    },

    /**
     * Recursively process children of a node
     *
     * Parameters:
     * model - {Legato.Format.ExtendedWMC.Model} the model object
     * node - {DOMElement} the current XML node
     */
    runChildNodes: function(model, node) {
        var children = node.childNodes;
        var childNode, processor, prefix, local;
        for(var i=0, len=children.length; i<len; ++i) {
            childNode = children[i];
            if(childNode.nodeType == 1) {
                prefix = this.getNamespacePrefix(childNode.namespaceURI);
                local = childNode.nodeName.split(":").pop();
                processor = this["read_" + prefix + "_" + local];
                if(processor) {
                    processor.apply(this, [model, childNode]);
                }
            }
        }
    },

    /**
     * Internal methods to parse XML elements
     */

    read_wmc_General: function(model, node) {
        this.runChildNodes(model, node);
    },


    read_wmc_BoundingBox: function(model, node) {
      var SRS = node.getAttribute("SRS");
      if (SRS) {
        var bb = new Legato.Format.ExtendedWMC.Model.BoundingBox();
        model.boundingBox = bb;
        bb.SRS = SRS;
        bb.bounds = new OpenLayers.Bounds(
            parseFloat(node.getAttribute("minx")),
            parseFloat(node.getAttribute("miny")),
            parseFloat(node.getAttribute("maxx")),
            parseFloat(node.getAttribute("maxy"))
        );
      }
    },

    read_wmc_LayerList: function(model, node) {
        model.layers = [];
        this.runChildNodes(model, node);
    },

    read_wmc_Layer: function(model, node) {
      var l = new Legato.Format.ExtendedWMC.Model.Layer();
      model.layerList.push(l);
      l.hidden = (node.getAttribute("hidden") == "1");
      l.queryable = (node.getAttribute("queryable") == "1");
      var id = node.getAttribute("id");
      this.runChildNodes(l, node);
      if (Legato.Lang.ObjectUtils.isString(id))
      {
        l.layerId = id;
      }
      else
      {
        l.layerId = l.name;
      }
    },

    read_wmc_Extension: function(obj, node) {
      if (obj.CLASS_NAME && obj.CLASS_NAME === 'Legato.Format.ExtendedWMC.Model.Layer') {
        this.runChildNodes(obj, node);
      } else if (obj.CLASS_NAME && obj.CLASS_NAME === 'Legato.Format.ExtendedWMC.Model') {
        this.runChildNodes(obj, node);
      }
    },

    read_l_Extension: function(obj, node) {
      var beanFactory = this.getBeanFactory();
      var extension;
      if (obj.CLASS_NAME && obj.CLASS_NAME === 'Legato.Format.ExtendedWMC.Model.Layer') {
        var layerInfo = obj;
        var service = layerInfo.server.service;
        var layerType = this.getLayerType(service);
        extension = beanFactory.retrieveBeanFromElement(node, layerType.Components());
        layerInfo.extension = extension;
      }
      else if (obj.CLASS_NAME && obj.CLASS_NAME === 'Legato.Format.ExtendedWMC.Model') {
        var model = obj;
        extension = beanFactory.retrieveBeanFromElement(node, OpenLayers.Map.Bean.Components());
        model.extension = extension;
      }
    },

    read_lb_beans: function(obj, node) {
      var beanFactory = this.getBeanFactory();
      var beans = beanFactory.retrieveBeansFromElement(node);
      obj.beans = beans;
    },

    read_ol_units: function(layerInfo, node) {
      layerInfo.extension.units = this.getChildValue(node);
    },

    read_ol_maxExtent: function(layerInfo, node) {
      var bounds = new OpenLayers.Bounds(
          node.getAttribute("minx"), node.getAttribute("miny"),
          node.getAttribute("maxx"), node.getAttribute("maxy")
      );
      layerInfo.extension.maxExtent = bounds;
    },
    read_ol_minExtent: function(layerInfo, node) {
      var bounds = new OpenLayers.Bounds(
          node.getAttribute("minx"), node.getAttribute("miny"),
          node.getAttribute("maxx"), node.getAttribute("maxy")
      );
      layerInfo.extension.minExtent = bounds;
    },
    read_ol_transparent: function(layerInfo, node) {
      if (!layerInfo.extension.params)
      {
        layerInfo.extension.params = {};
      }
      layerInfo.extension.params.TRANSPARENT = this.getChildValue(node);
    },

    read_ol_numZoomLevels: function(layerInfo, node) {
      layerInfo.extension.numZoomLevels = parseInt(this.getChildValue(node), 10);
    },

    read_ol_opacity: function(layerInfo, node) {
      layerInfo.extension.opacity = parseFloat(this.getChildValue(node));
    },

    read_ol_singleTile: function(layerInfo, node) {
      layerInfo.extension.singleTile = (this.getChildValue(node) == "true");
    },

    read_ol_isBaseLayer: function(layerInfo, node) {
      layerInfo.extension.isBaseLayer = (this.getChildValue(node) == "true");
    },

    read_ol_displayInLayerSwitcher: function(layerInfo, node) {
      layerInfo.extension.displayInLayerSwitcher = (this.getChildValue(node) == "true");
    },
    read_wmc_Server: function(layerInfo, node) {
      if (layerInfo.server === null) {
        layerInfo.server = new Legato.Format.ExtendedWMC.Model.Layer.Server();
      }
      layerInfo.server.version = node.getAttribute("version");
      layerInfo.server.service = node.getAttribute("service");
      layerInfo.server.title = node.getAttribute("title");
      this.runChildNodes(layerInfo.server, node);
    },

    read_wmc_FormatList: function(layerInfo, node) {
      this.runChildNodes(layerInfo, node);
    },

    read_wmc_Format: function(layerInfo, node) {
        var format = this.getChildValue(node);
        layerInfo.formatList.push(format);
        if(node.getAttribute("current") == "1") {
            layerInfo.currentFormat = format;
        }
    },

    read_wmc_StyleList: function(layerInfo, node) {
        this.runChildNodes(layerInfo, node);
    },

    read_wmc_Style: function(layerInfo, node) {
        var style = new Legato.Format.ExtendedWMC.Model.Layer.Style();
        this.runChildNodes(style, node);
//        if(node.getAttribute("current") == "1") {
//            // three style types to consider
//            // 1) linked SLD
//            // 2) inline SLD
//            // 3) named style
//            // running child nodes always gets name, optionally gets href or body
//            if(style.href) {
//                layerInfo.params.sld = style.href;
//            } else if(style.body) {
//                layerInfo.params.sld_body = style.body;
//            } else {
//                layerInfo.params.styles = style.name;
//            }
//        }
        layerInfo.styleList.push(style);
        if(node.getAttribute("current") == "1") {
          layerInfo.currentStyle = style;
      }
    },


     //TODO Implement me later
     /*
    read_wmc_SLD: function(style, node) {
        this.runChildNodes(style, node);
        // style either comes back with an href or a body property
    },
    */


     //TODO Implement me later
     /*
    read_sld_StyledLayerDescriptor: function(sld, node) {
        var xml = OpenLayers.Format.XML.prototype.write.apply(this, [node]);
        sld.body = xml;
    },
    */

    read_wmc_OnlineResource: function(obj, node) {
      obj.onlineResource = this.getAttributeNS(
          node, this.namespaces.xlink, "href"
      );
    },

    read_wmc_Name: function(obj, node) {
        var name = this.getChildValue(node);
        if(name) {
            obj.name = name;
        }
    },

    read_wmc_Title: function(obj, node) {
        var title = this.getChildValue(node);
        if(title) {
            obj.title = title;
        }
    },

     // TODO Implement me later
     /*
    read_wmc_MetadataURL: function(layerInfo, node) {
        var metadataURL = {};
        var links = node.getElementsByTagName("OnlineResource");
        if(links.length > 0) {
            this.read_wmc_OnlineResource(metadataURL, links[0]);
        }
        layerInfo.options.metadataURL = metadataURL.href;

    },
    */

    read_wmc_Abstract: function(obj, node) {
        var abst = this.getChildValue(node);
        if(abst) {
          if (obj.CLASS_NAME && obj.CLASS_NAME == 'Legato.Format.ExtendedWMC.Model.Layer') {
            obj.layerAbstract = abst;
          }
          else if (obj.CLASS_NAME && obj.CLASS_NAME == 'Legato.Format.ExtendedWMC.Model.Layer.Style') {
            obj.styleAbstract = abst;
          }
        }
    },

    read_wmc_SRS: function(obj, node) {
      var srs = this.getChildValue(node);
      if(srs) {
        if (obj.CLASS_NAME && obj.CLASS_NAME === 'Legato.Format.ExtendedWMC.Model.Layer') {
          obj.srs.push(srs);
        }
      }
    },

    read_wmc_Window: function(model, node) {
      if (model.window === null) {
        model.window = new Legato.Format.ExtendedWMC.Model.Window();
      }
      model.window.width = parseInt(node.getAttribute("width"), 10);
      model.window.height = parseInt(node.getAttribute("height"), 10);
    },

    // TODO Implement me later
     /*
    read_wmc_LatLonBoundingBox: function(layer, node) {
        layer.llbbox = [
            parseFloat(node.getAttribute("minx")),
            parseFloat(node.getAttribute("miny")),
            parseFloat(node.getAttribute("maxx")),
            parseFloat(node.getAttribute("maxy"))
        ];
    },*/

    read_wmc_LegendURL: function(style, node) {
        var legend = {
            width: node.getAttribute('width'),
            height: node.getAttribute('height'),
            format: node.getAttribute('format')
        };
        this.runChildNodes(legend, node);
        style.legend = legend;
    },

    read_sld_MinScaleDenominator: function(layerInfo, node) {
      var minScale = this.getChildValue(node);
      if (minScale) {
        layerInfo.minScaleDenominator = parseFloat(minScale);
      }
    },

    read_sld_MaxScaleDenominator: function(layerInfo, node) {
      var maxScale = this.getChildValue(node);
      if (maxScale) {
        layerInfo.maxScaleDenominator = parseFloat(maxScale);
      }
    },

    /**
     * It is not implemented, it is only a parser!
     */
    write: function(context, options) {
      OpenLayers.Console.userError(OpenLayers.i18n("writeNotImplemented"));
    },

    CLASS_NAME: "Legato.Format.ExtendedWMC.Parser"

});
