/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * $Author: valikov $
 * $Rev: 91043 $
 */

/**
 * Class: Legato.Format.ExtendedWMC
 * Main entry point to the extended WMC functionality
 */
Legato.Format.ExtendedWMC = OpenLayers
    .Class({

      parser : null,
      layerFactories : null,
      initialize : function(options) {
        this.layerFactories = {};
        for ( var layerType in Legato.Format.ExtendedWMC.DEFAULT_LAYER_FACTORIES) {
          if (Legato.Format.ExtendedWMC.DEFAULT_LAYER_FACTORIES
              .hasOwnProperty(layerType)) {
            this.layerFactories[layerType] = Legato.Format.ExtendedWMC.DEFAULT_LAYER_FACTORIES[layerType];
          }
        }
        if (Legato.Lang.ObjectUtils.isObject(options)
            && Legato.Lang.ObjectUtils
                .exists(options.layerFactories)) {
          Legato.Util.Ensure.ensureObject(options.layerFactories);
          for (layerType in options.layerFactories) {
            if (options.layerFactories.hasOwnProperty(layerType)) {
              this.layerFactories[layerType] = options.layerFactories[layerType];
            }
          }
        }
        this.parser = new Legato.Format.ExtendedWMC.Parser({
          layerFactories : this.layerFactories
        });
      },
      beanFactory : null,
      setBeanFactory : function(beanFactory) {
        this.beanFactory = beanFactory;
        this.parser.setBeanFactory(beanFactory);
      },
      getBeanFactory : function() {
        if (this.beanFactory === null) {
          this.beanFactory = new Legato.Beans.XMLBeanFactory();
        }
        return this.beanFactory;
      },

      /**
       * Method: read
       * Read WMC data from a string, and return an object with a
       * list of layers and some window metadata.
       *
       * Parameters: data - {String} or {DOMElement} data to read/parse. options -
       * {Object} Some options that will be added to all created layers
       *
       * Returns: {<Legato.Format.ExtendedWMC.Result>} Parsed ExtendedWMC data and a
       * list of OpenLayers.Layer objects
       */
      read : function(data, options) {
        var model = this.parser.read(data);
        var result = this.modelToResult(model, options);
        return result;
      },

      /**
       * Transforms a <Legato.Format.ExtendedWMC.Model> to
       * <Legato.Format.ExtendedWMC.Result>
       */
      modelToResult : function(model, options) {
        var result = new Legato.Format.ExtendedWMC.Result();
        result.model = model;
        var layers = this.processLayers(model, options);
        result.layers = layers;
        return result;
      },

      /**
       * Transforms a the layerList of <Legato.Format.ExtendedWMC.Model> to
       * <Openlayers.Layer> objects
       */
      processLayers : function(model, options) {
        var layers = [];
        if (model.layerList) {
          if (!Legato.Lang.ObjectUtils.exists(options)) {
            options = {};
          }
          if (!Legato.Lang.ObjectUtils.exists(options.transparent)) {
            options.transparent = true;
          }
          var originalTransparent = options.transparent;
          for ( var i = 0; i < model.layerList.length; i++) {
            var layerInfo = model.layerList[i];
            var layerType = "OGC:WMS";
            if (layerInfo.server && layerInfo.server.service) {
              layerType = layerInfo.server.service;
            }

            var layerFactory = this.layerFactories[layerType];

            if (!Legato.Lang.ObjectUtils.exists(layerFactory)) {
              throw new Legato.Lang.Exception(
                  'Layer factory for the layer type ['
                      + layerType
                      + '] is not registered.');
            } else {
              options.formatList = layerInfo.formatList;
              if(layerInfo.currentFormat === 'image/jpeg' ){
                options.transparent = "false";
              } else {
                options.transparent = originalTransparent;
              }
              var layer = layerFactory.createLayer(layerInfo,
                  options, this.getBeanFactory());
              layers.push(layer);
            }
          }
        }
        return layers;
      },

      CLASS_NAME : 'Legato.Format.ExtendedWMC'
    });

Legato.Format.ExtendedWMC.Bean = Legato.Format.Bean.Extend(
    'Legato.Format.ExtendedWMC',
    //
    Legato.Format.QName('ExtendedWMC'),
    //
    {
      _constructor : Legato.Format.ExtendedWMC
    });

/**
 * Class: Legato.Format.ExtendedWMC.Result
 *
 * Represents the results of the read operation
 */
Legato.Format.ExtendedWMC.Result = OpenLayers.Class({

  initialize : function() {
    this.layers = [];
  },

  /**
   * Property: layers {Array} <OpenLayers.Layer> objects
   */
  layers : null,

  /**
   * Property: model {<Legato.Format.ExtendedWMC.Model>} the parsed ExtendedWMC model
   */
  model : null,

  CLASS_NAME : 'Legato.Format.ExtendedWMC.Result'
});

Legato.Format.ExtendedWMCLoader = Legato.Lang
    .Class({
      map : null,
      url : null,
      params : null,
      defaultOptions : null,
      createDefaultBaseLayer : false,
      zoomToBoundingBox : true,
      resizeToWindow : true,
      initialize : function(options) {
        this.format = new Legato.Format.ExtendedWMC();
        this.map = options.map;
        var url = options.url;
        var questionMarkIndex = url.indexOf('?');
        if (questionMarkIndex > 0) {
          this.url = url.substring(0, questionMarkIndex);
          this.params = OpenLayers.Util.getParameters(url);
        } else {
          this.url = url;
          this.params = null;
        }
        this.defaultOptions = Legato.Lang.ObjectUtils
            .exists(options.defaultOptions) ? options.defaultOptions
            : {};
        this.zoomToBoundingBox = Legato.Lang.ObjectUtils
            .exists(options.zoomToBoundingBox) ? options.zoomToBoundingBox
            : true;
        this.resizeToWindow = Legato.Lang.ObjectUtils
            .exists(options.resizeToWindow) ? options.resizeToWindow
            : true;
        this.createDefaultBaseLayer = Legato.Lang.ObjectUtils
            .exists(options.createDefaultBaseLayer) ? options.createDefaultBaseLayer
            : false;
      },
      load : function() {
        var onSuccess = function(request) {
          try {
            var result = this.format.read(request.responseText,
                this.defaultOptions);

            if (this.resizeToWindow && result.model.window) {
              this.map.div.style.width = result.model.window.width
                  + 'px';
              this.map.div.style.height = result.model.window.height
                  + 'px';
              this.map.updateSize();
            }
            if (Legato.Lang.ObjectUtils
                .exists(result.model.extension)) {
              Legato.Lang.CollectionUtils.copyIfExists(
                  result.model.extension, this.map, [
                      'minExtent', 'maxExtent',
                      'minScale', 'maxScale',
                      'minResolution', 'maxResolution',
                      'units', 'tags' ]);
              if (Legato.Lang.ObjectUtils
                  .exists(result.model.extension.maxExtent)) {
                var maxExtent = result.model.extension.maxExtent;
                var viewSizeMax = this.map.getSize();
                var wResMax = maxExtent.getWidth()
                    / viewSizeMax.w;
                var hResMax = maxExtent.getHeight()
                    / viewSizeMax.h;
                this.map.maxResolution = Math.max(wResMax,
                    hResMax);
              }
              if (Legato.Lang.ObjectUtils
                  .exists(result.model.extension.minExtent)) {
                var minExtent = result.model.extension.minExtent;
                var viewSizeMin = this.map.getSize();
                var wResMin = minExtent.getWidth()
                    / viewSizeMin.w;
                var hResMin = minExtent.getHeight()
                    / viewSizeMin.h;
                this.map.minResolution = Math.min(wResMin,
                    hResMin);
              }
              if (Legato.Lang.ObjectUtils
                  .exists(result.model.extension.dpi)) {
                OpenLayers.DOTS_PER_INCH = result.model.extension.dpi;
              }
            }

            var layers = result.layers;

            for ( var i = 0; i < layers.length; i++) {
              var layer = layers[i];
              if (this.map.fractionalZoom
                  && Legato.Lang.ObjectUtils
                      .exists(layer.minScale)
                  && Legato.Lang.ObjectUtils
                      .exists(layer.maxScale)
                  && layer.minScale / layer.maxScale < 2.0) {
                // exactly 2 zoom levels because otherwise OpenLayers would otherwise create only one zoom level
                layer.options.numZoomLevels = 2;
              }
              this.map.addLayer(layer);
            }

            if (!Legato.Lang.ObjectUtils.exists(this.map.baseLayer)
                && this.createDefaultBaseLayer
                && layers.length > 0) {
              var firstLayer = layers[0];
              var projection = firstLayer.projection;
              var baseLayerOptions = {};

              // units
              // projection
              if (Legato.Lang.ObjectUtils.exists(projection)) {
                baseLayerOptions.projection = projection;
              }
              // maxExtent
              baseLayerOptions.layerId = "Legato.Layer.Blank";
              baseLayerOptions.legatoIgnore = true;
              baseLayerOptions.alpha = false;
              baseLayerOptions.isBaseLayer = true;
              baseLayerOptions.displayOutsideMaxExtent = false;
              baseLayerOptions.ratio = 1;

              var baseLayer = new OpenLayers.Layer.WMS(
                  "Default base layer", OpenLayers.Util
                      .getImagesLocation()
                      + "blank.gif?", null,
                  baseLayerOptions);
              this.map.addLayer(baseLayer);
            }
            if ((this.map.getExtent() === null || Legato.Lang.ObjectUtils
                .isTrue(this.zoomToBoundingBox))
                && Legato.Lang.ObjectUtils
                    .exists(result.model.boundingBox)
                && Legato.Lang.ObjectUtils
                    .exists(result.model.boundingBox.bounds)) {
              this.map.zoomToExtent(
                  result.model.boundingBox.bounds, true);
            }
            this.map.events.triggerEvent("legato.refresh", {});
          } catch (e) {
            Legato.Util.Logger.getLogger().fatal(
                'Could not parse ExtendedWMC from [' + this.url
                    + '].');
            Legato.Util.Logger.getLogger().fatal(e);
            throw e;
          }
        };

        var onFailure = function(request) {
          Legato.Util.Logger.getLogger().fatal(
              'Could not read ExtendedWMC from [' + this.url
                  + '].');
          throw new Legato.Lang.Exception(
              'Could not read ExtendedWMC from [' + this.url
                  + '].');
        };
        // Do the actual call.
        OpenLayers.loadURL(this.url, this.params, this, onSuccess,
            onFailure);
      },
      synchronize : function() {

        var onSuccess = function(request) {
          try {
            var needsSynchronization = false;
            var result = this.format.read(request.responseText,
                this.defaultOptions);

            var mapLayerIds = new Legato.Util.HashSet();
            var resultLayerIds = new Legato.Util.HashSet();

            var openLayerIds = Legato.OpenLayers.Map.Util
                .getLayerIds(Legato.OpenLayers.Map.Util
                    .getLayersWithoutLegatoIgnore(this.map.layers));

            for ( var i = 0; i < openLayerIds.length; i++) {
              mapLayerIds.add(openLayerIds[i]);
            }

            openLayerIds = Legato.OpenLayers.Map.Util
                .getLayerIds(Legato.OpenLayers.Map.Util
                    .getLayersWithoutLegatoIgnore(result.layers));

            for (i = 0; i < openLayerIds.length; i++) {
              resultLayerIds.add(openLayerIds[i]);
            }

            var filteredLayers = Legato.OpenLayers.Map.Util
                .getLayersWithoutLegatoIgnore(result.layers);

            var layer, layerId;

            for (i = 0; i < filteredLayers.length; i++) {
              layer = filteredLayers[i];
              layerId = Legato.Lang.ObjectUtils
                  .exists(layer.layerId) ? layer.layerId
                  : layer.name;
              if (!mapLayerIds.contains(layerId)) {
                needsSynchronization = true;
                this.map.addLayer(layer);
              }
            }

            filteredLayers = Legato.OpenLayers.Map.Util
                .getLayersWithoutLegatoIgnore(this.map.layers);

            for (i = 0; i < filteredLayers.length; i++) {
              layer = filteredLayers[i];
              layerId = Legato.Lang.ObjectUtils
                  .exists(layer.layerId) ? layer.layerId
                  : layer.name;
              if (!resultLayerIds.contains(layerId)) {
                needsSynchronization = true;
                this.map.removeLayer(layer);
              }
            }

            if (needsSynchronization) {
              this.map.events.triggerEvent("legato.refresh", {});
            }
          } catch (e) {
            Legato.Util.Logger.getLogger().error(
                'Could not parse synchronize viewcontext response ['
                    + this.url + '] with params '
                    + this.params);
            throw e;
          }
        };

        var onFailure = function(request) {
          Legato.Util.Logger.getLogger().error(
              'Could not synchronize viewcontext [' + this.url
                  + '] with params ' + this.params);
        };

        // Do the actual call.
        OpenLayers.loadURL(this.url, this.params, this, onSuccess,
            onFailure);
      },

      CLASS_NAME : 'Legato.Format.ExtendedWMCLoader'

    });

/**
 * Component: lf:ExtendedWMCLoader
 *
 * See also:
 * - <lf> namespace
 * - {<Legato.Format.ExtendedWMCLoader>}
 */

Legato.Format.ExtendedWMCLoader.Bean = Legato.Beans.Object.Extend(
    'Legato.Format.ExtendedWMCLoader',
    //
    Legato.Format.QName('ExtendedWMCLoader'),
    //
    {
      _constructor : Legato.Format.ExtendedWMCLoader,
      options : {
        /**
         * Element: map
         * {<ol:Map>} Target map.
         */
        map : OpenLayers.Map.Bean,
        /**
         * Element: url
         * {String} URL of the WMC file.
         */
        url : Legato.Lang.String,
        /**
         * Element: defaultOptions
         * {String {}} Default layer options.
         */
        defaultOptions : Legato.Lang['String{}'],
        /**
         * Element: zoomToBoundingBox
         * {Boolean}
         */
        zoomToBoundingBox : Legato.Lang.Boolean,
        /**
         * Element: resizeToWindow
         * {Boolean}
         */
        resizeToWindow : Legato.Lang.Boolean,
        /**
         * Element: createDefaultBaseLayer
         * {Boolean}
         */
        createDefaultBaseLayer : Legato.Lang.Boolean
      },
      afterInstancePopulated : function(instance) {
        instance.load();
      }
    });

/* BACK This is for backwards compatibility. */
Legato.Format.SuperWMCLoader = {};

Legato.Format.SuperWMCLoader.Bean = Legato.Beans.Object.Extend(
    'Legato.Format.ExtendedWMCLoader',
    //
    Legato.Format.QName('SuperWMCLoader'),
    //
    {
      _constructor : Legato.Format.ExtendedWMCLoader,
      options : {
        map : OpenLayers.Map.Bean,
        url : Legato.Lang.String,
        defaultOptions : Legato.Lang['String{}'],
        zoomToBoundingBox : Legato.Lang.Boolean,
        resizeToWindow : Legato.Lang.Boolean,
        createDefaultBaseLayer : Legato.Lang.Boolean
      },
      afterInstancePopulated : function(instance) {
        instance.load();
      }
    });

Legato.Format.ExtendedWMC.LayerFactory = Legato.Lang
    .Class({
      layerType : OpenLayers.Layer.Bean,
      initialize : function(layerType) {
        if (Legato.Lang.ObjectUtils.exists(layerType)) {
          this.layerType = layerType;
        }
      },
      createLayer : function(layerInfo, options, beanFactory) {
        var definition = {};
        definition.visibility = !layerInfo.hidden;
        if (layerInfo.queryable) {
          definition.queryable = true;
        }
        definition.name = layerInfo.name;
        definition.id = layerInfo.id;
        definition.layerId = Legato.Lang.ObjectUtils
            .exists(layerInfo.layerId) ? layerInfo.layerId
            : layerInfo.name;
        definition.title = layerInfo.title;
        definition.url = layerInfo.server.onlineResource;
        if (Legato.Lang.ObjectUtils
            .exists(layerInfo.minScaleDenominator)) {
          definition.minScale = layerInfo.minScaleDenominator;
        }
        if (Legato.Lang.ObjectUtils
            .exists(layerInfo.maxScaleDenominator)) {
          definition.maxScale = layerInfo.maxScaleDenominator;
        }
        if (layerInfo.srs.length > 0) {
          definition.projection = new OpenLayers.Projection(
              layerInfo.srs[0]);
        }
        if (Legato.Lang.ObjectUtils.exists(layerInfo.currentFormat)) {
          definition.format = layerInfo.currentFormat;
        }
        if (Legato.Lang.ObjectUtils.exists(layerInfo.currentStyle)) {
          definition.style = layerInfo.currentStyle.name;
          if (Legato.Lang.ObjectUtils
              .exists(layerInfo.currentStyle.legend
                  && layerInfo.currentStyle.legend.onlineResource)) {
            definition.legend = new Legato.Layer.Legend(
                layerInfo.currentStyle.legend.onlineResource);
          }
        }
        var name;
        if (options) {
          for (name in options) {
            if (options.hasOwnProperty(name)) {
              definition[name] = options[name];
            }
          }
        }
        if (layerInfo.extension) {
          for (name in layerInfo.extension) {
            if (layerInfo.extension.hasOwnProperty(name)) {
              definition[name] = layerInfo.extension[name];
            }
          }
        }
        this.populateLayerDefinition(definition, layerInfo, options,
            beanFactory);
        var beanDefinition = this.layerType.define(definition);
        return beanFactory.retrieveBean(beanDefinition);
      },
      populateLayerDefinition : function(definition, layerInfo, options,
          beanFactory) {

      }
    });

Legato.Format.ExtendedWMC.DEFAULT_LAYER_FACTORIES = {};
Legato.Format.ExtendedWMC.registerDefaultLayerType = function(name, layerType) {
  Legato.Format.ExtendedWMC.DEFAULT_LAYER_FACTORIES[name] = new Legato.Format.ExtendedWMC.LayerFactory(
      layerType);
};
Legato.Format.ExtendedWMC.registerDefaultLayerFactory = function(name,
    layerFactory) {
  Legato.Format.ExtendedWMC.DEFAULT_LAYER_FACTORIES[name] = layerFactory;
};