/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: leine $
 * @version $Rev: 72369 $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: ViewContextValueProvider
 *
 * Inherits from: - <Legato.Control>
 */
Legato.Control.ViewContextValueProvider = OpenLayers.Class(Legato.Control,
{
  /*
   * {Date} Holds the last timestamp where an viewcontext update has been
   * propagated to the configured url.
   */
  lastUpdate: null,

  /*
   * Property: frequency
   * {Double} This is the minimum interval in ms at which update AJAX requests are made.
   */
  frequency: null,

  /*
   * Property: delayedPropagation
   * {Number} Timeout id of delayed propagateValue.
   */
  delayedPropagation: null,

  /*
   * Property: url
   * {String} The URL which will been used for submitting the viewcontext.
   */
  url: null,

  /*
   * Property: method
   * {String} The HTTP method to use for the request
   */
  method: null,

  /*
   * Function: initialize
   * Basic ctor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control
   * The following keys are supported
   * - observer {Object} A hashmap containing an observer
   *
   * The following keys are supported for an observer element
   * - url {String} any url which will been used for submitting all viewcontext changes
   * - frequency {Double} This is the minimum interval in ms at which update AJAX requests are made.
   * - method {String} The HTTP method to use for the request.
   */
  initialize : function(options) {
    Legato.Control.prototype.initialize.apply(this, arguments);
    this.lastUpdate = new Date();

    Legato.Util.Ensure.ensureNotEmptyString(this.url, 'To use the ViewContextValueProvider you have to specify at least an url.');

    if(Legato.Lang.ObjectUtils.isNullOrUndefined(this.frequency) ||
        !Legato.Lang.ObjectUtils.isNumber(this.frequency) ||
        this.frequency < 500){
      this.frequency = 2000;
    }
    if(Legato.Lang.ObjectUtils.isNullOrUndefined(this.method) ||
    !Legato.Lang.ObjectUtils.isString(this.method)){
      this.method = 'POST';
    }
  },

  /*
   * Function: destroy
   *
   * See Also:
   * <OpenLayer.Control>
   */
  destroy : function(){
    this.map.events.un({
      'move':                 this.propagateValue,
      'addlayer':             this.propagateValue,
      'removelayer':          this.propagateValue,
      'moveend':              this.propagateValueFreqControl,
      'changebaselayer':      this.propagateValue,
      'legato.selectlayer':   this.propagateValue,
      'legato.deselectlayer':   this.propagateValue,
      'resize':               this.propagateValue,
      scope:                  this
    });

    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /*
   * Function: setMap
   *
   * See Also:
   * <OpenLayer.Control>
   */
  setMap : function(map) {
    OpenLayers.Control.prototype.setMap.apply(this, arguments);

    this.map.events.on({
      'addlayer':             this.propagateValue,
      'removelayer':          this.propagateValue,
      'changelayer':          this.propagateValue,
      'changebaselayer':      this.propagateValue,
      'moveend':              this.propagateValueFreqControl,
      'legato.selectlayer':   this.propagateValue,
      'legato.deselectlayer':   this.propagateValue,
      'resize':               this.propagateValue,
      scope:                  this
    });
   },

  /*
   * Function: draw
   *
   * See Also:
   * <OpenLayer.Control>
   */
  draw : function() {
    return this.div;
  },

  propagateValueFreqControl: function() {
    if(Legato.Lang.ObjectUtils.exists(this.lastUpdate) &&
      (new Date().getTime() - this.lastUpdate.getTime() < this.frequency)) {
      if(Legato.Lang.ObjectUtils.isNullOrUndefined(this.delayedPropagation)) {
        this.delayedPropagation = window.setTimeout(
            OpenLayers.Function.bind(function() {
                this.propagateValue();
            }, this),
            this.frequency
        );
      }
      return;
    }
    this.propagateValue();
  },

  /*
   * Get the current map viewContext and reports it back to the server
   */
  propagateValue: function(){
    //May occur if the map is still constucting itself
    if(this.map.calculateBounds() === null){
      return;
    }

    if(Legato.Lang.ObjectUtils.exists(this.delayedPropagation)) {
      window.clearTimeout(this.delayedPropagation);
      this.delayedPropagation = null;
    }

    this.lastUpdate = new Date();

    var viewContext = {
        boundingbox: this.map.calculateBounds().toBBOX(),
        layers: Legato.OpenLayers.Map.Util.getLayerIds(Legato.OpenLayers.Map.Util.getLayersWithoutLegatoIgnore(this.map.layers)),
        hiddenlayers: Legato.OpenLayers.Map.Util.getLayerIds(Legato.OpenLayers.Map.Util.getLayersWithoutLegatoIgnore(Legato.OpenLayers.Map.Util.getInvisibleLayers(this.map.layers))),
        'mapsize.width': this.map.getSize().w,
        'mapsize.height': this.map.getSize().h,
        selectedlayer: Legato.OpenLayers.Map.Util.getLayerIds(Legato.OpenLayers.Map.Util.getLayersWithoutLegatoIgnore(Legato.OpenLayers.Map.Util.getSelectedLayers(this.map.layers))).toString()
    };

    var req;
    if(this.method == 'GET'){
      req = new OpenLayers.Request.GET({
        url: this.url,
        params: viewContext
      });
    } else {
      req = new OpenLayers.Request.POST({
        url: this.url,
        params: viewContext
      });
    }
  },


  CLASS_NAME : 'Legato.Control.ViewContextValueProvider'
});

/**
 * Structure: lc:ViewContextValueProvider
 * XML based config for a <Legato.Control.ViewContextValueProvider>.
 *
 * See Also:
 * - <Legato.Control>
 * - <Legato.Beans.BeanFactory>
 * - <QName>
 *
 * A valid config example for a ViewContextValueProvider would be:
 * (start code)
 * <lc:ViewContextValueProvider url="http://foo.com/viewContext" method="POST" frequency="2000" />
 * (end)
 */
Legato.Control.ViewContextValueProvider.Bean = Legato.Control.Bean.Extend(

  'Legato.Control.ViewContextValueProvider',

  /*
   * Property: QName
   * The qualified name for this control. Needed by XML config to instantiate it.
   */
  Legato.Control.QName('ViewContextValueProvider'),
  {
    _constructor: Legato.Control.ViewContextValueProvider,
    options:
    {
    /*
     * Property: url
     *
     * The url which will been used for providing the current viewcontext of the client.
     */
      url: Legato.Lang.String,

     /*
      * Property: method
      * The HTTP method to use for the request. Possible values are GET or POST
      */
      method: Legato.Lang.String.Enum('HTTPMETHOD', {
        'POST': 'POST',
        'GET': 'GET'
      }),

      /*
       * Property: frequency
       * This is the minimum interval in ms at which update AJAX requests are made.
       */
      frequency: Legato.Lang.Double
    }
  }
);
