/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Class: WPS.Form
 */

WPS.Form = OpenLayers.Class({
  EVENT_TYPES : [ 'geoInputActivation', 'processSelectionCancellation', 'processExecutionCancellation', 'on', 'off' ],
  events : null,
  processSelector : null,
  processComponent : null,

  /**
   * Element: allowSelection {Boolean} State variable which remembers that the
   * form is currently on. Might also be set by the WPS.Inceptor class!
   *
   */
  isOn : false,
  map : null,
  currentProcessId : null,
  currentProcessModel : null,
  capabilities : null,
  container : null,
  childContainer : null,
  onCallback : null,
  offCallback : null,
  wpsOperations : null,
  predicate : null,
  resultFactory : null,
  popupCallback : null,
  errorCallback : null,
  replaceContainerCallback : null,
  txtExecute : null,
  txtSelect : null,
  txtCancel : null,
  componentFactoryOptions : null,

  /**
   * Constructor: WPS.Form
   *
   * Creates a new WPS form.
   *
   * Parameters: map - {<OpenLayers.Map>} The map. options - {Object} Optional
   * object whose properties will be set on the instance.
   */
  initialize : function(map, options) {
    Legato.Util.Ensure.ensureExists(map, "map must be defined");
    this.map = map;
    OpenLayers.Util.extend(this, options);

    this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES);
    if (this.container === null) {
      this.container = document.createElement('div');
      this.container.className = 'wpsForm';
    }
    if (this.wpsOperations === null) {
      this.wpsOperations = WpsOperations;
    }
    if (this.resultFactory === null) {
      this.resultFactory = new WPS.Result.Factory(this.map, this);
    }
    if (!Legato.Lang.ObjectUtils.isString(this.txtExecute)) {
      this.txtExecute = WPS.I18n.getMessage('WPS.Button.OK');
    }
    if (!Legato.Lang.ObjectUtils.isString(this.txtSelect)) {
      this.txtSelect = WPS.I18n.getMessage('WPS.Button.OK');
    }
    if (!Legato.Lang.ObjectUtils.isString(this.txtCancel)) {
      this.txtCancel = WPS.I18n.getMessage('WPS.Button.Cancel');
    }
    if (!Legato.Lang.ObjectUtils.isObject(this.componentFactoryOptions)) {
      this.componentFactoryOptions = {};
    }

    this.componentFactoryOptions.styles = this.styles;
  },

  initCapabilities : function(callback) {
    this.capabilitiesReady = false;
    var getCapabilitiesRequest = {
      service : 'WPS',
      acceptVersions : {
        version : [ '1.0.0' ]
      }
    };
    this.wpsOperations.getCapabilities(getCapabilitiesRequest, OpenLayers.Function.bind(function(capabilities) {
      if (Legato.Lang.ObjectUtils.exists(capabilities) && Legato.Lang.ObjectUtils.exists(capabilities.processOfferings)
          && Legato.Lang.ObjectUtils.exists(capabilities.processOfferings.process)) {
        var predicate;
        if (Legato.Lang.ObjectUtils.exists(this.predicate)) {
          predicate = this.predicate;
        } else {
          predicate = this.createDefaultPredicate();
        }
        var processes = [];
        for ( var processIndex = 0; processIndex < capabilities.processOfferings.process.length; processIndex++) {
          var process = capabilities.processOfferings.process[processIndex];
          if (predicate.evaluate(process)) {
            processes.push(process);
          }
        }
        this.capabilities = processes;

      } else {
        this.capabilities = null;
      }
      this.capabilitiesReady = true;
      if (Legato.Lang.ObjectUtils.isFunction(callback)) {
        callback();
      }
    }, this));
  },
  createDefaultPredicate : function() {
    /*
     * A WPS Process may be inserted by a possible MAP META TAG. If the current
     * MAP has a TAG named "wps:include:form:metadata:tag" and its value
     * matches the same named TAG a WPS Process, then this WPS Process will be listed
     * in our WPS Form.
     */
    return new WPS.Predicate.MapTags(this.map, {
      prefix : 'wps:include:form:',
      includeInfix :'metadata:tag',
      defaultIncludes : true,
      predicateCallback: function(value){
        return new WPS.Predicate.Metadata(value);
      }
    });
  },
  onWithCapabilities : function() {
    if (this.capabilitiesReady) {
      if (!Legato.Lang.ObjectUtils.exists(this.currentProcessId)) {
        this.processSelector = new WPS.ProcessSelector(this.capabilities, OpenLayers.Function.bind(this.selectProcess, this), OpenLayers.Function.bind(
            this.cancelSelection, this), {
          txtSelect : this.txtSelect,
          txtCancel : this.txtCancel
        });
        this.replaceChildContainer(this.processSelector.container);
      } else {
        //this will work because the process component is destroyed after pressing cancel / closing form
        if (!Legato.Lang.ObjectUtils.exists(this.processComponent)) {
          this.processComponent = new WPS.Component.Process(
                  this.currentProcessModel,
                  this.map,
                  OpenLayers.Function.bind(this.execute, this),
                  OpenLayers.Function.bind(this.cancelExecution, this), {
                txtExecute : this.txtExecute,
                txtCancel : this.txtCancel,
                componentFactoryOptions : this.componentFactoryOptions
              });
              this.processComponent.events.on({
                'geoInputActivation' : this.geoInputActivated,
            scope : this
          });
        }
        this.replaceChildContainer(this.processComponent.container);
      }

      this.isOn = true;
      this.events.triggerEvent('on', {});
      Legato.XML.ElementUtils.show(this.container);
      if (this.onCallback) {
        this.onCallback(this);
      }
      return;
    } else {
      setTimeout(OpenLayers.Function.bind(this.onWithCapabilities, this), 200);
    }
  },

  destroy : function() {
    if (Legato.Lang.ObjectUtils.exists(this.events)) {
      this.events.destroy();
      this.events = null;
    }
    this.destroyChildren();
  },

  destroyProcessSelector : function() {
    this.processSelector.destroy();
    this.processSelector = null;
    this.childContainer = null;
    this.container.innerHTML = '';
  },

  destroyProcessComponent : function() {
  if (Legato.Lang.ObjectUtils.exists(this.processComponent)) {
    if (Legato.Lang.ObjectUtils.exists(this.processComponent.events)) {
      this.processComponent.events.un({
        'geoInputActivation' : this.geoInputActivated,
        scope : this
      });
    }
    this.processComponent.destroy();
  }
    this.processComponent = null;
    this.childContainer = null;
    this.container.innerHTML = '';
  },

  destroyChildren : function() {
    if (Legato.Lang.ObjectUtils.exists(this.processSelector)) {
      this.destroyProcessSelector();
    }
    if (Legato.Lang.ObjectUtils.exists(this.processComponent)) {
      this.destroyProcessComponent();
    }
  },

  geoInputActivated : function(event) {
    this.events.triggerEvent('geoInputActivation', {});
  },

  on : function() {
    if (!Legato.Lang.ObjectUtils.exists(this.capabilities)) {
      this.initCapabilities(OpenLayers.Function.bind(this.onWithCapabilities, this));
    }
    else {
      this.onWithCapabilities();
    }
  },

  replaceChildContainer : function(container) {
    this.container.innerHTML = '';
    this.childContainer = container;
    this.container.appendChild(this.childContainer);
    if (Legato.Lang.ObjectUtils.isFunction(this.replaceContainerCallback)) {
      this.replaceContainerCallback(this.childContainer);
    }
  },

  selectProcess : function(processId, closeForm) {
    this.currentProcessId = null;
    var describeProcessRequest = {
      identifier : [ {
        value : processId
      } ]
    };
    this.wpsOperations.describeProcess(describeProcessRequest, OpenLayers.Function.bind(function(processDescriptions) {
      // var processDescription =
      // Legato.Lang.ArrayUtils.get(processDescriptions.processDescription,
      // 0);
      var processDescription = processDescriptions.processDescription[0];
      if (Legato.Lang.ObjectUtils.exists(processDescription)) {
        this.activateProcessDescription(processDescription, closeForm);
      } else {
        // should never happen
        alert('Could not retrieve WPS process description from server');
      }
    }, this));
  },
  activateProcessDescription : function(processDescription, closeForm) {
    var processModel = new WPS.Model.Process(this.map, processDescription);
    if (processModel.needsUserInput()) {
      this.currentProcessId = processDescription.identifier.value;
      this.currentProcessModel = processModel;
      var onExecute = OpenLayers.Function.bind(this.execute, this);
      var onCancel;
      if (Legato.Lang.ObjectUtils.isTrue(closeForm)) {
        onCancel = OpenLayers.Function.bind(this.cancelExecutionAndCloseForm, this);
      }
      else {
        onCancel = OpenLayers.Function.bind(this.cancelExecutionAndShowSelection, this);
      }
      this.processComponent = new WPS.Component.Process(this.currentProcessModel, this.map, onExecute, onCancel, {
        txtExecute : this.txtExecute,
        txtCancel : this.txtCancel,
        componentFactoryOptions : this.componentFactoryOptions,
        styles : this.styles,
        hideCancelButton : closeForm
      });
      this.processComponent.events.on({
        'geoInputActivation' : this.geoInputActivated,
        scope : this
      });
      this.replaceChildContainer(this.processComponent.container);
    } else {
      var data = processModel.getData();
      this.execute(null, data);
      processModel.destroy();
      if (Legato.Lang.ObjectUtils.isTrue(closeForm)) {
        this.off();
      }
    }

  },
  cancelSelection : function() {
    this.currentProcessId = null;
    this.events.triggerEvent('processSelectionCancellation', {});
  },
  cancelExecutionAndCloseForm : function() {
    this.cancelExecution();
    this.off();
  },
  cancelExecutionAndShowSelection : function() {
    this.cancelExecution();
  },
  cancelExecution : function() {
    this.destroyProcessComponent();
    if (Legato.Lang.ObjectUtils.exists(this.currentProcessModel)) {
      this.currentProcessModel.destroy();
      this.currentProcessModel = null;
    }

    this.events.triggerEvent('processExecutionCancellation', {
      processId : this.currentProcessId
    });

    this.currentProcessId = null;

    // If there are no capabilities we just turn off
    if (!Legato.Lang.ObjectUtils.exists(this.capabilities)) {
      this.off();
      return;
    }

    this.processSelector = new WPS.ProcessSelector(this.capabilities, OpenLayers.Function.bind(this.selectProcess, this), OpenLayers.Function.bind(
        this.cancelSelection, this), {
      txtCancel : this.txtCancel,
      txtExecute : this.txtExecute
    });
    this.replaceChildContainer(this.processSelector.container);
  },

  execute : function(process, executeRequest) {
    this.wpsOperations.execute(executeRequest, OpenLayers.Function.bind(function(executeResponse) {
      var executionContext = new WPS.Result.ExecutionContext(this.map);
      var result = this.resultFactory.createResult(executeResponse, this.popupCallback, this.errorCallback, this.onCallback);
      result.execute(executionContext);
      if (Legato.Lang.ObjectUtils.exists(this.processComponent) && Legato.Lang.Function.isInstance(this.processComponent.executeComplete)) {
        this.processComponent.executeComplete();
      }
      executionContext.destroy();
    }, this));
  },

  off : function() {
    this.destroyChildren();
    this.isOn = false;
    this.events.triggerEvent('off', {});
    Legato.XML.ElementUtils.hide(this.container);
    if (this.offCallback) {
      this.offCallback(this);
    }
  },

  activate : function() {
    if (this.isOn && Legato.Lang.ObjectUtils.exists(this.processComponent)) {
      this.processComponent.activate();
    }
  },

  deactivate : function() {
    if (this.isOn && Legato.Lang.ObjectUtils.exists(this.processComponent)) {
      this.processComponent.deactivate();
    }
  },

  CLASS_NAME : 'WPS.Form'
});

/**
 * Structure: lwps:Form XML-based configuration for a {<WPS.Form>}.
 *
 * Configuration example: (start code) <lwps:Form
 * xmlns:lwps="http://www.disy.net/WPS/wps"> <map> <lb:Reference target="map" />
 * </map> <wpsOperations> <lb:Reference target="WpsOperations" />
 * </wpsOperations> </lwps:Form> (end)
 */
WPS.Form.Bean = Legato.Beans.Object.Extend('WPS.Form',
//
WPS.QName('Form'),
//
{
  _constructor : WPS.Form,
  constructorArguments : {
    /**
     * Property: map {<OpenLayers.Map>} The map.
     */
    map : OpenLayers.Map.Bean
  },
  constructorOrder : [ 'map' ],
  options : {
    container : Legato.Lang.Element,
    onCallback : Legato.Lang.Function,
    offCallback : Legato.Lang.Function,
    popupCallback : Legato.Lang.Function,
    errorCallback : Legato.Lang.Function,
    replaceContainerCallback : Legato.Lang.Function,
    /**
     * Property: wpsOperations {Object} WPS operations object.
     */
    wpsOperations : Legato.Beans.Object,
    predicate : Legato.Beans.Object,
    txtExecute : Legato.Lang.String,
    txtSelect : Legato.Lang.String,
    txtCancel : Legato.Lang.String,
    componentFactoryOptions : Legato.Beans['Object{}'],
    styles : Legato.Beans.Object.List()
  }
});