/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * @author $Author: valikov $
 * @version $Rev: 70574 $
 * @requires Legato.js
 * @requires Legato/Lang.js
 * @requires Legato/Lang/Class.js
 * @requires Legato/Util.js
 */

/*
 * Class: Legato.Util.HashSet
 *
 * Implements a map-backed set of strings.
 */
Legato.Util.HashSet = Legato.Lang.Class( {
  map :null,
  _size :0,
  /*
   * Constructor: initialize
   *
   * Constructs the set.
   */
  initialize : function() {
    this.map = {};
  },
  /*
   * Function: size
   *
   * Returns: {Integer} Number of the elements in the set.
   */
  size : function() {
    return this._size;
  },
  /*
   * Function: isEmpty
   *
   * Returns: {Boolean} True if set is empty, false otherwise.
   */
  isEmpty : function() {
    return (this._size === 0);
  },
  /*
   * Function: contains
   *
   * Checks if the set contains the value or not.
   *
   * Parameters:
   *
   * value - {String} element to add to the set.
   *
   * Returns: {Boolean} True if given value is belongs to this set, false
   * otherwise.
   */
  contains : function(value) {
    Legato.Util.Ensure.ensureString(value);
    var marker = this.map[value];
    if (marker) {
      return true;
    } else {
      return false;
    }

  },
  /*
   * Function: add
   *
   * Adds the value to the set.
   *
   * Parameters:
   *
   * value - {String} element to be added to the set.
   *
   * Returns: {Boolean} True if an element was added to the set, false if it was
   * already a member of the set.
   *
   */
  add : function(value) {
    Legato.Util.Ensure.ensureString(value);
    var marker = this.map[value];
    if (marker) {
      return false;
    } else {
      this.map[value] = true;
      this._size = this._size + 1;
      return true;
    }
  },
  /*
   * Function: addAll
   *
   * Adds all the elements of the given collection to this set.
   *
   * Parameters:
   *
   * set - set of string to be added to this set.
   *
   * Returns: {Boolean} True if new element were added to the set, false
   * otherwise.
   */
  addAll : function(set) {
    Legato.Util.Ensure.ensureObject(set);
    Legato.Util.Ensure.ensureFunction(set.toArray);
    var array = set.toArray();
    var changed = false;
    for (var index = 0; index < array.length; index++) {
      var item = array[index];
      var added = this.add(item);
      changed = (changed || added);
    }
    return changed;

  },
  /*
   * Function: remove
   *
   * Removes the element from the set.
   *
   * Parameters:
   *
   * value - {String} value to be removed from the set.
   *
   * Returns: {Boolean} True if the value was removed from the given set, false
   * if not (i.e. the value was not an element of this set).
   */
  remove : function(value) {
    Legato.Util.Ensure.ensureString(value);
    var marker = this.map[value];
    if (marker) {
      delete this.map[value];
      this._size = this._size - 1;
      return true;
    } else {
      return false;
    }
  },
  /*
   * Function: toArray
   *
   * Returns: {Array(String)} Array of elements contained in this set. The order
   * is not deterministic.
   */
  toArray : function() {
    var array = [];
    for ( var value in this.map)
    {
      if (this.map.hasOwnProperty(value)) {
        array.push(value);
      }
    }
    return array;
  },
  /*
   * Function: clear
   *
   * Clears the set.
   */
  clear : function() {
    this.map = {};
    this._size = 0;
  }
});