/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Class: WPS.Inceptor
 *
 * Activates a default process description in the target WPS form.
 *
 * The default process description is provided by the configured inception
 * function of the form:
 * (start code)
 * function incept(activate) {
 *   var myProcessDescription = ...;
 *   activate(myProcessDescription);
 * }
 * (end)
 */
WPS.Inceptor = OpenLayers.Class({
  map : null,
  form : null,
  inception : null,
  resultFactory: null,
  autoExecutionPredicate: null,
  /**
   * Constructor: WPS.Inceptor
   *
   * Creates a new WPS inceptor.
   *
   * Parameters:
   * map - {<OpenLayers.Map>} The map.
   * form - {<WPS.Form>} Target WPS form.
   * inception - {Function} Inception function which provides
   * the default process description.
   */
  initialize : function(map, form, inception, options) {
    Legato.Util.Ensure.ensureObject(map);
    Legato.Util.Ensure.ensureObject(form);
    Legato.Util.Ensure.ensureFunction(inception);
    this.map = map;
    this.form = form;
    this.inception = inception;
//    this.map.events.on({
//      'legato.refresh' : this.incept,
//      scope : this
//    });
    this.autoExecutionPredicate = new WPS.Predicate.Metadata('auto', {about : 'execution'});
    this.resultFactory = new WPS.Result.Factory(map, null);

    this.incept();
  },
  destroy : function() {
//    this.map.events.un({
//      'legato.refresh' : this.incept,
//      scope : this
//    });
  },
  incept : function() {
    var that = this;
    that.inception(function(processDescription) {
      if (Legato.Lang.ObjectUtils.exists(processDescription)) {
        if (that.autoExecutionPredicate.evaluate(processDescription))
        {
          var processModel = new WPS.Model.Process(that.map, processDescription);

          var executeRequest = processModel.getData();
          that.form.execute(null, executeRequest);
        }
        else
        {
          that.form.activateProcessDescription(processDescription, true);
          //Has to be set, otherwise the form does not know that it has been activated by the inceptor
          that.form.isOn = true;
          if (that.form.onCallback) {
            that.form.onCallback(that.form);
          }
        }
      }
    });
  },
  CLASS_NAME : 'WPS.Inceptor'
});
/**
 * Structure: lwps:Inceptor
 * XML-based configuration for a {<WPS.Inceptor>}.
 *
 * Configuration example:
 * (start code)
 * <lwps:Inceptor xmlns:lwps="http://www.disy.net/WPS/wps">
 *   <map>
 *     <lb:Reference target="map" />
 *   </map>
 *   <form>
 *     <lb:Reference target="WPSForm" />
 *   </form>
 *   <inception>
 *     <lb:Reference target="WpsInception" />
 *   </inception>
 * </lwps:Inceptor>
 * (end)
 */
WPS.Inceptor.Bean = Legato.Beans.Object.Extend('WPS.Inceptor',
//
WPS.QName('Inceptor'),
//
{
  _constructor : WPS.Inceptor,
  constructorArguments : {
    /**
     * Property: map
     * {<OpenLayers.Map>} The map.
     */
    map : Legato.Beans.Object,
    /**
     * Property: form
     * {<WPS.Form>} Target WPS form.
     */
    form : Legato.Beans.Object,
    /**
     * Property: inception
     * {Function} Inception function.
     */
    inception : Legato.Lang.Function
  },
  constructorOrder : [ 'map', 'form', 'inception' ]
});