/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: valikov $
 * @version $Rev: 91043 $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: Legato.Control.BBoxValueProvider
 * ValueProvider which gives you the possibility to be notified about every
 * change of the current BBox Selection. The notification could be a JS Callback
 * or a Object where its property will be overwritten each time the watched value
 * changes.
 *
 *
 * Inherits from: - <Legato.Control>
 */
Legato.Control.BBoxValueProvider = OpenLayers.Class(Legato.Control,
{
  /*
   * Function: initialize
   * Basic ctor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control
   * The following keys are supported
   * - observer {Object} A hashmap containing an observer
   *
   * The following keys are supported for an observer element
   * - target {object} an object which wants to observe,
   * - property{function|String} the propety or function which should be used for providing the observed value
   * - bBoxProperty {String} any Property name of an <OpenLayers.Bounds> instance
   * - write {Boolean} true = value changes of observer will be given back to current map, false otherwise
   */
  initialize : function(options) {
    OpenLayers.Control.prototype.initialize.apply(this, arguments);
  },

  /*
   * Function: destroy
   *
   * See Also:
   * <OpenLayer.Control>
   */
  destroy : function() {
    this.map.events.un( {
      'moveend' : this.propagateValue,
      scope : this
    });

    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /*
   * Function: setMap
   *
   * See Also:
   * <OpenLayer.Control>
   */
  setMap : function(map) {
    OpenLayers.Control.prototype.setMap.apply(this, arguments);

    this.map.events.on( {
      'moveend' : this.propagateValue,
      'legato.refresh' : this.examineValue,
      scope : this
    });
   },

  /*
   * Function: draw
   *
   * See Also:
   * <OpenLayer.Control>
   */
  draw : function() {
    this.examineValue();
    this.propagateValue();
    return this.div;
  },

  /*
   * Inits this control by the given option targets based on their
   * given values if exists.
   */
  examineValue: function(){
    if(!Legato.Lang.ObjectUtils.isFunction(this.getValue)){
      return;
    }
    var bbox = this.getValue();
    if(Legato.OpenLayers.Map.Util.isValidBBox(bbox) &&
       Legato.OpenLayers.Map.Util.getBaseLayers(this.map.layers).length > 0){
      this.map.zoomToExtent(bbox);
    }
  },

  /*
   * updates all subscribed target's values by requesting the current used
   * bbox. Their values will been provided back to given targets
   */
  propagateValue: function(){
    if(!Legato.Lang.ObjectUtils.isFunction(this.setValue)){
      return;
    }

    var bbox = this.map.calculateBounds();

    this.setValue(bbox);
  },

  CLASS_NAME : 'Legato.Control.BBoxValueProvider'
});

/**
 * Structure: lc:BBoxValueProvider
 * XML based config for a <Legato.Control.BBoxValueProvider>.
 *
 * See Also:
 * - <Legato.Control>
 * - <Legato.Beans.BeanFactory>
 * - <QName>
 *
 * A valid config example for a BBoxValueProvider would be:
 * (start code)
 * <lc:BBoxValueProvider left="bboxLeft" right="bboxRight" top="bboxTop" bottom="bboxBottom"/>
 * (end)
 */
Legato.Control.BBoxValueProvider.Bean = Legato.Control.Bean.Extend(

  'Legato.Control.BBoxValueProvider',

  /*
   * Property: QName
   * The qualified name for this control. Needed by XML config to instantiate it.
   */
  Legato.Control.QName('BBoxValueProvider'),
  {
   /*
    * Creates a basic <Legato.Control.BBoxValueProvider> instance. By default this instance
    * will update the DOMElements by any change made in map. It only provides a basic setter
    * function. So if given DOMElements contain any values the constructed map will load
    * with this initial bbox.
    *
    * Parameters:
    * left - {String} The configured DOMElement ID where the left bbox value should be written into
    * bottom - {String} The configured DOMElement ID where the bottom bbox value should be written into
    * right - {String} The configured DOMElement ID where the right bbox value should be written into
    * top - {String} The configured DOMElement ID where the top bbox value should be written into
    *
    * Returns:
    * <Legato.Control.BBoxValueProvider>
    */
    factoryFunction : function(left, bottom, right, top)
    {
      var leftElement   = null;
      var bottomElement = null;
      var rightElement  = null;
      var topElement    = null;

      if(Legato.Lang.ObjectUtils.isString(left)){
        leftElement = OpenLayers.Util.getElement(left);
        Legato.Util.Ensure.ensureExists(leftElement, 'BBoxValueProvider: The configured DOMElement with the id' + left + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(bottom)){
        bottomElement = OpenLayers.Util.getElement(bottom);
        Legato.Util.Ensure.ensureExists(bottomElement, 'BBoxValueProvider: The configured DOMElement with the id' + bottom + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(right)){
        rightElement = OpenLayers.Util.getElement(right);
        Legato.Util.Ensure.ensureExists(rightElement, 'BBoxValueProvider: The configured DOMElement with the id' + right + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(top)){
        topElement = OpenLayers.Util.getElement(top);
        Legato.Util.Ensure.ensureExists(topElement, 'BBoxValueProvider: The configured DOMElement with the id' + top + ' could not been found.');
      }

      return new Legato.Control.BBoxValueProvider({
        elements: {
          left: leftElement,
          bottom: bottomElement,
          right: rightElement,
          top: topElement
        },
        setValue: function(BBox){
          OpenLayers.Console.debug('BBoxValueProvider.setValue:' + BBox);
          if(!Legato.Lang.ObjectUtils.exists(BBox)){
            return;
          }

          this.elements.left.value   = BBox.left;
          this.elements.right.value  = BBox.right;
          this.elements.top.value    = BBox.top;
          this.elements.bottom.value = BBox.bottom;
        },
        getValue: function(){
          OpenLayers.Console.debug('BBoxValueProvider.getValue');
          try{
            var left    = Legato.Lang.NumberUtils.toFloat(this.elements.left.value);
            var right   = Legato.Lang.NumberUtils.toFloat(this.elements.right.value);
            var top     = Legato.Lang.NumberUtils.toFloat(this.elements.top.value);
            var bottom  = Legato.Lang.NumberUtils.toFloat(this.elements.bottom.value);

            return new OpenLayers.Bounds(left, bottom, right, top);
          }catch(ex){
            OpenLayers.Console.warn(ex);
            return new OpenLayers.Bounds(null, null, null, null);
          }
        }
      });
    },
    constructorArguments:
    {
      /*
      * Property: left
      * Defines the {DOMElement} id where the left <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      left: Legato.Lang.String,

     /*
      * Property: right
      * Defines the {DOMElement} id where the right <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      right: Legato.Lang.String,

     /*
      * Property: top
      * Defines the {DOMElement} id where the top <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      top: Legato.Lang.String,

     /*
      * Property: bottom
      * Defines the {DOMElement} id where the bottom <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      bottom: Legato.Lang.String
    },
    constructorOrder: ['left', 'bottom', 'right', 'top']
  }
);
