/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: valikov $
 * @version $Rev: 91043 $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: Legato.Control.LayersValueProvider
 * ValueProvider which gives you the possibility to be notified about every
 * change of all current used <OpenLayers.Layer> instances inside the <OpenLayers.Map>.
 *
 * You can watch a defined property of all <OpenLayers.Layer> instances. All layer values
 * will be given back by a comma seperated list.
 *
 * Inherits from: - <Legato.Control>
 */
Legato.Control.LayersValueProvider = OpenLayers.Class(Legato.Control,
{
  /*
   * Function: initialize
   * Basic ctor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control
   * The following keys are supported
   * - target {object} an object which wants to observe,
   * - property{function|String} the propety or function which should be used for providing the observed value
   * - layerValue {String} any Property name of an <OpenLayers.Layer> instance
   * - layerVisible {Boolean} true = only watch visible layers, false = only watch invisible layers. Default is true.
   */
  initialize : function(options) {
    OpenLayers.Control.prototype.initialize.apply(this, arguments);
  },

  /*
   * Function: destroy
   *
   * See Also:
   * <OpenLayer.Control>
   */
   destroy : function() {
    this.map.events.un( {
      'addlayer' : this.propagateValue,
      'changelayer' : this.propagateValue,
      'removelayer' : this.propagateValue,
      'changebaselayer' : this.propagateValue,
      scope : this
    });

    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /*
   * Function: setMap
   *
   * See Also:
   * <OpenLayer.Control>
   */
  setMap : function(map) {
    OpenLayers.Control.prototype.setMap.apply(this, arguments);

    this.map.events.on( {
      'addlayer' : this.propagateValue,
      'changelayer' : this.propagateValue,
      'removelayer' : this.propagateValue,
      'changebaselayer' : this.propagateValue,
      scope : this
    });
   },

  /*
   * Function: draw
   *
   * See Also:
   * <OpenLayer.Control>
   */
  draw : function() {
    this.propagateValue();
    return this.div;
  },

  /*
   * updates all subscribed target's values by iterating
   * throw all layers and firing the corresponding target setters
   * if valid and present.
   */
  propagateValue: function(){
    if(!Legato.Lang.ObjectUtils.isFunction(this.setValue)){
      return;
    }

    this.setValue(this.map.layers);
  },

  CLASS_NAME : 'Legato.Control.LayersValueProvider'
});

/**
 * Structure: lc:LayersValueProvider
 * XML based config for a <Legato.Control.LayersValueProvider>.
 *
 * See Also:
 * - <Legato.Control>
 * - <Legato.Beans.BeanFactory>
 * - <QName>
 *
 * A valid config example for a LayersValueProvider would be:
 * (start code)
 * <lc:LayersValueProvider layerList="layerList" layerListVisible="layerListVisible" layerListInvisible="layerListInvisible"/>
 * (end)
 */
Legato.Control.LayersValueProvider.Bean = Legato.Control.Bean.Extend(

  'Legato.Control.LayersValueProvider',

  /*
   * Property: QName
   * The qualified name for this control. Needed by XML config to instantiate it.
   */
  Legato.Control.QName('LayersValueProvider'),
  {
   /*
    * Creates a basic <Legato.Control.LayersValueProvider> instance. By default this control will write
    * into three configured DOMElements. All changes in given DOMElements will not cause an update
    * of given map.
    *
    * Parameters:
    * layerList - {String} The configured DOMElement ID where all layers in their correct order should be written into
    * layerListVisible - {String} The configured DOMElement ID where all visible layers (baselayers and overlays) should be written into
    * layerListInvisible - {String} The configured DOMElement ID where all invisible layers (baselayers and overlays) should be written into
    *
    * Returns:
    * <Legato.Control.LayersValueProvider>
    */
    factoryFunction : function(layerList, layerListVisible, layerListInvisible)
    {
      var layerlistElement = null;
      var layerListVisibleElement = null;
      var layerListInvisibleElement = null;

      if(Legato.Lang.ObjectUtils.isString(layerList)){
        layerlistElement = OpenLayers.Util.getElement(layerList);
        Legato.Util.Ensure.ensureExists(layerlistElement, 'LayersValueProvider: The configured DOMElement with the id' + layerList + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(layerListVisible)){
        layerListVisibleElement = OpenLayers.Util.getElement(layerListVisible);
        Legato.Util.Ensure.ensureExists(layerListVisibleElement, 'LayersValueProvider: The configured DOMElement with the id' + layerListVisible + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(layerListInvisible)){
        layerListInvisibleElement = OpenLayers.Util.getElement(layerListInvisible);
        Legato.Util.Ensure.ensureExists(layerListInvisibleElement, 'LayersValueProvider: The configured DOMElement with the id' + layerListInvisible + ' could not been found.');
      }

      return new Legato.Control.LayersValueProvider({
        elements: {
          layerList: layerlistElement,
          layerListVisible: layerListVisibleElement,
          layerListInvisible: layerListInvisibleElement
        },
        setValue: function(layers){
            if(!Legato.Lang.ObjectUtils.exists(layers)){
              return;
            }

            this.elements.layerList.value = Legato.Lang.CollectionUtils.transform(layers, function(layer){
              return layer.name;
            }).toString();

            this.elements.layerListVisible.value = Legato.Lang.CollectionUtils.transform(Legato.OpenLayers.Map.Util.getVisibleLayers(layers), function(layer){
              return layer.name;
            }).toString();

            this.elements.layerListInvisible.value = Legato.Lang.CollectionUtils.transform(Legato.OpenLayers.Map.Util.getInvisibleLayers(layers), function(layer){
              return layer.name;
            }).toString();
          }
        });
    },
    constructorArguments:
    {
      /*
      * Property layerList
       * Defines the {DOMElement} id where all <OpenLayers.Layer> names should be written into.
       * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      layerList: Legato.Lang.String,

     /*
      * Property: layerListVisible
       * Defines the {DOMElement} id where all visible <OpenLayers.Layer> names should be written into.
       * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      layerListVisible: Legato.Lang.String,

     /*
      * Property: layerListInvisible
       * Defines the {DOMElement} id where all invisible <OpenLayers.Layer> names should be written into.
       * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      layerListInvisible: Legato.Lang.String
    },
    constructorOrder: ['layerList', 'layerListVisible', 'layerListInvisible']
  }
);