/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * Class: Legato.Beans.XMLBeanFactory
 */
Legato.Beans.XMLBeanFactory = Legato.Lang
    .Class(
        Legato.Beans.BeanFactory,
        {
          options :null,
          /*
           * Constructor: initialize
           */
          initialize : function() {
            Legato.Beans.BeanFactory.prototype.initialize
                .apply(this, arguments);
            this.options = {};
          },
          loadURL : function(options) {
            Legato.Util.Ensure.ensureObject(options,
                'Options argument must be an object.');
            Legato.Util.Ensure.ensureString(options.url,
                'Options must contain [url] string element.');
            Legato.Util.Ensure.ensureFunction(options.onSuccess,
                'Options must contain [onSuccess] function element.');
            if (Legato.Lang.ObjectUtils.exists(options.onFailure)) {
              Legato.Util.Ensure.ensureFunction(options.onFailure,
                  'Options element [onFailure] must be a function.');
            }
            // if(typeof options.onSuccess != 'function'){
            // throw new Error('An onSuccess methods needs to be defined');
            // }
            Legato.Util.extend(this.options, options);
            
            var onSuccess = function(request) {
              try{
                var responseText = request.responseText;
                var format = new OpenLayers.Format.XML();
                var doc = format.read(responseText);
                var root = doc.documentElement;
                var beanDefinitions = this.parse(root);
                var that = this;
                var os = function()
                {
                  that.options.onSuccess(request);
                };
                this.loadBeanDefinitions(beanDefinitions, os);
              } catch(e){
                Legato.Util.Logger.getLogger().fatal('Could not parse the resource [' + this.options.url + '].');
                Legato.Util.Logger.getLogger().fatal(e);
                throw e;
              }
            };
            var onFailure = function(request) {
              if (typeof this.options.onFailure == 'function') {
                this.options.onFailure(request);
                return;
              }
              Legato.Util.Logger.getLogger().fatal('Could not load the resource [' + this.options.url + '].');
              throw new Legato.Lang.Exception(
                  'Could not load the resource [' + this.options.url + '].');
            };
            var url;
            var params;
            var questionMarkIndex = this.options.url.indexOf('?');
            if (questionMarkIndex > 0) {
              url = this.options.url.substring(0, questionMarkIndex);
              params = OpenLayers.Util.getParameters(this.options.url);
            } else {
              url = this.options.url;
              params = null;
            }

            var that = this;
            OpenLayers.Request.GET({
              url: url,
              params: params,
              scope: that,
              success: onSuccess,
              failure: onFailure
            });
          },
          retrieveBeansFromElement : function(element) {
            var beanDefinitions = this.parse(element);
            return this.retrieveBeans(beanDefinitions);
          },
          retrieveBeanFromElement : function(element, type) {
            var beanDefinition = this.parseBeanDefinition(element, type);
            return this.retrieveBean(beanDefinition);
          },
          parse : function(root) {
            var children = root.childNodes;
            var beans = {};
            for ( var index = 0; index < children.length; index++) {
              var childNode = children[index];
              if (childNode.nodeType == 1) {
                var element = childNode;
                var namespaceURI = element.namespaceURI;
                var localPart = element.localName
                    || element.nodeName.split(":").pop();
                var draftName = Legato.XML.ElementUtils.getAttributeNS(element,
                    Legato.Beans.NAMESPACE_URI, 'name');
                var beanName = (draftName === null || draftName == '' ? localPart
                    : draftName);
                var beanDefinition = this.parseElement(element);
                beans[beanName] = beanDefinition;
              }
            }
            return beans;
          },
          parseElement : function(element) {
            if (Legato.Lang.StringUtils.isEmpty(element.namespaceURI)) {
              return this.parseSingleComponent(element, Legato.Lang.String);
            } else {
              var type = this.getType(element);
              return this.parseBeanDefinition(element, type);
            }
          },
          parseBeanDefinition : function(element, type) {
            var definition = {};
            for ( var name in type.components) {
              if (type.components.hasOwnProperty(name)) {
                var attributeValue = element.getAttribute(name);
                if (!(attributeValue === null || attributeValue == '')) {
                  definition[name] = attributeValue;
                }
              }
            }
            var children = element.childNodes;
            for ( var index = 0; index < children.length; index++) {
              var childNode = children[index];
              if (childNode.nodeType == 1) {
                var localPart = childNode.localName
                    || childNode.nodeName.split(":").pop();
                if (type.components[localPart]) {
                  var componentName = localPart;
                  var componentType = type.components[componentName];
                  definition[componentName] = this.parseComponent(childNode,
                      componentType);
                }
              }
            }
            var beanDefinition = type.define(definition);
            return beanDefinition;
          },
          parseComponent : function(element, componentType) {
            if (componentType.isList) {
              return this.parseListComponent(element, componentType);
            } else if (componentType.isMap) {
              return this.parseMapComponent(element, componentType);
            } else {
              return this.parseSingleComponent(element, componentType);
            }
          },
          parseListComponent : function(element, componentType) {
            var list = [];
            var children = element.childNodes;
            for ( var index = 0; index < children.length; index++) {
              var childNode = children[index];
              if (childNode.nodeType == 1) {
                var beanDefinition = this.parseElement(childNode);
                list.push(beanDefinition);
              }
            }
            return list;
          },
          parseMapComponent : function(element, componentType) {
            var map = {};
            var attributes = element.attributes;
            for ( var attributeIndex = 0; attributeIndex < attributes.length; attributeIndex++) {
              var attribute = attributes[attributeIndex];
              if (attribute.namespaceURI === null
                  || attribute.namespaceURI == '') {
                var attributeName = attribute.nodeName;
                var attributeValue = attribute.nodeValue;
                map[attributeName] = attributeValue;
              }
            }
            var children = element.childNodes;
            for ( var elementIndex = 0; elementIndex < children.length; elementIndex++) {
              var childNode = children[elementIndex];
              if (childNode.nodeType == 1
                  && (childNode.namespaceURI === null || childNode.namespaceURI == '')) {
                var localPart = childNode.nodeName;
                var draftName = Legato.XML.ElementUtils.getAttributeNS(
                    childNode, Legato.Beans.NAMESPACE_URI, 'name');
                var beanName = (draftName === null || draftName == '' ? localPart
                    : draftName);
                var beanDefinition = this.parseSingleComponent(childNode,
                    componentType.elementType);
                map[beanName] = beanDefinition;
              }
            }
            return map;
          },
          parseSingleComponent : function(element, componentType) {
            var children = element.childNodes;
            for ( var index = 0; index < children.length; index++) {
              var childNode = children[index];
              if (childNode.nodeType == 1) {
                return this.parseElement(childNode);
              } else if (childNode.nodeType == 3
                  && (OpenLayers.String.trim(childNode.nodeValue) != '')) {
                return childNode.nodeValue;
              }
            }
            return null;
          },
          getType : function(element) {
            var namespaceURI = element.namespaceURI;
            var localPart = element.localName
                || element.nodeName.split(":").pop();
            var type = Legato.Beans.ComplexType.getTypeNS(namespaceURI,
                localPart);
            Legato.Util.Ensure.ensureExists(type,
                'Could not load the type for the element [{' + namespaceURI
                    + '}:' + localPart + '].');
            return type;
          },
          CLASS_NAME :'Legato.Beans.XMLBeanFactory'
        });