/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @requires prototype.js
 */

 /*
  * The Loading Indicator Class creates a loading indicator
  * over given element which means that this current
  * area reloads something.
  */
LoadingIndicator = Class.create({

  /*
  * Standard Constructor
  */
  initialize: function() {
    this.loadingCssClass  = "loadingIndicator";
    this.loadingIcon      = "../../../js/theme/default/img/Loading.gif";
    this.loadingIconAlt   = "Loading";
  },

  postInit:function(){
  },

  /*
   * Tries to get the element by searching for the id. If found
   * a loading indicator div will be inserted at the position of
   * the given element. The given element will become a child
   * of the loadingindicator element. Still visible but overlaid
   * by loadingIndicator.
   */
  show: function(options) {
    if(typeof options.element === 'undefined' ||
       options.element === null){
      return false;
    }

    var indicatorDiv = this.getIndicatorDiv(options);

    if(indicatorDiv !== false){
      indicatorDiv.show();
    }
  },


  /*
   * Tries to determine the loadingindicator element for given element.
   * The found the loadingindicator element will been removed.
   */
  hide: function(element){
    if(element === null || typeof element !== 'object'){
      return false;
    }
    
    element = $(element);

    //Delete div
    if(element.down().hasClassName('loadingIndicatorContainer')){
      element.down().remove();
    }
  },

  getIndicatorDiv: function(options){
    if(typeof options.element == 'undefined' ||
       options.element === null){
      return false;
    }

    options.element = $(options.element);
    
    indicatorContainer = new Element('div', {
      id : this.loadingCssClass + '_' + this.getUniqueId()
    });

    indicatorDiv = new Element('div', {
      id : this.loadingCssClass + '_' + this.getUniqueId()
    });

    indicatorImage = new Element('img', {
      src: this.loadingIcon,
      alt: this.loadingIconAlt
    });

    /*
     * Please beware:
     * Creating an element and adding the class attribute in the constructor
     * does not work in IE. Therefore we're using addClassName()
     */
    indicatorContainer.addClassName(this.loadingCssClass + 'Container');
    indicatorDiv.addClassName(this.loadingCssClass);
    indicatorImage.addClassName(this.loadingCssClass);

    //Get current width and height of given element and calculate its current position
    var divH = options.element.getHeight();
    var divW = options.element.getWidth();
    var divOffsetTop = options.element.offsetTop;
    var divOffsetLeft = options.element.offsetLeft;

    indicatorContainer.appendChild(indicatorDiv);
    indicatorContainer.appendChild(indicatorImage);

    Element.insert(options.element, {'top': indicatorContainer});
    indicatorDiv.hide();

    //Determine max zIndex
    var zIndex = this.getHighestzIndex({
      frame: options.frame,
      pattern: '#' + options.element.id + ' div'
    });

    //And ensure that the ajaxloader elements will always been visible
    ++zIndex;

    //Centers the loading icon over given element
    var imgTop = (divH / 2) - (indicatorImage.height / 2);
    var imgLeft = (divW / 2) - (indicatorImage.width / 2);

    indicatorContainer.setStyle({
      'zIndex': zIndex
    });

    indicatorImage.setStyle({
      'top': imgTop + 'px',
      'left': imgLeft + 'px'
    });

    //hovers the loading indicator div over given element
    indicatorDiv.setStyle({
      'height': divH + 'px',
      'width': divW + 'px'
    });

    return indicatorDiv;
  },

  /*
   * Generates an "unique" id by taking the current timestamp and multipling it
   * with a random number. Then the first 18 digits from the result will been
   * returned
   */
   getUniqueId: function(url){
     return ((new Date()).getTime() + "" + Math.floor(Math.random() * 1000000)).substr(0, 18);
   },

   /*
    * Determines the highest zIndex starting by given element and
    * searching all its child components. The highest index will
    * then been returned.
    *
    * An example for a valid pattern could be:
    * #foo div: start at div.id = 'chart' and search in all child divs
    */
   getHighestzIndex: function(options){

     if(typeof options.pattern == 'undefined'){
       disyCommons.logger.debug('Insufficent arguments given for ' + this + ' getHighestzIndex()');
       return false;
     }

     var highestZIndex = 0;

     if(typeof options.frame != 'undefined'){
       //Frame based search
       parent.$(options.frame).contentDocument.defaultView.$$(options.pattern).each(function(element){
         var zIndex = parseInt(element.style.zIndex, 10);
         highestZIndex = zIndex > highestZIndex ? zIndex : highestZIndex;
       });
     } else {
       //Non Frame based search => will look in current document
       $$(options.pattern).each(function(element){
         var zIndex = parseInt(element.style.zIndex, 10);
         highestZIndex = zIndex > highestZIndex ? zIndex : highestZIndex;
       });
     }

     return highestZIndex;
   }

});

var loadingIndicator = new LoadingIndicator();