/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Class: WPS.Control.GeometryEditor
 *
 * The Geometry editor gives access to the legato map client. It is possible
 * to create and edit geometrys using JS code.
 *
 * Inherits from: - <WPS.Contro>
 *
 */
WPS.Control.GeometryEditor = OpenLayers.Class( WPS.Control, {

  /*
   * {<OpenLayers.Map>} the current used map
   */
  map: null,


  /*
   * {<DOMElement>} the element where this control will be rendered
   */
  div: null,

  /*
   * {Object} the current wps process description hashmap
   */
  processDescription: null,

  /*
   * {<WPS.Model.Process>} the current wps process model
   */
  currentProcessModel: null,

  /*
   * {<WPS.Component.Process>} the current wps process component
   */
  processComponent: null,

  styles: null,

  /*
   * Function: initialize
   * Basic constructor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control
   * The following keys are supported
   * - map {<OpenLayers.Map>} The current map
   */
  initialize : function(options) {
    Legato.Util.Ensure.ensureExists(options.map, "form must be defined");
    OpenLayers.Control.prototype.initialize.apply(this, arguments);
  },

  /*
   * Function: destroy
   *
   * See Also:
   * <OpenLayer.Control>
   */
  destroy : function() {
    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /*
   * Function: draw
   *
   * See Also:
   * <OpenLayer.Control>
   */
  draw: function () {
    OpenLayers.Control.prototype.draw.apply(this, arguments);
    return this.div;
  },


  /*
   * Function: resets the Geometry Editor and clears the editor form
   */
  reset:function(){
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(this.processComponent) &&
       !Legato.Lang.ObjectUtils.isNullOrUndefined(this.processComponent.container)){
     this.div.removeChild(this.processComponent.container);
     this.processComponent.destroy();
     this.currentProcessModel.destroy();
     this.processComponent = null;
     this.currentProcessModel = null;
     this.processDescription = null;
    }
  },

  /**
   * Function: createGeometry
   *
   * Shows a Form in configured div and enables the user to paint a geometry based on given
   * process settings.
   *
   * Parameters:
   * processDesc - {Object} A Hashmap containing the description for the process
   * The following keys are supported
   * - id {String} The id for this process
   * - title {String} A title describing this process
   * - description {String} A detailed description for this process
   *
   * paramDesc - {Object} A Hashmap containing the description for the process parameter
   * The following keys are supported
   * - id {String} The id for this process
   * - title {String} A title describing this process
   * - description {String} A detailed description for this process
   * - type {String} The geometry type which will been painted by the user (e.g. Point, Polygon, etc)
   *
   * onExecute - {Function} A callback method which will been executed after the user has finished
   *   the procress. If this callback returns true the editor resets itself automatically.
   * onCancel - {Function} A callback method which will been executed if the user has canceled
   *   this process. If this callback returns true the editor resets itself automatically.
   */
  createGeometry: function(processDesc, paramDesc, onExecute, onCancel){
    this.reset();
    this.processDescription = this.createCreateGeometryProcressDesc(processDesc, paramDesc);
    this.execute(this.processDescription, onExecute, onCancel);
  },

  /**
   * Function: createGeometry
   *
   * Shows a Form in configured div and enables the user to edit the given geometry.
   *
   * Parameters:
   * processDesc - {Object} A Hashmap containing the description for the process
   * The following keys are supported
   * - id {String} The id for this process
   * - title {String} A title describing this process
   * - description {String} A detailed description for this process
   *
   * paramDesc - {Object} A Hashmap containing the description for the process parameter
   * The following keys are supported
   * - id {String} The id for this process
   * - title {String} A title describing this process
   * - description {String} A detailed description for this process
   * - geometry {JSON} A GEO JSON Object representing the geometry for editing
   *
   * onExecute - {Function} A callback method which will been executed after the user has finished
   *   the procress. If this callback returns true the editor resets itself automatically.
   * onCancel - {Function} A callback method which will been executed if the user has canceled
   *   this process. If this callback returns true the editor resets itself automatically.
   */
  editGeometry: function(processDesc, paramDesc, onExecute, onCancel){
    this.reset();
    this.processDescription = this.createEditGeometryProcressDesc(processDesc, paramDesc);
    this.execute(this.processDescription, onExecute, onCancel);

    //Zoom to given geometry
    var parser = new OpenLayers.Format.GeoJSON();
    var geometry = parser.parseGeometry(paramDesc.geometry);
    this.map.zoomToExtent(geometry.getBounds());
  },

  /*
   * Creates a WPS Process Description, creates a new processComponent and displays it.
   */
  execute: function(processDescription, onExecute, onCancel){
    this.currentProcessModel = new WPS.Model.Process(this.map, processDescription);
    this.processComponent = new WPS.Component.Process(
      this.currentProcessModel,
      this.map,
      OpenLayers.Function.bind(function(process,executeRequest){
        if(onExecute(this.getProcessModelAsGeoJSON(process.processModel.models)) === true){
          this.reset();
        }
      }, this),
      OpenLayers.Function.bind(function(process){
        if(onCancel() === true){
          this.reset();
        }
      }, this),
      {
        componentFactoryOptions: {
          styles: this.styles
        }
      }
    );

    this.processComponent.events.on({
      'geoInputActivation' : this.geoInputActivated,
      scope : this
    });

    this.geoInputActivated();
    this.div.appendChild(this.processComponent.container);
  },

  geoInputActivated : function() {
    this.events.triggerEvent('activate', {});
  },

  /*
   * Creates a basic Procress Description which has all what a createProcressDesc and
   * a EditProcressDescription have in common
   */
  createBasicProcessDesc: function(processDesc, paramDesc){
    var processDescription = {
        identifier: {
          value: processDesc.id
        },
        title: {
          value: processDesc.title
        },
        'abstract': {
          value: processDesc.description
        },
        dataInputs: {
          input: [ {
            identifier: {
              value: paramDesc.id
            },
            title: {
              value: paramDesc.title
            },
            'abstract': {
              value: paramDesc.description
            },
            minOccurs: 1,
            maxOccurs: 1,
            complexData: {
              'default': {
                format: {
                  schema: 'http://schemas.opengis.net/gml/3.1.1/base/gml.xsd#'
                }
              }
            }
        }]
      }
    };

    return processDescription;
  },

  /*
   * Create a ProcressDescription for creating a new geoemtry.
   */
  createCreateGeometryProcressDesc: function(processDesc, paramDesc){
    var processDescription = this.createBasicProcessDesc(processDesc, paramDesc);
    processDescription.dataInputs.input[0].complexData['default'].format.schema += paramDesc.type;
    return processDescription;
  },

  /*
   * Create a ProcressDescription for editing a given GeoJSON geoemtry.
   */
  createEditGeometryProcressDesc: function(processDesc, paramDesc){
    var processDescription = this.createBasicProcessDesc(processDesc, paramDesc);
    processDescription.dataInputs.input[0].complexData['default'].format.schema += paramDesc.geometry.type;
    processDescription.dataInputs.input[0].complexData.defaultValue = {
      content: [{
        value: paramDesc.geometry
      }]
    };
    return processDescription;
  },

  /*
   * Converts the passed process models into geoJSON based format and returns it
   *
   * Parameters:
   * models - {Array} An array of processmodels
   *
   * See Also:
   * <WPS.Model.Process>
   */
  getProcessModelAsGeoJSON: function(models){
    var geoJSON = new OpenLayers.Format.GeoJSON();
    if (models.length === 1) {
      var modelValue = models[0].value;
      return geoJSON.extract.geometry.apply(geoJSON, [modelValue]);
    }
    var components = [];
    for (i = 0; i < models.length; i++) {
      components[i] = models[i].value;
    }

    return geoJSON.extract.geometry.apply(geoJSON, [new OpenLayers.Geometry.Collection(components)]);
  },

  activate : function() {
    if (!Legato.Lang.ObjectUtils.exists(this.processComponent)) {
      return;
    }
    this.processComponent.activate();
  },

  deactivate : function() {
    if (!Legato.Lang.ObjectUtils.exists(this.processComponent)) {
      return;
    }
    this.processComponent.deactivate();
  },

  CLASS_NAME : "WPS.Control.GeometryEditor"
});

/**
 * Structure: lwpsc:GeometryEditor
 * XML based config for a <WPS.Control.GeometryEditor>.
 *
 * See Also:
 * - <Legato.Control>
 * - <Legato.Beans.BeanFactory>
 * - <QName>
 *
 * A valid config example for a GeometryEditor would be:
 * (start code)
 * <lwpsc:GeometryEditor id="geometryEditor" div="geometryEditor">
 *   <map>
 *     <lb:Reference target="map" />
 *   </map>
 * </lwpsc:GeometryEditor>
 * (end)
 */
WPS.Control.GeometryEditor.Bean = WPS.Control.Bean.Extend('WPS.Control.GeometryEditor',
  WPS.Control.QName('GeometryEditor'),
  {
    _constructor : WPS.Control.GeometryEditor,
    options : {
      map : OpenLayers.Map,
      styles : Legato.Beans.Object.List()
    }
  }
);