/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

Legato.Control.NavigationHistoryNext = OpenLayers.Class(
    Legato.Control.AbstractNavigationHistoryButton, {
      /**
       * Property: title  
       * {string}  This property is used for showing a tooltip over the  
       * NavigationHistoryNext control. Default value is "Next".  
       */ 
      title : 'Next',
      
      /**
       * Constructor: OpenLayers.Control.NavigationHistoryNext
       * 
       * Parameters: 
       * history - {Legato.Control.NavigationHistory}
       * options - {Object}
       */
      initialize : function(history, options) {
        Legato.Control.AbstractNavigationHistoryButton.prototype.initialize
            .apply(this, [ history, options ]);
        this.history.events.register("nextChange", this, this.onNextChange);
      },

      /**
       * Method: onNextChange
       * Called when the next history stack changes.
       *
       * Parameters:
       * state - {Object} An object representing the state to be restored
       *     if next is triggered again or null if no next states remain.
       * length - {Integer} The number of remaining next states that can
       *     be restored.
       */
      onNextChange : function(event) {
        var state = event.state;
        if (state && !this.active) {
          this.activate();
        } else if (!state && this.active) {
          this.deactivate();
        }
      },

      /**
       * APIMethod: trigger 
       * Restore the next state. If no items are in the
       * next history stack, this has no effect. The next history stack is
       * populated as states are restored from the previous history stack.
       * 
       * Returns: {Object} Item representing state that was restored. Undefined
       * if no items are in the next history stack.
       */
      trigger : function() {
        var state = this.history.nextStack.shift();
        if (!Legato.Lang.ObjectUtils.isUndefined(state)) {
          this.history.previousStack.unshift(state);
          this.history.restoring = true;
          this.history.restore(state);
          this.history.restoring = false;
          this.history.onPreviousChange(this.history.previousStack[1],
              this.history.previousStack.length - 1);
          this.history.onNextChange(this.history.nextStack[0],
              this.history.nextStack.length);
        }
        return state;
      },

      CLASS_NAME : "Legato.Control.NavigationHistoryNext"

    });

/**
 * Structure: lc:NavigationHistoryNext XML based config for a <Legato.Control.NavigationHistoryNext>.
 * 
 * See Also: - <Legato.Control> - <Legato.Beans.BeanFactory> - <QName>
 * 
 * A valid config example for a NavigationHistoryNext would be:
 * (start code)
 * <lc:NavigationHistoryNext title="Tooltip for next">
 *   <history>
 *     <lb:Reference target="NavigationHistory" />
 *   </history>
 * </lc:NavigationHistoryNext>
 * (end)
 */
Legato.Control.NavigationHistoryNext.Bean = Legato.Control.Bean.Extend(
    'Legato.Control.NavigationHistoryNext', Legato.Control
        .QName('NavigationHistoryNext'), {
      factoryFunction : function(history, options) {
        return new Legato.Control.NavigationHistoryNext(history, options);
      },
      constructorArguments : {
        history : Legato.Control.NavigationHistory.Bean
      },
      constructorOrder : [ 'history' ]
    });