/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

WPS.Component = OpenLayers.Class({
  EVENT_TYPES : [ 'activate', 'deactivate', 'validitychange' ],
  id : null,
  model : null,
  map : null,
  container : null,
  titleElement : null,
  checkedBox : null,
  events : null,
  status : null,
  validitySymbol : null,

  initialize : function(id, model, map, options) {
    Legato.Util.Ensure.ensureExists(id, 'Id must be defined.');
    Legato.Util.Ensure.ensureExists(model, 'Model must be defined.');
    Legato.Util.Ensure.ensureExists(map, 'Map must be defined.');
    OpenLayers.Util.extend(this, options);
    this.id = id;
    this.model = model;
    this.map = map;
    this.events = new OpenLayers.Events(this, null, this.EVENT_TYPES);

    this.container = document.createElement('div');
    this.container.id = this.id + '_Container';
    if (this.model.required) {
      OpenLayers.Element.addClass(this.container, 'required');
    }
    this.container.onmouseover = OpenLayers.Function.bindAsEventListener(
        this.hoverIn, this);
    this.container.onmouseout = OpenLayers.Function.bindAsEventListener(
        this.hoverOut, this);
    this.container.component = this;
    var checkedElement = document.createElement('div');
    OpenLayers.Element.addClass(checkedElement, 'isInputChecked');

    this.container.appendChild(checkedElement);

    this.checkedBox = document.createElement('input');
    this.checkedBox.type = 'checkbox';
    this.checkedBox.name = this.id + '_Checked';
    this.checkedBox.checked = this.model.checked;
    this.checkedBox.onclick = OpenLayers.Function.bindAsEventListener(
        this.checkedChanged, this);
    checkedElement.appendChild(this.checkedBox);

    this.validitySymbol = document.createElement('div');
    OpenLayers.Element.addClass(this.validitySymbol, 'validitySymbol');
    this.container.appendChild(this.validitySymbol);

    this.titleElement = document.createElement('div');
    OpenLayers.Element.addClass(this.titleElement, 'title');
    this.titleElement.innerHTML = model.title;
    this.container.appendChild(this.titleElement);

    var inputElement = this.initializeInputs();
    /*jslint sub:true*/
    inputElement.title = model['abstract'];

    if (Legato.Lang.ObjectUtils.exists(this.model.getValue())) {
      this.onModelValueChange({
        value : this.model.getValue()
      });
    }

    this.model.events.on({
      'valuechange' : this.onModelValueChange,
      'validitychange' : this.onModelValidityChange,
      scope : this
    });

    if (this.model.getValue() === null) {
      this.onDeactivation();
    } else {
      this.onActivation();
    }

    this.initializeMetaData();
    this.updateValiditySymbol();
  },
  destroy : function() {

    this.container = null;

    if (Legato.Lang.ObjectUtils.exists(this.events)) {
      this.events.destroy();
      this.events = null;
    }

    if (Legato.Lang.ObjectUtils.exists(this.checkedBox)) {
      this.checkedBox.onclick = null;
      this.checkedBox = null;
    }

    if (Legato.Lang.ObjectUtils.exists(this.container)) {
      this.container.component = null;
      this.container.onmouseover = null;
      this.container.onmouseout = null;
      this.container = null;
    }
    this.model.events.un({
      'valuechange' : this.onModelValueChange,
      'validitychange' : this.onModelValidityChange,
      scope : this
    });
  },

  isActive : function() {
    return (WPS.Component.STATUS_ACTIVE_HOVER === this.status || WPS.Component.STATUS_ACTIVE === this.status);
  },

  isInactive : function() {
    return (WPS.Component.STATUS_INACTIVE_HOVER === this.status || WPS.Component.STATUS_INACTIVE === this.status);
  },

  hoverIn : function() {
    this.onHoverIn();
  },

  hoverOut : function() {
    this.onHoverOut();
  },

  initializeInputs : function() {
    throw new Legato.Lang.Exception(this.CLASS_NAME +
        ' must implement the initializeInputs function from from abstract class ' +
        WPS.Component.CLASS_NAME);
  },

  initializeMetaData : function() {
    if (!Legato.Lang.ObjectUtils.isNotEmptyArray(this.model.metadata)) {
      return;
    }
    for ( var i = 0; i < this.model.metadata.length; i++) {
      this.handleMetaData(this.model.metadata[i]);
    }
  },

  handleMetaData : function(metadata) {
    var about = Legato.Lang.ObjectUtils.exists(metadata.about) ? metadata.about : null;
    var href = Legato.Lang.ObjectUtils.exists(metadata.href) ? metadata.href : null;
    var title = Legato.Lang.ObjectUtils.exists(metadata.title) ? metadata.title : null;

    if (about === 'visibility' && href === 'hidden') {
      OpenLayers.Element.hide(this.container);
    }
  },

  isAccessable : function(){
    return OpenLayers.Element.visible(this.container);
  },

  checkedChanged : function(event) {
    var checked = (event.target || event.srcElement).checked;
    this.model.setChecked(checked);
  },


  onModelValueChange : function(event) {
    throw new Legato.Lang.Exception(this.CLASS_NAME +
        ' must implement the onModelValueChange function from from abstract class ' +
        WPS.Component.CLASS_NAME);
  },

  updateValiditySymbol : function() {
    this.validitySymbol.title = '';
    if (this.model.valid) {
      OpenLayers.Element.addClass(this.container, 'valid');
      OpenLayers.Element.removeClass(this.container, 'invalid');
    } else {
      OpenLayers.Element.addClass(this.container, 'invalid');
      OpenLayers.Element.removeClass(this.container, 'valid');
      this.validitySymbol.title += this.model.validationMessages.join('; ');
    }
  },

  onModelValidityChange : function(event) {
    this.updateValiditySymbol();
    this.events.triggerEvent('validitychange', {
      component : this
    });
  },

  activate : function() {
    this.onActivation();
  },

  deactivate : function() {
    this.onDeactivation();
  },

  onHoverIn : function() {
    if (WPS.Component.STATUS_INACTIVE === this.status) {
      this.status = WPS.Component.STATUS_INACTIVE_HOVER;
      this.replaceStatusClass('inputContainerInactiveHover');
    } else {
      this.status = WPS.Component.STATUS_ACTIVE_HOVER;
      this.replaceStatusClass('inputContainerActiveHover');
    }
  },

  onHoverOut : function(triggerEvent) {
    if (WPS.Component.STATUS_ACTIVE_HOVER === this.status) {
      this.status = WPS.Component.STATUS_ACTIVE;
      this.replaceStatusClass('inputContainerActive');
    } else {
      this.status = WPS.Component.STATUS_INACTIVE;
      this.replaceStatusClass('inputContainerInactive');
    }
  },

  onActivation : function(triggerEvent) {
    this.status = WPS.Component.STATUS_ACTIVE_HOVER;
    this.replaceStatusClass('inputContainerActiveHover');
    if (triggerEvent) {
      this.events.triggerEvent('activate', {
        component : this
      });
    }
    this.model.setChecked(true);
    this.checkedBox.checked = true;
  },

  onDeactivation : function(triggerEvent) {
    this.status = WPS.Component.STATUS_INACTIVE;
    this.replaceStatusClass('inputContainerInactive');
    if (triggerEvent) {
      this.events.triggerEvent('deactivate', {
        component : this
      });
    }
  },

  replaceStatusClass : function(currentClass) {
    if (this.container) {
      OpenLayers.Element.removeClass(this.container,
          'inputContainerInactive');
      OpenLayers.Element.removeClass(this.container,
          'inputContainerInactiveHover');
      OpenLayers.Element.removeClass(this.container,
          'inputContainerActive');
      OpenLayers.Element.removeClass(this.container,
          'inputContainerActiveHover');
      OpenLayers.Element.addClass(this.container,
          currentClass);
    }
  },

  CLASS_NAME : 'WPS.Component'
});

WPS.Component.STATUS_ACTIVE_HOVER = 1;
WPS.Component.STATUS_ACTIVE = 2;
WPS.Component.STATUS_INACTIVE_HOVER = 3;
WPS.Component.STATUS_INACTIVE = 4;

