/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @requires OpenLayers/Handler.js
 * @requires OpenLayers/Handler/MouseWheel.js
 */

/**
 * Class: Legato.Handler.MouseWheel
 * Handler for wheel up/down events
 *
 * Extension to OpenLayers:
 * Functionality to detect mousewheel-interaction on the map-div itself
 *
 * Inherits from:
 *  - <OpenLayers.Handler.MouseWheel>
 */
Legato.Handler.MouseWheel = OpenLayers.Class(OpenLayers.Handler.MouseWheel, {
    /**
     * Property: interval
     * {Integer} In order to increase server performance, an interval (in
     *     milliseconds) can be set to reduce the number of up/down events
     *     called. If set, a new up/down event will not be set until the
     *     interval has passed.
     *     Defaults to 0, meaning no interval.
     */
    interval: 100,

    /**
     * Constructor: Legato.Handler.MouseWheel
     *
     * Parameters:
     * control - {<OpenLayers.Control>}
     * callbacks - {Object} An object containing a single function to be
     *                          called when the drag operation is finished.
     *                          The callback should expect to recieve a single
     *                          argument, the point geometry.
     * options - {Object}
     */
    initialize: function(control, callbacks, options) {
        OpenLayers.Handler.prototype.initialize.apply(this, arguments);
        this.wheelListener = OpenLayers.Function.bindAsEventListener(
            this.onWheelEvent, this
        );
    },

    /**
     * Method: onWheelEvent
     * Catch the wheel event and handle it xbrowserly
     *
     * Parameters:
     * e - {Event}
     */
    onWheelEvent: function(e){
        // make sure we have a map and check keyboard modifiers
        if (!this.map || !this.checkModifiers(e)) {
            return;
        }

        // Ride up the element's DOM hierarchy to determine if it or any of
        //  its ancestors was:
        //   * specifically marked as scrollable
        //   * one of our layer divs
        //   * the map div
        //
        var overScrollableDiv = false;
        var overLayerDiv = false;
        var overMapDiv = false;

        var elem = OpenLayers.Event.element(e);
        while((elem !== null) && !overMapDiv && !overScrollableDiv) {

            if (!overScrollableDiv) {
                try {
                  var overflow=null;
                    if (elem.currentStyle) {
                        overflow = elem.currentStyle.overflow;
                    } else {
                        var style =
                            document.defaultView.getComputedStyle(elem, null);
                        overflow = style.getPropertyValue("overflow");
                    }
                    overScrollableDiv = ( overflow &&
                        ((overflow === "auto") || (overflow === "scroll") ));
                } catch(err) {
                    //sometimes when scrolling in a popup, this causes
                    // obscure browser error
                }
            }

            if (!overLayerDiv) {
                for(var i=0, len=this.map.layers.length; i<len; i++) {
                    // Are we in the layer div? Note that we have two cases
                    // here: one is to catch EventPane layers, which have a
                    // pane above the layer (layer.pane)
                    if (elem === this.map.layers[i].div
                        || elem === this.map.layers[i].pane
                        || elem === this.map.div) {
                        overLayerDiv = true;
                        break;
                    }
                }
            }
            overMapDiv = (elem === this.map.div);

            elem = elem.parentNode;
        }

        // Logic below is the following:
        //
        // If we are over a scrollable div or not over the map div:
        //  * do nothing (let the browser handle scrolling)
        //
        //    otherwise
        //
        //    If we are over the layer div:
        //     * zoom/in out
        //     then
        //     * kill event (so as not to also scroll the page after zooming)
        //
        //       otherwise
        //
        //       Kill the event (dont scroll the page if we wheel over the
        //        layerswitcher or the pan/zoom control)
        //
        if (!overScrollableDiv && overMapDiv) {
            if (overLayerDiv) {
                var delta = 0;
                if (!e) {
                    e = window.event;
                }
                if (e.wheelDelta) {
                    delta = e.wheelDelta/120;
                    if (window.opera && window.opera.version() < 9.2) {
                        delta = -delta;
                    }
                } else if (e.detail) {
                    delta = -e.detail / 3;
                }
                this.delta = this.delta + delta;

                if(this.interval) {
                    window.clearTimeout(this._timeoutId);
                    this._timeoutId = window.setTimeout(
                        OpenLayers.Function.bind(function(){
                            this.wheelZoom(e);
                        }, this),
                        this.interval
                    );
                } else {
                    this.wheelZoom(e);
                }
            }
            OpenLayers.Event.stop(e);
        }
    },

    CLASS_NAME: "Legato.Handler.MouseWheel"
});
