/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: nedkov $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: ControlContainer
 *
 * Inherits from: - <Legato.Control>
 */
Legato.Control.ControlContainer = OpenLayers.Class(Legato.Control,
{
  /**
   * Property: type
   * {OpenLayers.Control.TYPE}
   */
  type: OpenLayers.Control.TYPE_BUTTON,

  /**
   * Property: controls
   * {Array(<OpenLayers.Control>)}
   */
  controls: null,

  panel_div : null,

  panel_div_ul : null,

  /**
   * Property: parentPanel
   * {<Legato.Control.ExtendedPanel>)}
   */
  parentPanel : null,

  drawMode : null,

  menuDrawOptions : null,

  showActiveControl : false,

  activeControl : null,

  childControlActivateDeactivate : {
    activate : function(){},
    deactivate : function(){}
  },

  initialize : function(options) {
    OpenLayers.Control.prototype.initialize.apply(this, [options]);
    this.active = false;
    if(!Legato.Lang.ObjectUtils.exists(this.drawMode)){
      this.drawMode = Legato.Control.ControlContainer.DRAW_MODE_TOOLBAR;
    }
    if(Legato.Lang.StringUtils.isEmpty(options.displayClass) &&
       Legato.Lang.ObjectUtils.exists(this.controls) &&
       this.controls.length > 0){
      this.activeControl = this.controls[0];
      this.displayClass = this.activeControl.displayClass;
    }
    this.menuDrawOptions = OpenLayers.Util.extend({
      headerVisible: true,
      headerPropertyName : 'title',
      iconVisible : true,
      iconPropertyName : 'icon',
      titleVisible : true,
      titlePropertyName : 'title',
      descriptionVisible : true,
      descriptionPropertyName : 'description'
    }, this.menuDrawOptions);

    if(this.showActiveControl && this.drawMode === Legato.Control.ControlContainer.DRAW_MODE_TOOLBAR){
      var controlContainer = this;
      this.childControlActivateDeactivate =  {
          activate : function(){
            var childDisplayClass = this.displayClass;
            var childPanelDiv = this.panel_div;

            var containerPanelDiv = controlContainer.panel_div;
            var containerDisplayClass = controlContainer.displayClass;

            containerPanelDiv.className = containerPanelDiv.className.replace(containerDisplayClass, childDisplayClass);
            controlContainer.displayClass = childDisplayClass;

            controlContainer.activate();
            controlContainer.activeControl = this;
          },
          deactivate : function(){
            controlContainer.deactivate();
          }
      };
    }

    this.createPanelDiv();
    this.appendChildControls(this.controls);
  },

  addControls : function(controls){
    this.controls = this.controls.concat(controls);
    this.appendChildControls(controls);

    if(Legato.Lang.ObjectUtils.exists(this.parentPanel)){
      if (this.parentPanel.map) { // map.addControl() has already been called on the parentPanel
        this.parentPanel.controls = this.parentPanel.controls.concat(controls);
        this.parentPanel.addControlsToMap(controls);
      }
    }
  },

  getPanelDiv : function(){
    return this.panel_div;
  },

  setParentPanel: function(extendedPanel){
    this.parentPanel = extendedPanel;
  },

  createPanelDiv: function(){
    this.panel_div = this.createControlPanelDiv(this);
    OpenLayers.Element.addClass(this.panel_div, 'ControlContainer');

    var container_div = document.createElement('div');
    container_div.style.display = 'none';
    this.panel_div.appendChild(container_div);

    switch(this.drawMode) {
      case Legato.Control.ControlContainer.DRAW_MODE_TOOLBAR:
        OpenLayers.Element.addClass(container_div, 'ToolbarControlContainer');
        break;
      case Legato.Control.ControlContainer.DRAW_MODE_MENU:
        OpenLayers.Element.addClass(container_div, 'MenuControlContainer');
        this.appendHeader(container_div);
        break;
    }

    this.panel_div_ul = document.createElement('ul');
    container_div.appendChild(this.panel_div_ul);

    this.onHoverPanelDiv(this.panel_div);
  },

  createControlPanelDiv: function(control){
    var panel_div = document.createElement("div");

    OpenLayers.Element.addClass(panel_div, control.displayClass
        + "ItemInactive");
    OpenLayers.Element.addClass(panel_div, "ItemInactive");
    OpenLayers.Element.addClass(panel_div, "olButton");

    if (control.title != "" && !panel_div.title) {
      panel_div.title = control.title;
    }

    return panel_div;
  },

  onHoverPanelDiv : function(panel_div){
    panel_div.onmouseover = function(){
      for(var i = 0, length = this.children.length; i < length; i++ ){
        this.children[i].style.display = 'block';
      }
    };

    panel_div.onmouseout = function(){
      for(var i = 0, length = this.children.length; i < length; i++ ){
        this.children[i].style.display = 'none';
      }
    };
  },

  appendHeader: function(container_div){
    if(this.menuDrawOptions.headerVisible){
      var headerValue = this[this.menuDrawOptions.headerPropertyName];
      if(!Legato.Lang.StringUtils.isEmpty(headerValue)){
        var headerTitleDiv = document.createElement('div');
        OpenLayers.Element.addClass(headerTitleDiv, 'MenuHeaderTitle');
        headerTitleDiv.innerHTML = headerValue;
        container_div.appendChild(headerTitleDiv);
      }
    }
  },

  appendChildControls: function(controls){
    var currentControl;
    for ( var j = 0; j < controls.length; j++) {
      currentControl = controls[j];
      currentControl.events.on(this.childControlActivateDeactivate);
      var childControlLi = this.createChildControlLi(currentControl);
      this.panel_div_ul.appendChild(childControlLi);
    }
  },

  createChildControlLi: function(childControl){
    var childControlLi = document.createElement("li");
    var childControlDiv = null;

    switch(this.drawMode) {
      case Legato.Control.ControlContainer.DRAW_MODE_TOOLBAR:
        childControlDiv = this.createControlPanelDiv(childControl);
        OpenLayers.Element.addClass(childControlDiv, 'ControlContainerToolbarItem');
        break;
      case Legato.Control.ControlContainer.DRAW_MODE_MENU:
        childControlDiv = document.createElement('div');
        OpenLayers.Element.addClass(childControlDiv, 'ControlContainerMenuItem');
        OpenLayers.Element.addClass(childControlDiv, childControl.displayClass);
        OpenLayers.Element.addClass(childControlDiv, 'olButton');

        var childControlMenuIcon = document.createElement('span');
        OpenLayers.Element.addClass(childControlMenuIcon, 'MenuIcon');

        if(childControl.hasOwnProperty(this.menuDrawOptions.iconPropertyName) && childControl[this.menuDrawOptions.iconPropertyName] !== null){
          childControlMenuIcon.style.backgroundImage = 'url(' + childControl[this.menuDrawOptions.iconPropertyName] + ')';
          OpenLayers.Element.addClass(childControlMenuIcon, 'CustomMenuIcon');
        }
        childControlDiv.appendChild(childControlMenuIcon);

        var childControlMenuTitle = document.createElement('span');
        if(this.menuDrawOptions.titleVisible && childControl.hasOwnProperty(this.menuDrawOptions.titlePropertyName)){
          OpenLayers.Element.addClass(childControlMenuTitle, 'MenuTitle');
          childControlMenuTitle.innerHTML = childControl[this.menuDrawOptions.titlePropertyName];
          childControlDiv.appendChild(childControlMenuTitle);
        }

        if(this.menuDrawOptions.descriptionVisible && childControl.hasOwnProperty(this.menuDrawOptions.descriptionPropertyName)){
          var childControlMenuSubTitle = document.createElement('span');
          OpenLayers.Element.addClass(childControlMenuSubTitle, 'MenuSubTitle');
          childControlMenuSubTitle.innerHTML = childControl[this.menuDrawOptions.descriptionPropertyName];
          childControlDiv.appendChild(childControlMenuSubTitle);
        }else{
          OpenLayers.Element.addClass(childControlMenuTitle, 'WithoutSubTitle');
        }
        break;
      default:
        throw new Legato.Lang.Exception('Invalid ControlContainer.drawMode = [' + this.drawMode +'].');
    }

    childControl.panel_div = childControlDiv;
    childControlLi.appendChild(childControlDiv);

    var that = this;
    childControlLi.onmouseup = function(){
      that.hideChildren(that.panel_div);
    };

    return childControlLi;
  },

  destroy : function(){
    for(var i = 0, len = this.controls.length; i < len; i++){
      if(Legato.Lang.ObjectUtils.exists(this.controls[i].events)){
        this.controls[i].events.un(this.childControlActivateDeactivate);
      }
    }
    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  trigger : function(){
    if(Legato.Lang.ObjectUtils.exists(this.activeControl)){
      this.parentPanel.activateControl(this.activeControl);
      if (!Legato.Util.Browser.isTouchCapable()){
        this.hideChildren(this.panel_div);
      }
    }
  },

  hideChildren: function(element){
    for(var i = 0, length = element.children.length; i < length; i++ ){
      element.children[i].style.display = 'none';
    }
  },

  CLASS_NAME: 'Legato.Control.ControlContainer'
});

Legato.Control.ControlContainer.DRAW_MODE_TOOLBAR = 'TOOLBAR';
Legato.Control.ControlContainer.DRAW_MODE_MENU = 'MENU';

/**
 * Structure: lc:ControlContainer
 * XML based config for a <Legato.Control.ControlContainer>.
 *
 * See Also:
 * - <Legato.Control>
 * - <Legato.Beans.BeanFactory>
 * - <QName>
 *
 * A valid config example for a ControlContainer would be:
 * (start code)
 * <lc:ControlContainer>
 *  <controls>
 *  </controls>
 * </lc:ControlContainer>
 * (end)
 */
Legato.Control.ControlContainer.Bean = Legato.Control.Bean.Extend(
    'Legato.Control.ControlContainer',
    Legato.Control.QName('ControlContainer'),
    {
      _constructor: Legato.Control.ControlContainer,
      options:
      {
        displayClass: Legato.Lang.String,
        controls: OpenLayers.Control.Bean.List(),
        drawMode : Legato.Lang.String.Enum('Legato.Control.ControlContainer.DRAW_MODE', {
          'TOOLBAR' : Legato.Control.ControlContainer.DRAW_MODE_TOOLBAR,
          'MENU' : Legato.Control.ControlContainer.DRAW_MODE_MENU
        }),
        menuDrawOptions: Legato.Beans.Object.Map(),
        showActiveControl: Legato.Lang.Boolean
      }
    }
  );
