/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * Class: Legato.Control.ControlSwitcher
 * The ControlSwitcher control is able to manage a list of legato controls.
 * The ControlSwitcher always ensures that only one control of given list is active.
 * If a defaultControl has been configured this control will always be active if
 * no other control of given list is active.
 */
Legato.Control.ControlSwitcher = OpenLayers.Class(Legato.Control,
{
  /*
   * Function: initialize
   * Basic ctor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control
   * The following keys are supported
   * - controls {Object} A hashmap containing all Legato.Control instance which will
   *   been controlled by this control
   * - defaultControl {Object} If configured this control becomes automatically active if all other
   *   ones are deactivated.
   */
  initialize : function(controls, defaultControl, options) {
    this.controls = controls;
    this.defaultControl = defaultControl;
    OpenLayers.Control.prototype.initialize.apply(this, [options]);
  },

  /*
   * Function: destroy
   *
   * See Also:
   * <OpenLayer.Control>
   */
  destroy : function() {
    for(var i = 0; i < this.controls.length; i++){
      this.controls[i].events.un({
        'activate': this.onControlActivate,
        'deactivate': this.onControlDeactivate,
        scope: this
      });
    }

    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /*
   * Triggered if one control becomes active. We then determine
   * if another control is active as well and if so we'll deactivate
   * it.
   */
  onControlActivate: function(event){
    for(var i = 0; i < this.controls.length; i++){
      if(this.controls[i].active && this.controls[i].id != event.object.id){
        this.controls[i].deactivate();
      }
    }
  },

  /*
   * Triggered if one control becomes inactive. If a defaultControl has been
   * configured we check if at least one control is active and if not we'll
   * activate the configured default control.
   */
  onControlDeactivate: function(event){
    if(Legato.Lang.ObjectUtils.exists(this.defaultControl)){
      var active = false;

      for(var i = 0; i < this.controls.length; i++){
        if(this.controls[i].active){
          active = true;
          break;
        }
      }

      if(!active){
        this.defaultControl.activate();
      }
    }
  },

  /*
   * Function: setMap
   *
   * See Also:
   * <OpenLayer.Control>
   */
  setMap : function(map) {
    OpenLayers.Control.prototype.setMap.apply(this, arguments);

    for(var i = 0; i < this.controls.length; i++){
      this.controls[i].events.on({
        'activate': this.onControlActivate,
        'deactivate': this.onControlDeactivate,
        scope: this
      });
    }

    if(Legato.Lang.ObjectUtils.exists(this.defaultControl)){
      this.defaultControl.activate();
    }
   },

  /*
   * Function: draw
   *
   * See Also:
   * <OpenLayer.Control>
   */
  draw : function() {
    return this.div;
  },

  CLASS_NAME : 'Legato.Control.ControlSwitcher'
});

Legato.Control.ControlSwitcher.Bean = Legato.Control.Bean.Extend(
  'Legato.Control.ControlSwitcher',

  /*
   * Property: QName
   * The qualified name for this control. Needed by XML config to instantiate it.
   */
  Legato.Control.QName('ControlSwitcher'),
  {
   /*
    * Creates a basic <Legato.Control.ControlSwitcher> instance. The Controls Switcher will
    * allways ensure that just one control is active. If another control becomes active
    * all other ones will been deactivated.
    *
    * Parameters:
    * controls - {Object}
    *
    * Returns:
    * <Legato.Control.ControlSwitcher>
    */
    factoryFunction : function(controls, defaultControl, options)
    {
      Legato.Util.Ensure.ensureExists(controls, 'ControlSwitcher: To use the ControlSwitcher control you must configure at least one legato control.');

      if(Legato.Lang.ObjectUtils.isArray(controls)){
        for(var i = 0; i < controls.length; i++){
          Legato.Util.Ensure.ensureObject(controls[i], 'ControlSwitcher: All configured controls which are passed to the ControlSwitcher Class must be objetcs.');
        }
      }

      if(Legato.Lang.ObjectUtils.exists(defaultControl)){
        Legato.Util.Ensure.ensureObject(defaultControl, 'ControlSwitcher: If a defaultControl has been configured it needs to be a legato control as well.');
      }

      OpenLayers.Console.debug('Creating ControlSwitcher with: ' + controls);

      return new Legato.Control.ControlSwitcher(controls, defaultControl, options);
    },
    constructorArguments:
    {
     /*factoryFunction
      * Property: controls
      */
      controls: OpenLayers.Control.Bean.List(),

      /*
       * If configured this control becomes automatically active if all other
       * ones are deactivated.
       */
      defaultControl: OpenLayers.Control.Bean
    },
    constructorOrder: ['controls', 'defaultControl']
  }
);
