/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: leine $
 * @version $Revision: 69543 $
 * @requires Legato.js
 * @requires OpenLayers/Control.js
 */

/**
 * Class: Legato.OpenLayers.Map.Util
 *
 */
Legato.OpenLayers.Map.Util = {};

/*
 * Function: getOverlays
 * Just returns from given layers overlays
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found overlays
 */
Legato.OpenLayers.Map.Util.getOverlays = function(layers){
  return Legato.Lang.CollectionUtils.filter(
    layers,
    function(layer){
      return (!layer.isBaseLayer);
    }
  );
};

/*
 * Function: getSelectedLayers
 * Just returns from given layers all selected layers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found selected layers
 */
Legato.OpenLayers.Map.Util.getSelectedLayers = function(layers){
  return Legato.Lang.CollectionUtils.filter(
      layers,
      function(layer){
        return (layer.selected);
      }
  );
};

/*
 * Function: getQueryableLayers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} Found queryable layers
 */
Legato.OpenLayers.Map.Util.getQueryableLayers = function(layers){
  return Legato.Lang.CollectionUtils.filter( layers,
      function(layer){
        return (Legato.Lang.ObjectUtils.isBoolean(layer.queryable) && layer.queryable);
      }
  );
};

/*
 * Function: getSnappableLayers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} Found snappable layers
 */
Legato.OpenLayers.Map.Util.getSnappableLayers = function(layers){
  return Legato.Lang.CollectionUtils.filter( layers,
      function(layer){
    return (Legato.Lang.ObjectUtils.isObject(layer.options) &&
      Legato.Lang.ObjectUtils.isObject(layer.options.attributes) &&
      Legato.Lang.ObjectUtils.isString(layer.options.attributes.snappable) &&
      Boolean(layer.options.attributes.snappable));
  }
  );
};


/*
 * Function: getUnselectedLayers
 * Just returns from given layers all unselected layers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found unselected layers
 */
Legato.OpenLayers.Map.Util.getUnselectedLayers = function(layers){
  return Legato.Lang.CollectionUtils.filter(
      layers,
      function(layer){
        return (!layer.selected);
      }
  );
};

/*
 * Function: getLayerById
 * Retrieves a layer by the given layerId from the list of layers.
 *
 * Returns:
 * {<OpenLayers.Layer>} Requested layer or null if it is not found
 */
Legato.OpenLayers.Map.Util.getLayerById = function(layers, layerId) {
  var filteredLayers = Legato.Lang.CollectionUtils.filter(layers, 
      function(layer) {
        return layer.layerId === layerId;
      });
  return (filteredLayers.length > 0) ? filteredLayers[0] : null;
};

/*
 * Function: isVectorLayer
 * Determines if given layer is a vector layer.
 *
 * Returns:
 * {Boolean} True if Vector layer, false otherwise
 */
Legato.OpenLayers.Map.Util.isVectorLayer = function(layer){
  return Legato.Lang.ObjectUtils.exists(layer.CLASS_NAME) && layer.CLASS_NAME === 'OpenLayers.Layer.Vector';
};

Legato.OpenLayers.Map.Util.isIgnoredLayer = function(layer) {
  return Legato.Lang.ObjectUtils.isBoolean(layer.options.legatoIgnore) && layer.options.legatoIgnore;
};

/*
 * Function: isListedLayer
 * Determines if given layer should appear in legend list.
 *
 * Returns:
 * (Boolean) True if the layer should appear in legend list, false otherwise
 */
Legato.OpenLayers.Map.Util.isListedLayer = function(layer)
{
  return !(
      (Legato.Lang.ObjectUtils.isBoolean(layer.options.legatoIgnore) && layer.options.legatoIgnore)
      ||
      (Legato.Lang.ObjectUtils.isBoolean(layer.displayInLayerSwitcher) && !layer.displayInLayerSwitcher)
  );
};

Legato.OpenLayers.Map.Util.isListedBaseLayer = function(layer)
{
  return layer.isBaseLayer && Legato.OpenLayers.Map.Util.isListedLayer(layer);
};

Legato.OpenLayers.Map.Util.isListedOverlay = function(layer)
{
  return !layer.isBaseLayer && Legato.OpenLayers.Map.Util.isListedLayer(layer);
};

Legato.OpenLayers.Map.Util.getListedBaseLayers = function(layers)
{
  return Legato.Lang.CollectionUtils.filter(layers, Legato.OpenLayers.Map.Util.isListedBaseLayer);
};

Legato.OpenLayers.Map.Util.getListedOverlays = function(layers)
{
  return Legato.Lang.CollectionUtils.filter(layers, Legato.OpenLayers.Map.Util.isListedOverlay);
};

/*
 * Function: getLayersWithoutLegatoIgnore
 * Just returns from given layers all layers which do'nt have the
 * legatoIgnore flag.
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found selected layers without the legatoIgnore flag
 */
Legato.OpenLayers.Map.Util.getLayersWithoutLegatoIgnore = function(layers){
  return Legato.Lang.CollectionUtils.filter(
      layers,
      function(layer){
        return !Legato.OpenLayers.Map.Util.isIgnoredLayer(layer);
      }
  );
};

/*
 * Function: getLayersLegatoIgnore
 * Just returns from given layers all layers which have the
 * legatoIgnore flag set to true.
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found selected layers where the legatoIgnore flag is true
 */
Legato.OpenLayers.Map.Util.getLayersLegatoIgnore = function(layers){
  return Legato.Lang.CollectionUtils.filter(
      layers,
      function(layer){
        return Legato.OpenLayers.Map.Util.isIgnoredLayer(layer);
      }
  );
};

/*
 * Function: getBaseLayers
 * Just returns from given layers baselayers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found baselayers
 */
Legato.OpenLayers.Map.Util.getBaseLayers = function(layers){
  return Legato.Lang.CollectionUtils.filter(
    layers,
    function(layer){
      return (layer.isBaseLayer);
    }
  );
};

/*
 * Function: getVisibleLayers
 * Just returns from given layers visible layers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found visible layers
 */
Legato.OpenLayers.Map.Util.getVisibleLayers = function(layers){
  return Legato.Lang.CollectionUtils.filter(
    layers,
    function(layer){
      if (!Legato.OpenLayers.Map.Util.isListedLayer(layer)) {
        return false;
      }
      return layer.visibility;
    }
  );
};

/*
 * Function: getInvisibleLayers
 * Just returns from given layers invisible layers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found invisible layers
 */
Legato.OpenLayers.Map.Util.getInvisibleLayers = function(layers){
  return Legato.Lang.CollectionUtils.filter(
    layers,
    function(layer){
      if (!Legato.OpenLayers.Map.Util.isListedLayer(layer)) {
        return true;
      }
      return !layer.visibility;
    }
  );
};

/*
 * Function: getInvisibleBaseLayers
 * Just returns from given layers invisible baselayers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found invisible baselayers
 */
Legato.OpenLayers.Map.Util.getInvisibleBaseLayers = function(layers){
  //Due we don't expect any performance issues we concatenate
  return Legato.OpenLayers.Map.Util.getInvisibleLayers(Legato.OpenLayers.Map.Util.getBaseLayers(layers));
};

/*
 * Function: getVisibleBaseLayers
 * Just returns from given layers visible baselayers
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found visible baselayers
 */
Legato.OpenLayers.Map.Util.getVisibleBaseLayers = function(layers){
  //Due we don't expect any performance issues we concatenate
  return Legato.OpenLayers.Map.Util.getVisibleLayers(Legato.OpenLayers.Map.Util.getBaseLayers(layers));
};

/*
 * Function: getInvisibleOverlays
 * Just returns from given layers invisible overlays
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found invisible overlays
 */
Legato.OpenLayers.Map.Util.getInvisibleOverlays = function(layers){
  //Due we don't expect any performance issues we concatenate
  return Legato.OpenLayers.Map.Util.getInvisibleLayers(Legato.OpenLayers.Map.Util.getOverlays(layers));
};

/*
 * Function: getVisibleOverlays
 * Just returns from given layers visible overlays
 *
 * Returns:
 * {Array(<OpenLayers.Layer>)} All found visible overlays
 */
Legato.OpenLayers.Map.Util.getVisibleOverlays = function(layers){
  //Due we don't expect any performance issues we concatenate
  return Legato.OpenLayers.Map.Util.getVisibleLayers(Legato.OpenLayers.Map.Util.getOverlays(layers));
};

/*
 * Function: getLayerIds
 * Extracts from given layers array all layerids and returns them as an array
 *
 * Returns:
 * {Array(String)} All layer ids
 */
Legato.OpenLayers.Map.Util.getLayerIds = function(layers){
  var layerids = [];

  for(var i = 0; i < layers.length; i++){
    var layerid = Legato.Lang.ObjectUtils.isNullOrUndefined(layers[i].layerId) ?
        layers[i].name : layers[i].layerId;
    layerids.push(layerid);
  }

  return layerids;
};

/*
 * Function: getBoxSelectionAsVector
 * Returns a new box vector based on drawn mouse position box.
 *
 * Parameters:
 * params - {Object}
 * map <OpenLayers.Map> an OpenLayers Map
 * position <OpenLayers.Bounds> a Selection which wants to be transformed into a vector layer
 *
 * Returns:
 * {<OpenLayers.Feature.Vector>} - The given Selection as Vector
 */
Legato.OpenLayers.Map.Util.getBoxSelectionAsVector = function(map, position){
  Legato.Util.Ensure.ensureObject(map);
  Legato.Util.Ensure.ensureObject(position);

  var lowerLeftLonLat = map.getLonLatFromPixel(
    new OpenLayers.Pixel(position.left, position.bottom));
  var upperRightLonLat = map.getLonLatFromPixel(
    new OpenLayers.Pixel(position.right, position.top));
  var bounds = new OpenLayers.Bounds(
    lowerLeftLonLat.lon,
    lowerLeftLonLat.lat,
    upperRightLonLat.lon,
    upperRightLonLat.lat
  );

  return new OpenLayers.Feature.Vector(bounds.toGeometry());
};

/*
 * Function: getPointSelectionAsVector
 *
 * Parameters:
 * params
 * map <OpenLayers.Map> an OpenLayers Map
 * position <OpenLayers.Bounds> a Selection which wants to be transformed into a vector layer
 *
 * Returns:
 * {<OpenLayers.Feature.Vector>} - The given Selection as Vector
 */
Legato.OpenLayers.Map.Util.getPointSelectionAsVector = function(map, position){
  Legato.Util.Ensure.ensureObject(map);
  Legato.Util.Ensure.ensureObject(position);

  var lonlat = map.getLonLatFromPixel(position);
  var point = new OpenLayers.Geometry.Point(lonlat.lon, lonlat.lat);

  return new OpenLayers.Feature.Vector(point);
};

/*
 * Function: isValidBBox
 * Validates given BBox if it contains some reasonable values
 *
 * Parameters:
 * bbox - {<OpenLayers.Bounds>} The BBox to validate
 *
 * Returns:
 * {Boolean} - True if valid, false otherwise
 */
Legato.OpenLayers.Map.Util.isValidBBox = function(bbox){
  if(!Legato.Lang.ObjectUtils.isObject(bbox)){
    return false;
  }

  //Now validate that our created bbox does not contain any null values
  if(Legato.Lang.ObjectUtils.isNullOrUndefined(bbox.left) ||
     Legato.Lang.ObjectUtils.isNullOrUndefined(bbox.right) ||
     Legato.Lang.ObjectUtils.isNullOrUndefined(bbox.top) ||
     Legato.Lang.ObjectUtils.isNullOrUndefined(bbox.bottom)){
    return false;
  }

  //And also validate that given values could be allright
  if(((bbox.top - bbox.bottom) <= 0.000001) || ((bbox.right - bbox.left) <= 0.000001)) {
    return false;
  }

  return true;
};

/*
 * TODO: Needed?
 */
Legato.OpenLayers.Map.Util.getDummyLayer = function() {
  return new OpenLayers.Layer.WMS(
      // name
      "dummy",
      // url
      "http://www2.demis.nl/wms/wms.asp?wms=WorldMap&",
      // params
      {
        layers: 'Bathymetry',
        format:'image/png'
      });
};

/*
 * Function: moveLayer
 * Moves the given layer up or down
 *
 * Parameters:
 * params - {Object} a hashmap which contains
 * - layer {<OpenLayers.Layer>} the layer to move
 * - up {Boolean} true =  layer will be moved up, false = moved down
 */
Legato.OpenLayers.Map.Util.moveLayer = function(params){
    var layer    = params.layer;
    var map      = layer.map;
    var overlays = Legato.OpenLayers.Map.Util.getOverlays(map.layers);

    var i = OpenLayers.Util.indexOf(overlays, layer);

    if(params.up){
      --i;
    } else {
      ++i;
    }

    if(i < 0 || i > overlays.length - 1){
      return;
    }

    OpenLayers.Util.removeItem(overlays, layer);
    overlays.splice(i, 0, layer);

    for(var j = 0; j < overlays.length; j++){
      map.removeLayer(overlays[j]);
    }

    map.addLayers(overlays);
};

/*
 * Function: getViewPortPxFromLonLat
 * Returns a pixel location given a map location. This method will return
 * fractional pixel values.
 *
 * Parameters:
 * lonlat - {<OpenLayers.LonLat>}
 * resolution - {Number}
 * extent - {<OpenLayers.Bounds>}
 *
 * Returns:
 * {<OpenLayers.Pixel>} An <OpenLayers.Pixel> which is the passed-in
 * <OpenLayers.LonLat>,translated into view port pixels.
 */
Legato.OpenLayers.Map.Util.getViewPortPxFromLonLat = function (lonlat, resolution, extent) {
  var px = null;
  if (lonlat !== null) {
    px = new OpenLayers.Pixel(
      (1/resolution * (lonlat.lon - extent.left)),
      (1/resolution * (extent.top - lonlat.lat))
    );
  }
  return px;
};