/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author$
 * @version $Revision$
 * @requires Legato.js
 * @requires Legato/Util.js
 */

/**
 * Class: Legato.Lang.CollectionUtils
 * Some common utils for collections
 */
Legato.Lang.CollectionUtils = {};

/**
 * Function: Legato.Lang.CollectionUtils.filter
 *
 * Filters given array based on given evaluate method. If evaluate
 * method returns true, arrayelement will been keeped. False otherwise.
 *
 * Parameters:
 * array - {Object} The array to filter
 * evaluate - {Function} the filter function
 *
 * Returns:
 * {Object} The filtered array
 */
Legato.Lang.CollectionUtils.filter = function(array, evaluate){
  Legato.Util.Ensure.ensureArray(array);
  Legato.Util.Ensure.ensureFunction(evaluate);

  var filtered = [];
  for(var i = 0; i < array.length; i++){
    if(evaluate(array[i])){
      filtered.push(array[i]);
    }
  }

  return filtered;
};

/**
 * Function: Legato.Lang.CollectionUtils.filterObject
 *
 * Filters given object based on given evaluate method. If key is in evaluate,
 * object property will be keeped. False otherwise.
 *
 * Parameters:
 * object - {Object} The object to filter
 * evaluate - {Array} the array of properties to keep
 *
 * Returns:
 * {Object} The filtered object
 */
Legato.Lang.CollectionUtils.filterObject = function(object, evaluate){
  Legato.Util.Ensure.ensureObject(object);
  Legato.Util.Ensure.ensureArray(evaluate);

  var filtered = [];
  Legato.Lang.CollectionUtils.copyIfExists(object, filtered, evaluate);
  return filtered;
};

/**
 * Function: Legato.Lang.CollectionUtils.transform
 *
 * Transforms given array based on given transformer method. The new build
 * array will then been returned.
 *
 * Parameters:
 * array - {Object} The array to filter
 * evaluate - {Function} the filter function
 *
 * Returns:
 * {Object} The filtered array
 */
Legato.Lang.CollectionUtils.transform = function(array, transformer){
  Legato.Util.Ensure.ensureArray(array);
  Legato.Util.Ensure.ensureFunction(transformer);

  var transformed = [];

  for(var i = 0; i < array.length; i++){
    transformed.push(transformer(array[i]));
  }

  return transformed;
};

/**
 * Function: Legato.Lang.CollectionUtils.each
 *
 * Calls a given function on each element of an array
 *
 * Parameters:
 * array - {Object} The array to iterate
 * func - {Function} the function to call
 *
 * Returns:
 * nothing
 */
Legato.Lang.CollectionUtils.each = function(array, func) {
  Legato.Util.Ensure.ensureArray(array);
  Legato.Util.Ensure.ensureFunction(func);

  for(var i = 0; i < array.length; i++){
    func(array[i]);
  }
};

/**
 * Function: copyIfExists
 * 
 * Copies all Properties from source to destination if they exist in source
 * 
 * Returns:
 * 
 * nothing
 * 
 * Parameters:
 * 
 * source - source object
 * destination - destination object
 * properties - array with property names
 */
Legato.Lang.CollectionUtils.copyIfExists = function(source, destination, properties) {
  Legato.Util.Ensure.ensureArray(properties);

  Legato.Lang.CollectionUtils.each(properties, function(property) {
    if (Legato.Lang.ObjectUtils.exists(source[property])) {
      destination[property] = source[property];
    }
  });
};
