/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: leine $
 * @version $Rev: 72369 $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: Legato.Control.GeoLocation
 *
 * Control which zooms the current Map to the current position of the user if available.
 *
 * Inherits from: - <Legato.Control>
 */
Legato.Control.GeoLocation = OpenLayers.Class(Legato.Control, {


  currentPositionLayer: null,

  zoomLevel: null,

  icon: null,

  iconUrl: null,

  iconHeight: null,

  iconWidth: null,

  name: 'My Position',

  zoomToCurrentPosition: true,

  showInLegend: true,

  /**
   * Constructor: Legato.Control.WMSGetFeatureInfo Create a WMSGetFeatureInfo.
   *
   * Parameters:
   */
  initialize : function(options) {
    Legato.Control.prototype.initialize.apply(this, [ options ]);

    if(this.iconUrl !== null &&
       this.iconHeight !== null &&
       this.iconWidth !== null){
      var size = new OpenLayers.Size(this.iconWidth,this.iconHeight);
      var offset = new OpenLayers.Pixel(-(size.w/2), -size.h);
      this.icon = new OpenLayers.Icon(this.iconUrl,size,offset);
    }
  },

  /**
   * APIMethod: setMap
   *
   * See Also: <OpenLayer.Control>
   */
  setMap : function(map) {
    OpenLayers.Control.prototype.setMap.apply(this, arguments);

    /*
     * Get max zoomLevel if not defined
     */
    if(this.zoomLevel === null){
      /*
       * We're decrementing the zoomLevel cause otherwise the isValidZoomLevel
       * method will allways return false which prevents the client from zooming
       * to the max level.
       */
      this.zoomLevel = this.map.getNumZoomLevels() - 1;
    }
  },

  /**
   * Function: draw
   *
   * See Also: <OpenLayer.Control>
   */
  draw : function() {
    OpenLayers.Control.prototype.draw.apply(this);
  },

  /**
   * APIMethod: destroy
   *
   * See Also: <OpenLayer.Control>
   */
  destroy : function() {
    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  /**
   * APIMethod: activate
   *
   * See Also: <OpenLayer.Control>
   */
  activate : function() {
    if(!Legato.Util.GeoLocation.isAvailable()){
      return false;
    }

    OpenLayers.Control.prototype.activate.apply(this, arguments);

    /*
     * Might occur if user has deleted the layer via the legend control
     */
    if(this.currentPositionLayer !== null && this.map.getLayer(this.currentPositionLayer.id) === null){
      this.currentPositionLayer.setVisibility(true);
      this.map.addLayer(this.currentPositionLayer);
    }

    /*
     * Position has not been shown yet
     */
    if(this.currentPositionLayer === null){
      Legato.Util.GeoLocation.getCurrentPosition(
          this.onPositionDefined.bind(this), this.onPositionError.bind(this)
      );
    }

    if(this.zoomToCurrentPosition && this.map.isValidLonLat(this.currentPosition)){
      if(this.map.isValidZoomLevel(this.zoomLevel)){
        this.map.setCenter(this.currentPosition, this.zoomLevel);
      }
      this.map.setCenter(this.currentPosition);
    }
  },

  /**
   * APIMethod: deactivate
   *
   * See Also: <OpenLayer.Control>
   */
  deactivate : function() {
    OpenLayers.Control.prototype.deactivate.apply(this, arguments);
  },

  /**
   * triggered if some error occured
   */
  onPositionError: function(error){
    switch(error.code){
      case error.PERMISSION_DENIED:
        break;
      case error.POSITION_UNAVAILABLE:
        break;
      case error.TIMEOUT:
        break;
      default:
        break;
    }
    this.deactivate();
  },

  /**
   * Triggered if current Position could be located
   */
  onPositionDefined: function(position){
    this.currentPosition = new OpenLayers.LonLat(position.coords.longitude, position.coords.latitude);

    if(this.zoomToCurrentPosition){
      this.map.setCenter(this.currentPosition, this.zoomLevel);
    }

    if(this.icon === null){
      this.currentPositionLayer = new OpenLayers.Layer.Vector(
        this.name,
        {
          styleMap: new OpenLayers.StyleMap({
            // Set the external graphic and background graphic images.
            externalGraphic: "http://openlayers.org/dev/img/marker-gold.png",
            backgroundGraphic: "http://openlayers.org/dev/examples/marker_shadow.png",

            // Makes sure the background graphic is placed correctly relative
            // to the external graphic.
            backgroundXOffset: 0,
            backgroundYOffset: -7,

            // Set the z-indexes of both graphics to make sure the background
            // graphics stay in the background (shadows on top of markers looks
            // odd; let's not do that).
            graphicZIndex: 11,
            backgroundGraphicZIndex: 10,

            pointRadius: 10
          }),
          isBaseLayer: false,
          rendererOptions: {yOrdering: true}
        }
      );
      var features = [];
      features.push(new OpenLayers.Feature.Vector(new OpenLayers.Geometry.Point(position.coords.longitude, position.coords.latitude)));
      this.currentPositionLayer.addFeatures(features);
      this.currentPositionLayer.options.legatoIgnore = !this.showInLegend;
      this.map.addLayer(this.currentPositionLayer);
    } else {
      this.currentPositionLayer = new OpenLayers.Layer.Markers(this.name);
      this.currentPositionLayer.options.legatoIgnore = !this.showInLegend;
      this.map.addLayer(this.currentPositionLayer);
      this.currentPositionLayer.addMarker(new OpenLayers.Marker(this.currentPosition, this.icon));
    }
  },

  CLASS_NAME :'Legato.Control.GeoLocation'
});

/**
 * Structure: lc:WMSGetFeatureInfo XML based config for a
 * <Legato.Control.WMSGetFeatureInfo>.
 *
 * See Also: - <Legato.Control> - <Legato.Beans.BeanFactory> - <QName>
 *
 * A valid config example for a WMSGetFeatureInfo would be:
 * (start code)
 * <lc:WMSGetFeatureInfo>
 *   <infoFormat>text/xml</infoFormat>
 *   <hover>true</hover>
 *   <maxFeatures>1</maxFeatures>
 *   <queryVisible>true</queryVisible>
 *   <handlerOptions>
 *   <hover>
 *    <delay>200</delay>
 *  </hover>
 *  </handlerOptions>
 * </lc:WMSGetFeatureInfo>
 * (end)
 */
Legato.Control.GeoLocation.Bean = Legato.Control.Bean.Extend(

'Legato.Control.GeoLocation',

/*
 * Property: QName The qualified name for this control. Needed by XML config to
 * instantiate it.
 */
Legato.Control.QName('GeoLocation'), {
  _constructor :Legato.Control.GeoLocation,
  options : {
    /*
     * Property: hover
     *
     * See Also: <OpenLayers.Control.WMSGetFeatureInfo.hover>
     */
    zoomLevel: Legato.Lang.Integer,
    iconUrl: Legato.Lang.String,
    iconHeight: Legato.Lang.Integer,
    iconWidth: Legato.Lang.Integer,
    name: Legato.Lang.String,
    zoomToCurrentPosition: Legato.Lang.Boolean,
    showInLegend: Legato.Lang.Boolean
  }
});