/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: valikov $
 * @version $Rev: 91043 $
 */

/**
 * Class: Legato.BootLoader
 * The Legato BootLoader Class gives you the possibility to configure
 * and load Legato without defining the body onload Event.
 *
 * You can specify the following js variables to use this class:
 *
 * - legatoBootLoader (must be set to true to enable the bootloader)
 * - legatoBootLoaderConfigUrl (define if not config.xml)
 * - legatoBootLoaderOnCreate (callBack method after BeanFactory has been created but did not loaded the config yet)
 * - legatoBootLoaderOnConfigured (callBack method after BeanFactory has been fully loaded)
 * - legatoBootLoaderOnDestroy (callBack method before BeanFactory destroys itself and the current map)
 *
 * Please be adviced that you must define them before including this class.
 *
 * Inherits from: - <Legato.Control>
 */
Legato.BootLoader = Legato.Lang.Class({

  /*
   * Property: configUrl
   * {String} the default configUrl for Legato
   */
  configUrl: 'config.xml',

  /*
   * Property: onCreateCallback
   * {Function} a configurable callback method which will been executed when
   * the Legato BeanFactory has been created but did not load the config yet.
   */
  onCreateCallback: null,

  /*
   * Property: onConfiguredCallback
   * {Function} a configurable callback method which will been executed when
   * the Legato BeanFactory has loaded the configuration.
   */
  onConfiguredCallback: null,

  /*
   * Property: onDestroyCallback
   * {Function} a configurable callback method which will been executed when
   * before the BeanFactory will destroy all objects and the current map.
   */
  onDestroyCallback: null,

  /*
   * Property: beanFactory
   * {Legato.Beans.BeanFactory}
   */
  beanFactory: null,

  /*
   * Property: beanFactory
   * {OpenLayers.Map}
   */
  map: null,

  /*
   * Function: initialize
   * Basic ctor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control (currently not used)
   */
  initialize : function(options) {
    OpenLayers.Control.prototype.initialize.apply(this, arguments);

    if(typeof legatoI18nLocale != 'undefined' && Legato.Lang.ObjectUtils.isString(legatoI18nLocale)) {
      Legato.I18n.setLocale(legatoI18nLocale);
    }
    
    if(typeof legatoBootLoaderConfigUrl != 'undefined' &&
       !Legato.Lang.ObjectUtils.isNullOrUndefined(legatoBootLoaderConfigUrl)){
      this.configUrl = legatoBootLoaderConfigUrl;
    }

    if(typeof legatoBootLoaderOnCreate != 'undefined' &&
        Legato.Lang.ObjectUtils.isFunction(legatoBootLoaderOnCreate)){
      this.onCreateCallback = legatoBootLoaderOnCreate;
    }

    if(typeof legatoBootLoaderOnConfigured != 'undefined' &&
        Legato.Lang.ObjectUtils.isFunction(legatoBootLoaderOnConfigured)){
      this.onConfiguredCallback = legatoBootLoaderOnConfigured;
    }

    if(typeof legatoBootLoaderOnDestroy != 'undefined' &&
        Legato.Lang.ObjectUtils.isFunction(legatoBootLoaderOnDestroy)){
      this.onDestroyCallback = legatoBootLoaderOnDestroy;
    }

    this.load();
    Legato.Util.DOM.addDOMUnLoadEvent(this.unload.bind(this));
  },

  /*
   * Function: initialize
   * Basic ctor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control (currently not used)
   */
  load: function(){
    this.beanFactory = new Legato.Beans.XMLBeanFactory();

    if(Legato.Lang.ObjectUtils.isFunction(this.onCreateCallback)){
      this.onCreateCallback(this.beanFactory);
    }

    this.beanFactory.loadURL({
      url: this.configUrl,
      onFailure : function(request) {
        OpenLayers.Console.error(request);
      },
      onSuccess : function(request) {
        this.map = this.beanFactory.getBean('map');
        if (Legato.Lang.ObjectUtils.exists(this.onConfiguredCallback)) {
          this.onConfiguredCallback(this.beanFactory);
        }
      }.bind(this)
    });
  },

  /*
   * Function: unload
   * Executed when the DOM unloads
   */
  unload: function(){
    if (Legato.Lang.ObjectUtils.exists(this.onDestroyCallback)) {
      this.onDestroyCallback(this.beanFactory);
    }
    if (Legato.Lang.ObjectUtils.exists(this.beanFactory)) {
      this.beanFactory.destroy();
    }
  },

  CLASS_NAME : 'Legato.BootLoader'
});

/*
 * The BootLoader creates itself if the js variable legatoBootLoader is
defined and is set to true
 */
Legato.Util.DOM.addDOMLoadEvent(function() {
  if(typeof legatoBootLoader != 'undefined' &&
      Legato.Lang.ObjectUtils.isTrue(legatoBootLoader)){
      legatoBootLoader = new Legato.BootLoader();
    }
  }
);