/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: valikov $
 * @version $Rev: 70190 $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: Legato.Service.LayerCapabilities
 * Central Legato LayerCapabilities Service which provides some additional informations for a
 * given layer.
 *
 * Inherits from: - <Legato.Service>
 */
Legato.Service.LayerCapabilities = OpenLayers.Class(Legato.Singleton, Legato.Service,
{
  /*
   * Used for internal caching. Key is the layer url, value the layers capabilites
   */
  cache: null,

  /*
   * Function: initialize
   * Basic ctor
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this control
   */
  initialize : function(options) {
    Legato.Service.prototype.initialize.apply(this, arguments);
    this.cache = [];
  },

  /*
   * common validate functions which ensures that given arguments
   * contain the needed objects.
   *
   * Parameters:
   * args - {Object} A hashmap which contains a layer and a callBack function
   */
  validateArguments: function(args){
    if(typeof args.layer != 'object' ||
       typeof args.layer.CLASS_NAME == 'undefined' ||
       args.layer.CLASS_NAME != 'OpenLayers.Layer.WMS'){
      throw('Need a valid argument layer which needs to be an OpenLayers.Layer.WMS instance');
    }

    if(typeof args.onComplete != 'function'){
      throw ('Missing needed argument onComplete which has to be a function');
    }

    /*
     * Cause this service allways returns its result in given arg hashmap with the named key
     * result we have to ensure
     */
    if(!Legato.Lang.ObjectUtils.isNullOrUndefined(args.result)){
      throw new Legato.Lang.IllegalArgumentException('Given arguments allready contain a result. Please ensure that the key result is never used');
    }
  },

  /*
   * Determines if wanted capabilitites are cached allready
   * and if so provides them from cache. If not an AJAX request
   * is triggered.
   *
   * Parameters:
   * args - {Object} A hashmap which contains a <OpenLayers.Layer> and a callBack function
   */
  loadCapabilitites: function(args){
    if(typeof this.cache[args.layer.url] == 'undefined'){
      this.cache[args.layer.url]    = Legato.Service.BUSY;
      try{
        OpenLayers.loadURL(
          args.layer.url,
          ({
            REQUEST: 'GetCapabilities',
            SERVICE: 'WMS'
          }),
          this,
          function(request){
            /*
            * Fired if AJAX request returned successfully
            */
            var wms = new OpenLayers.Format.WMSCapabilities();
            var doc = wms.read(request.responseText);
            this.cache[args.layer.url] = doc;
            args.onComplete(this.cache[args.layer.url]);
          },
          function(request){
            /*
            * Fire if AJAX request causes an error
            */
            if(typeof args.onError  != 'function'){
              OpenLayers.Console.warn(request);
              return;
            }
            args.onError(request );
          }
        );
        return;
      } catch(exp){
        this.cache[args.layer.url] = null;
        if(typeof args.onError  != 'function'){
          OpenLayers.Console.warn(exp);
          return;
        }
        args.onError(exp);
      }
    }
    args.onComplete(this.cache[args.layer.url]);
  },

  /*
   * Function: getCapabilitites
   * Gets the capabiltites for given layer. If capabilities could been loaded
   * the onComplete callback method will be executed. All given args will been
   * given back in callback method plus a new key entry 'result' where the capabilities result
   * will be stored into.
   *
   * (start code)
   * //The followin example shows a valid result
   * onComplete(
   *   result: {
   *     capabilities: {
   *       ...
   *     }
   *   }
   * )
   *
   * //This is an example if current service is busy
   * onComplete(
   *   result: Legato.Service.BUSY
   * )
   * (end)
   *
   *
   * Parameters:
   * args - {Object} A hashmap which contains a layer and a callBack function
   *
   * The following keys are supported:
   * - layer {<OpenLayers.Layer>} The layer from which you want its capabilites
   * - onComplete - {Function} fired if capabilites are loaded or if service is currently busy
   *
   * See also:
   * <Legato.Service.BUSY>
   */
  getCapabilitites: function(args){
    this.validateArguments(args);
    this.loadCapabilitites({
      layer: args.layer,
      onComplete: OpenLayers.Function.bind(function(doc){
        var context = doc === Legato.Service.BUSY ?
          this.createResultContext(Legato.Service.BUSY, args) :
          this.createResultContext({capabilitites: doc}, args);
        args.onComplete(context);
        return;
      }, this),
      onError: args.onError
    });
  },

  /*
   * Small helper function which creates a new context hashmap,
   * copies given context into it, appends afterwards given
   * result and returns it.
   */
  createResultContext: function(result, previousContext){
    //Create a new context and put our result in it
    var context = {};
    //Copy the previous context into our new context
    OpenLayers.Util.extend(context, previousContext);
    //Append given result
    context.result = result;
    return context;
  },

  /*
   * Function: getLegendUrl
   * Tries to determine the legendUrl for given layer and executes the callBack method
   * if found.
   * (start code)
   * //The followin example shows a valid result
   * onComplete(
   *   result: {
   *     legendUrl: {
   *       ...
   *     }
   *   }
   * )
   *
   * //This is an example if current service is busy
   * onComplete(
   *   result: Legato.Service.BUSY
   * )
   * (end)
   *
   * Parameters:
   * args - {Object} A hashmap which contains a layer and a callBack function
   *
   * The following keys are supported:
   * - layer {<OpenLayers.Layer>} The layer from which you want its capabilites
   * - onComplete - {Function} fired if capabilites are loaded or if service is currently busy
   *
   * See also:
   * <Legato.Service.BUSY>
   */
  getLegendUrl: function(args){
    this.validateArguments(args);
    this.loadCapabilitites({
      layer: args.layer,
      onComplete: OpenLayers.Function.bind(function(doc){
        if(doc === Legato.Service.BUSY){
          OpenLayers.Console.debug('I am busy right now. Try again later');
          args.onComplete(this.createResultContext(doc, args));
          return;
        }
        var layercapabilities = null;

        if(Legato.Lang.ObjectUtils.isNullOrUndefined(doc) ||
           Legato.Lang.ObjectUtils.isNullOrUndefined(doc.capability) ||
           Legato.Lang.ObjectUtils.isNullOrUndefined(doc.capability.layers)){
          return;
        }

        for(var i = 0; i < doc.capability.layers.length; i++){
          if(doc.capability.layers[i].name != args.layer.name){
            continue;
          }
          layercapabilities = doc.capability.layers[i];
        }

        if(layercapabilities === null || layercapabilities.styles.length === 0){
          OpenLayers.Console.debug('There is no legendUrl');
          args.onComplete(this.createResultContext({legendUrl: null}, args));
          return;
        }

        OpenLayers.Console.debug('There is a legendUrl');
        args.onComplete(this.createResultContext({legendUrl: layercapabilities.styles[0].legend}, args));
        return;
      }, this),
      onError: args.onError
    });
  },

  CLASS_NAME : 'Legato.Service.LayerCapabilities'
});