/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * @author $Author: valikov $
 * @version $Rev: 70574 $
 */

/*
 * Class: Legato.Graph.DirectedGraph
 * 
 * Implements a directed graph.
 */
Legato.Graph.DirectedGraph = Legato.Lang.Class( {
  /*
   * Property: vertices
   * 
   * {Object} Name/{<Legato.Graph.Vertex>} map.
   */
  vertices :null,
  /*
   * Constructor: initialize
   * 
   * Constructs a new directed graph.
   */
  initialize : function() {
    this.vertices = {};
  },
  /*
   * Function: getVertex
   * 
   * Returns: {<Legato.Graph.Vertex>} Existing vertex for the given name or
   * null if no vertex found.
   * 
   * Parameters:
   * 
   * name - {String} name of the vertex.
   */
  getVertex : function(name) {
    Legato.Util.Ensure.ensureString(name);
    var vertex = this.vertices[name];
    if (Legato.Lang.ObjectUtils.exists(vertex)) {
      return vertex;
    } else {
      return null;
    }
  },
  /*
   * Function: getOrCreateVertex
   * 
   * Returns: {<Legato.Graph.Vertex>} Either existing or a new vertex for the
   * given name.
   * 
   * Parameters:
   * 
   * name - {String} name of the vertex.
   */
  getOrCreateVertex : function(name) {
    Legato.Util.Ensure.ensureString(name);
    var vertex = this.vertices[name];
    if (!vertex) {
      vertex = new Legato.Graph.Vertex(name);
      this.vertices[name] = vertex;
    }
    return vertex;
  },
  /*
   * Function: addEdge
   * 
   * Creates an edge between two vertices.
   * 
   * Parameters:
   * 
   * parentName - {String} name of the parent vertex. Parameters:
   * 
   * childName - {String} name of the child vertex.
   */
  addEdge : function(parentName, childName) {
    Legato.Util.Ensure.ensureString(parentName);
    Legato.Util.Ensure.ensureString(childName);
    var parentVertex = this.getOrCreateVertex(parentName);
    var childVertex = this.getOrCreateVertex(childName);
    parentVertex.outgoing.add(childName);
    childVertex.incoming.add(parentName);
  },
  /*
   * Function: removeEdge
   * 
   * Removes an edge between two vertices.
   * 
   * Parameters:
   * 
   * parentName - {String} name of the parent vertex. Parameters:
   * 
   * childName - {String} name of the child vertex.
   */
  removeEdge : function(parentName, childName) {
    Legato.Util.Ensure.ensureString(parentName);
    Legato.Util.Ensure.ensureString(childName);
    var parentVertex = this.getVertex(parentName);
    if (parentVertex) {
      parentVertex.outgoing.remove(childName);
    }
    var childVertex = this.getVertex(childName);
    if (childVertex) {
      childVertex.incoming.remove(parentName);
    }
  },
  /*
   * Function: clone
   * 
   * Clones the graph.
   * 
   * Returns: {<Legato.Graph.DirectedGraph>} A clone of this graph.
   */
  clone : function() {
    var graph = new Legato.Graph.DirectedGraph();
    for ( var name in this.vertices) {
      if (this.vertices.hasOwnProperty(name)) {
        graph.vertices[name] = this.vertices[name].clone();
      }
    }
    return graph;
  },
  /*
   * Function: toArray
   * 
   * Performs the topological sorting of the given graph and returns the array
   * of vertex names in the sorted order.
   * 
   * Returns:
   * 
   * {Array(String)} Topologically sorted array of vertex names.
   */
  toArray : function() {
    var graph = this.clone();
    var array = [];
    var roots = new Legato.Util.HashSet();
    for ( var vertexName in graph.vertices) {
      if (graph.vertices.hasOwnProperty(vertexName)) {
        var vertex = graph.vertices[vertexName];
        if (!vertex.hasIncomingEdges()) {
          roots.add(vertexName);
        }
      }
    }
    while (!roots.isEmpty()) {
      var rootName = roots.toArray()[0];
      roots.remove(rootName);
      var rootVertex = graph.getVertex(rootName);
      array.push(rootName);
      var outgoingNames = rootVertex.outgoing.toArray();
      for (var index = 0; index < outgoingNames.length; index++) {
        var outgoingName = outgoingNames[index];
        var outgoingVertex = graph.getVertex(outgoingName);
        graph.removeEdge(rootName, outgoingName);
        if (!outgoingVertex.hasIncomingEdges()) {
          roots.add(outgoingName);
        }
      }
    }
    return array;
  }
});