/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author$
 * @version $Rev$
 */

/**
 * Class: Legato.Util.Accessor
 * Small helper class which implements some sort of an setter, getter Pattern.
 * With this class you're able to wrap some other classes so that the wrapped
 * classed is accessed just with the getter setter Pattern.
 */
Legato.Util.Accessor = Legato.Lang.Class({

  /*
   * Property: target
   * {Object} The wrapped target class
   */
  target: null,

  /*
   * Property: field
   * If defined the field will be used if existing in target
   */
  field: null,

  /*
   * Property: setter
   * If defined this function will be used as setter if existing in target
   */
  setter: null,

  /*
   * Property: getter
   * If defined this function will be used as getter if existing in target
   */
  getter: null,

  /*
   * Function: initialize
   * Basic ctor for the accessor class
   *
   * Parameters:
   * options - {Object} A Hashmap containing option parameters for this class
   * The following keys are supported
   * - target {Object} the used target instance
   * - field {String} Optional field specification. Will be used as getter and setter if defined
   * - setter {String} Optional function specification which must exists in target
   * - getter {String} Optional function specification which must exists in target
   */
  initialize : function(options) {
    this.target = options.target || null;
    this.field  = options.field  || null;
    this.setter = options.setter || null;
    this.getter = options.getter || null;

    if(typeof this.target == 'undefined'){
      throw new Error('A target is needed');
    }

    if(typeof this.field != 'string' && (typeof this.target[this.setter] != 'function' || typeof this.target[this.getter] != 'function')){
      throw new Error('You need to specifiy at least the used target field or a setter and getter function which exists in given target');
    }
  },

  /*
   * Function: setValue
   * Setter for wraped target instance value
   *
   * Parameters:
   * value - {Object} The value to set. Could be any type
   */
  setValue : function(value) {
    if (this.field) {
      this.target[this.field] = value;
    } else {
      this.target[this.setter](value);
    }
  },

  /*
   * Function: getValue
   * Getter for wraped target instance value
   *
   * Returns:
   * {Object} Could be any type
   */
  getValue : function() {
    var value = this.field ? this.target[this.field] : this.target[this.getter]();
    return value;
  },

  CLASS_NAME : 'Legato.Util.Accessor'
});