/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/**
 * @author $Author: valikov $
 * @version $Rev: 91043 $
 * @base Legato.Control
 * @requires Legato/Control.js
 */

/**
 * Class: Legato.Control.SelectionBoxValueProvider
 * A menu button which integrates the <OpenLayers.Control.Panel>. If triggered an
 * vector will been painted by moving the mouse. If selection is done and the getValue
 * callback function is implemented, you'll reveive the currently painted bbox.
 * On the oposite you have the possibility to initialize this control by implementing
 * the setValue function.
 *
 * Inherits from:
 * - <Legato.Control>
 *
 * See Also:
 * - <OpenLayers.Control.Panel>
 */
Legato.Control.SelectionBoxValueProvider = OpenLayers.Class(Legato.Control,
{
    /*
     * Property: type
     * {String} OpenLayers.Control.TYPE_TOOL
     */
    type: OpenLayers.Control.TYPE_TOOL,

    /*
     * Property: selectionLayer
     * {OpenLayers.Feature.Vector} The vector layer which will be created and
     * managed by this control
     *
     * See Also:
     * <OpenLayers.Feature.Vector>
     */
    selectionLayer: null,

    /*
     * Property: styleMap
     * {<OpenLayers.StyleMap>}
     */
    styleMap: null,

    /*
     * Property: Selection
     * {String} The default name of the selection box layer. Will be used
     * if not defined in ctor.
     */
    defaultLayerName: 'Selection',

    /*
     * Property: defaultOpacity
     * {Number}  The default opacity value of the selectionbox layer. Will be used
     * if not defined in ctor.
     */
    defaultOpacity: 1,

   /*
    * Function: initialize
    * Basic ctor
    *
    * Parameters:
    * options - {Object} A Hashmap containing option parameters for this control
    * The following keys are supported
    * - target {object} an object which wants to observe,
    * - property{function|String} the propety or function which should be used for providing the observed value
    * - bBoxProperty {String} any Property name of an <OpenLayers.Bounds> instance
    */
    initialize : function(options) {
      OpenLayers.Control.prototype.initialize.apply(this, arguments);

      /*
       * Normalize given options
       */
      this.layerName = this.layerName || this.defaultLayerName;
      try{
        this.opacity = Legato.Lang.NumberUtils.toFloat(this.opacity);
      } catch(ex) {
        this.opacity = this.defaultOpacity;
      }
    },

   /*
    * Function: destroy
    *
    * See Also:
    * <OpenLayer.Control>
    */
    destroy : function() {
      OpenLayers.Control.prototype.destroy.apply(this, arguments);
    },

  /*
   * Function: draw
   *
   * See Also:
   * <OpenLayer.Control>
   */
    draw: function() {
        this.initVectorLayer();
        //Initializes the handler and callback methods
        this.handler = new OpenLayers.Handler.Box( this, {
          done: this.onSelectionBoxDrawn
        }, {
          keyMask: this.keyMask
        });
    },

    /*
     * Inits the vector layer
     */
    initVectorLayer: function(){
      if(!Legato.Lang.ObjectUtils.isFunction(this.getValue)){
        return;
      }

      var bbox = this.getValue();

      if(Legato.OpenLayers.Map.Util.isValidBBox(bbox)){
        this.createVectorLayer();
        this.selectionLayer.addFeatures(new OpenLayers.Feature.Vector(bbox.toGeometry()));
      }
    },

    /*
     * Creates the managed vectorlayer which will been used for drawing
     * the mouse selection as a bbox vector.
     */
    createVectorLayer: function(){
      var selectionLayerOptions = {};
      if (Legato.Lang.ObjectUtils.exists(this.styleMap))
      {
        selectionLayerOptions.styleMap = this.styleMap;
      }
      else
      {
        selectionLayerOptions.styleMap = new OpenLayers.StyleMap({
              "default": new OpenLayers.Style({
                  fillColor: "#ffffff",
                  fillOpacity: "0.5",
                  strokeColor: "#ff0000",
                  strokeWidth: "2"
              })
          });
      }

      this.selectionLayer = new OpenLayers.Layer.Vector(this.layerName, selectionLayerOptions);
      this.selectionLayer.setOpacity(this.opacity);
      this.selectionLayer.options.legatoIgnore = true;
      this.map.addLayer(this.selectionLayer);
      var index = Math.max(this.map.Z_INDEX_BASE.Feature - 1,
          this.selectionLayer.getZIndex()) + 1;
      this.selectionLayer.setZIndex(index);
      this.selectionLayer.events.on({
        'featureadded' : this.propagateValue,
        'featureremoved' : this.propagateValue,
        'featuremodified' : this.propagateValue,
        scope : this
      });
    },


    /**
     * Triggered if the managed vector layer is enabled and the mouse
     * has painted a selection box. The painted box and its coordinates
     * will been given as argument.
     *
     * Parameters:
     * position - {<OpenLayers.Pixel>} The current mouse position, a.k. bbox selection
     */
    onSelectionBoxDrawn: function (position) {
      if(this.selectionLayer === null){
        this.createVectorLayer();
      }

      if(this.selectionLayer.features.length > 0){
        this.selectionLayer.destroyFeatures();
      }

      var boxVector = Legato.OpenLayers.Map.Util.getBoxSelectionAsVector(this.map, position);

      this.selectionLayer.addFeatures(boxVector);
    },

   /*
    * updates all subscribed target's values
    */
    propagateValue: function(){
      if(!Legato.Lang.ObjectUtils.isFunction(this.setValue)){
        return;
      }

      var bbox = new OpenLayers.Bounds(null,null,null,null);

      if(this.selectionLayer.features.length == 1){
        bbox = this.selectionLayer.features[0].geometry.getBounds();
      }

      this.setValue(bbox);
    },

    CLASS_NAME : 'Legato.Control.SelectionBoxValueProvider'
});

/**
 * Structure: lc:SelectionBoxValueProvider
 * XML based config for a <Legato.Control.lc:SelectionBoxValueProvider>.
 *
 * See Also:
 * - <Legato.Control>
 * - <Legato.Beans.BeanFactory>
 * - <QName>
 *
 * A valid config example for a Legend would be:
 * (start code)
 * <lc:SelectionBoxValueProvider opacity="0.5" layerName="Foo" left="selectionLeft" right="selectionRight" top="selectionTop" bottom="selectionBottom"/>
 * (end)
 */
Legato.Control.SelectionBoxValueProvider.Bean = Legato.Control.Bean.Extend(

  'Legato.Control.SelectionBoxValueProvider',

  /*
   * Property: QName
   * The qualified name for this control. Needed by XML config to instantiate it.
   */
  Legato.Control.QName('SelectionBoxValueProvider'),
  {
   /*
    * Creates a basic <Legato.Control.SelectionBoxValueProvider> instance. By default this instance
    * will update the DOMElements by any selection change. If DOMElements contain at construction time
    * their values will been propagated to the ValueProvider which will create a selection box if valid.
    *
    * Parameters:
    * left - {String} The configured DOMElement ID where the left bbox value should be written into
    * bottom - {String} The configured DOMElement ID where the bottom bbox value should be written into
    * right - {String} The configured DOMElement ID where the right bbox value should be written into
    * top - {String} The configured DOMElement ID where the top bbox value should be written into
    *
    * Returns:
    * <Legato.Control.SelectionBoxValueProvider>
    */
    factoryFunction : function(left, bottom, right, top, options)
    {
      var leftElement   = null;
      var bottomElement = null;
      var rightElement  = null;
      var topElement    = null;

      if(Legato.Lang.ObjectUtils.isString(left)){
        leftElement = OpenLayers.Util.getElement(left);
        Legato.Util.Ensure.ensureExists(leftElement, 'SelectionBoxValueProvider: The configured DOMElement with the id' + left + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(bottom)){
        bottomElement = OpenLayers.Util.getElement(bottom);
        Legato.Util.Ensure.ensureExists(bottomElement, 'SelectionBoxValueProvider: The configured DOMElement with the id' + bottom + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(right)){
        rightElement = OpenLayers.Util.getElement(right);
        Legato.Util.Ensure.ensureExists(rightElement, 'SelectionBoxValueProvider: The configured DOMElement with the id' + right + ' could not been found.');
      }
      if(Legato.Lang.ObjectUtils.isString(top)){
        topElement = OpenLayers.Util.getElement(top);
        Legato.Util.Ensure.ensureExists(topElement, 'SelectionBoxValueProvider: The configured DOMElement with the id' + top + ' could not been found.');
      }

      return new Legato.Control.SelectionBoxValueProvider({
        opacity: options.opacity,
        layerName: options.layerName,
        elements: {
          left: leftElement,
          bottom: bottomElement,
          right: rightElement,
          top: topElement
        },
        setValue: function(BBox){
          if(!Legato.Lang.ObjectUtils.exists(BBox)){
            return;
          }
          this.elements.left.value    = BBox.left;
          this.elements.right.value   = BBox.right;
          this.elements.top.value     = BBox.top;
          this.elements.bottom.value  = BBox.bottom;
        },
        getValue: function(){
          try{
            var left    = Legato.Lang.NumberUtils.toFloat(this.elements.left.value);
            var right   = Legato.Lang.NumberUtils.toFloat(this.elements.right.value);
            var top     = Legato.Lang.NumberUtils.toFloat(this.elements.top.value);
            var bottom  = Legato.Lang.NumberUtils.toFloat(this.elements.bottom.value);

            return new OpenLayers.Bounds(left, bottom, right, top);
          }catch(ex){
            OpenLayers.Console.warn(ex);
            return new OpenLayers.Bounds(null, null, null, null);
          }
        }
      });
    },
    options:
    {
     /*
      * Property: opacity
      * {Number} - A numeric value which defined the opacity of drawn selection box. Must be between
      * 0 and 1. If a non parsable value is specified the default value of the control will
      * been used.
      *
      * See Also:
      * <Legato.Control.SelectionBoxValueProvider.defaultOpacity>
      */
      opacity: Legato.Lang.Number,

     /*
      * Property: layerName
      * {String} - Optional Parameter which defines the name of SelectionBox layer. If not specified
      *
      * See Also:
      * <Legato.Control.SelectionBoxValueProvider.defaultLayerName>
      */
      layerName: Legato.Lang.String
    },
    constructorArguments:
    {
     /*
      * Property: left
      * Defines the {DOMElement} id where the left <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      left: Legato.Lang.String,

     /*
      * Property: right
      * Defines the {DOMElement} id where the right <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      right: Legato.Lang.String,

     /*
      * Property: top
      * Defines the {DOMElement} id where the top <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      top: Legato.Lang.String,

     /*
      * Property: bottom
      * Defines the {DOMElement} id where the bottom <OpenLayers.Bounds> value should be written into.
      * The DOMElement must exists in current DOM. Otherwise an Exception is thrown.
      */
      bottom: Legato.Lang.String
    },
    constructorOrder: ['left', 'bottom', 'right', 'top']
  }
);
