/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

Legato.Control.SnappingController = OpenLayers.Class(Legato.Control, {
  map : null,

  snappingResolver : null,

  snappingLayer : null,

  layerId : null,

  logger : null,

  format : new OpenLayers.Format.GeoJSON(),

  initialize : function(options) {
    this.logger = Legato.Util.Logger.getLogger();

    OpenLayers.Control.prototype.initialize.apply(this, arguments);

    this.map.events.on({
      'addlayer' : this.registerFeatureLayerDrawEvents,
      'removelayer' : this.deregisterFeatureLayerDrawEvents,
      scope : this
    });
  },

  /*
   * Function: destroy
   *
   * See Also: <OpenLayer.Control>
   */
  destroy : function() {
    this.map.events.un({
      'addlayer' : this.registerFeatureLayerDrawEvents,
      'removelayer' : this.deregisterFeatureLayerDrawEvents,
      scope : this
    });

    OpenLayers.Control.prototype.destroy.apply(this, arguments);
  },

  deregisterFeatureLayerDrawEvents : function(options) {
    if (Legato.OpenLayers.Map.Util.isVectorLayer(options.layer)) {
      options.layer.events.un({
        'legato.sketchpointadded' : this.onSketchpointAdded,
        scope : this
      });
    }

    // Reset if snapping Layer has been removed
    if (Legato.Lang.ObjectUtils.exists(options.layer.layerId) && options.layer.layerId === this.layerId) {
      this.layerId = null;
      this.snappingLayer = null;
    }
  },

  registerFeatureLayerDrawEvents : function(options) {
    if (Legato.OpenLayers.Map.Util.isVectorLayer(options.layer)) {
      options.layer.events.on({
        'legato.sketchpointadded' : this.onSketchpointAdded,
        scope : this
      });
    }
  },

  /**
   * Method: setLayerId Enables snapping if the given layerId exists in current
   * map.
   *
   * Parameters: layerId - {String} The layerId on wich the snapping should be
   * based
   */
  setLayerId : function(layerId) {
    var layers = this.map.getLayersBy('layerId', layerId);
    this.snappingLayer = layers.size() === 1 ? layers[0] : null;
    this.layerId = this.snappingLayer !== null ? layerId : null;
  },

  getLayerId : function() {
    return this.layerId;
  },

  /*
   * Triggered when the user as added a new point to its current sketch
   */
  onSketchpointAdded : function(event) {
    if (!this.isSnappingActive()) {
      return;
    }

    var snappableLayer = event.layer;
    var snappableGeometry = event.geometry;
    var snappablePoint = event.vertex.clone(null);

    /*
     * Done inline to ensure that this controller remains stateless.
     */

    this.snappingResolver.getSnapPoint(this.layerId, this.map.getResolution(), this.format.extract.geometry.call(this.format, snappablePoint), OpenLayers.Function
        .bind(function(snappedPoint) {
          snappedPoint = this.format.parseGeometry(snappedPoint);

          // Snapping occured?
          if (snappablePoint.equals(snappedPoint)) {
            // this.logger.debug('No snapping needed. Continue!');
          } else {
            if (this.updateGeometry(snappableGeometry, snappablePoint, snappedPoint)) {

//              this.logger.debug('Snappable point:');
//              this.logger.debug(snappablePoint);

              /*
               * If the given geometry has been updated trigger an event so that
               * every listener gets notified about this.
               */
              snappableLayer.events.triggerEvent('legato.sketchupdated', {
                layer : snappableLayer,
                geometry : snappableGeometry
              });
            }
          }
        }, this));
  },

  /*
   * Returns true if snapping is active, false otherwise
   */
  isSnappingActive : function() {
    /*
     * Might occur if snappingcontroller has been preconfigured by the legato
     * config. If so the control is created where there are no layers yet on the
     * map.
     */
    if (this.snappingLayer === null && this.layerId !== null) {
      this.setLayerId(this.layerId);
    }

    return this.snappingLayer !== null;
  },

  /*
   * Updates the given geometry by determining the given oldPoint and changing
   * this one to the given newpoint.
   *
   * Note: oldPoint needs to be a vertex otherwise this method want update the
   * geometry.
   */
  updateGeometry : function(geometry, oldPoint, newPoint) {

    var vertices = geometry.getVertices();
    for ( var i = 0; i < vertices.length; i++) {
      if (vertices[i].equals(oldPoint)) {
        vertices[i].x = newPoint.x;
        vertices[i].y = newPoint.y;
        return true;
      }
    }
    return false;
  },

  CLASS_NAME : 'Legato.Control.SnappingController'
});

Legato.Control.SnappingController.Bean = OpenLayers.Control.Bean.Extend(

'Legato.Control.SnappingController',

/*
 * Property: QName The qualified name for this control. Needed by XML config to
 * instantiate it.
 */
Legato.Control.QName('SnappingController'), {
  _constructor : Legato.Control.SnappingController,
  options : {
    map : OpenLayers.Map,
    snappingResolver : Legato.Control,
    layerId : Legato.Lang.String
  }
});
