/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * @author $Author: valikov $
 * @version $Rev: 70574 $
 */

/*
 * Class: Legato.Beans.BeanFactory
 */
Legato.Beans.BeanFactory = Legato.Lang
    .Class( {
      beanDefinitions :null,
      beanNames :null,
      converters : {},
      instances :null,
      unloadDestroy :null,
      /*
       * Constructor: initialize
       *
       * Creates a bean factory with given bean definitions.
       *
       * Parameters:
       *
       * beanDefinitions - {Object} bean definitions.
       */
      initialize : function(beanDefinitions) {
        this.instances = this.createDefaultInstances();
        if (Legato.Lang.ObjectUtils.exists(beanDefinitions)) {
          this.loadBeanDefinitions(beanDefinitions);
        }
        this.unloadDestroy = OpenLayers.Function.bind(this.destroy, this);
        // // always call beanFactory.destroy()
        OpenLayers.Event.observe(window, 'unload', this.unloadDestroy);
      },
      createDefaultInstances: function ()
      {
        var instances = {};
        return instances;
      },
      loadBeanDefinitions : function(beanDefinitions) {
        Legato.Util.Ensure.ensureObject(beanDefinitions);
        this.setBeanDefinitions(beanDefinitions);
        this.createInstances();
        this.populateInstances();
        this.afterInstancesPopulated();
      },
      retrieveBean : function(beanDefinition) {
        var instance = this.createInstance(beanDefinition);
        this.populateInstance(beanDefinition, instance);
        this.afterInstancePopulated(beanDefinition, instance);
        return instance;
      },
      retrieveBeans : function(beanDefinitions) {
        Legato.Util.Ensure.ensureObject(beanDefinitions);
        var result = {};
        for ( var name in beanDefinitions) {
          if (beanDefinitions.hasOwnProperty(name)) {
            result[name] = this.retrieveBean(beanDefinitions[name]);
          }
        }
        return result;
      },
      setBeanDefinitions : function(beanDefinitions) {
        this.beanDefinitions = this.explodeBeanDefinitions(beanDefinitions);
        this.beanNames = this.getBeanNames(this.beanDefinitions);
      },
      explodeBeanDefinitions : function(beans) {
        var beanDefinitions = {};
        for ( var name in beans) {
          if (beans.hasOwnProperty(name)) {
            this.explodeBeanDefinition(beanDefinitions, name, beans[name]);
          }
        }
        return beanDefinitions;
      },
      explodeBeanDefinition : function(beanDefinitions, path, beanDefinition) {
        beanDefinitions[path] = beanDefinition;

        var type = beanDefinition.type;
        var definition = beanDefinition.definition;

        for ( var name in definition) {
          if (definition.hasOwnProperty(name)) {
            var value = definition[name];
            if (value instanceof Legato.Beans.BeanDefinition) {
              var newPath = path + '.' + name;
              beanDefinitions[newPath] = value;
              definition[name] = new Legato.Beans.Reference(newPath);
              this.explodeBeanDefinition(beanDefinitions, newPath, value);
            } else if (Legato.Lang.ObjectUtils.isArray(value)) {
              for ( var index = 0; index < value.length; index++) {
                var item = value[index];
                if (item instanceof Legato.Beans.BeanDefinition) {
                  var newItemPath = path + '.' + name + '[' + index + ']';
                  beanDefinitions[newItemPath] = item;
                  value[index] = new Legato.Beans.Reference(newItemPath);
                  this
                      .explodeBeanDefinition(beanDefinitions, newItemPath, item);
                }
              }
            } else if (Legato.Lang.ObjectUtils.isObject(value)) {
              for ( var propertyName in value) {
                if (value.hasOwnProperty(propertyName)) {
                  var propertyValue = value[propertyName];
                  if (propertyValue instanceof Legato.Beans.BeanDefinition) {
                    var newPropertyValuePath = path + '.' + name + '[\''
                        + propertyName + '\']';
                    beanDefinitions[newPropertyValuePath] = propertyValue;
                    value[propertyName] = new Legato.Beans.Reference(
                        newPropertyValuePath);
                    this.explodeBeanDefinition(beanDefinitions,
                        newPropertyValuePath, propertyValue);
                  }
                }
              }
            }
          }
        }
      },
      getBeanNames : function(beanDefinitions) {
        var graph = new Legato.Graph.DirectedGraph();

        var addDependencies = function(graph, name, value) {
          if (value instanceof Legato.Beans.Reference) {
            graph.addEdge(value.target, name);
          } else if (Legato.Lang.ObjectUtils.isArray(value)) {
            for ( var index = 0; index < value.length; index++) {
              addDependencies(graph, name, value[index]);
            }
          } else if (Legato.Lang.ObjectUtils.isObject(value)) {
            for ( var propertyName in value) {
              if (value.hasOwnProperty(propertyName)) {
                addDependencies(graph, name, value[propertyName]);
              }
            }
          }
        };
        
        for ( var name in beanDefinitions) {
          if (beanDefinitions.hasOwnProperty(name)) {
            var vertex = graph.getOrCreateVertex(name);
            var beanDefinition = beanDefinitions[name];
            var type = beanDefinition.type;
            var definition = beanDefinition.definition;

            var constructorOrder = Legato.Lang.ObjectUtils
                .exists(type.constructorOrder) ? type.constructorOrder : [];

            for ( var index = 0; index < constructorOrder.length; index++) {
              addDependencies(graph, name, definition[constructorOrder[index]]);
            }

            var options = Legato.Lang.ObjectUtils.exists(type.options) ? type.options
                : {};

            for ( var optionName in options) {
              if (options.hasOwnProperty(optionName)) {
                addDependencies(graph, name, definition[optionName]);
              }
            }
          }
        }
        return graph.toArray();
      },
      createInstances : function() {
        // Create instances
        for ( var index = 0; index < this.beanNames.length; index++) {
          var name = this.beanNames[index];
          var bean = this.beanDefinitions[name];
          this.putBean(name, this.createInstance(bean));
        }
      },
      createInstance : function(bean) {
        var type = bean.type;
        var definition = bean.definition;
        var constructorArguments = type.constructorArguments;
        var constructorOrder = type.constructorOrder;
        var constructorArgumentInstances = [];
        for ( var index = 0; index < constructorOrder.length; index++) {
          var constructorArgumentName = constructorOrder[index];
          var constructorArgumentType = constructorArguments[constructorArgumentName];
          var constructorArgumentInstance = this.convert(
              constructorArgumentType, definition[constructorArgumentName]);
          constructorArgumentInstances[index] = constructorArgumentInstance;
        }
        if (type.options) {
          var optionsInstance = null;
          for ( var optionName in type.options) {
            if (Legato.Lang.ObjectUtils.exists(definition[optionName])) {
              var optionType = type.options[optionName];
              var optionInstance = this.convert(optionType,
                  definition[optionName]);
              if (optionsInstance === null) {
                optionsInstance = {};
              }
              optionsInstance[optionName] = optionInstance;
            }
          }
          if (optionsInstance !== null) {
            constructorArgumentInstances.push(optionsInstance);
          }
        }
        var instance;
        if (type._constructor) {
          instance = {};
          /*jslint forin:true*/
          for ( var n in type._constructor.prototype) {
            instance[n] = type._constructor.prototype[n];
          }
          type._constructor.apply(instance, constructorArgumentInstances);
        } else if (type.factoryFunction) {
          instance = type.factoryFunction.apply(null,
              constructorArgumentInstances);
        } else {
          throw new Error(
              'Could not create an instance of type [' + type + '].');
        }
        if (Legato.Lang.ObjectUtils.isFunction(instance.setBeanFactory)) {
          instance.setBeanFactory(this);
        }
        return instance;
      },
      populateInstances : function() {
        var names = this.beanNames;
        // Set fields
        for ( var index = 0; index < names.length; index++) {
          var name = names[index];
          var bean = this.beanDefinitions[name];
          var instance = this.instances[name];
          this.populateInstance(bean, instance);
        }
      },
      populateInstance : function(bean, instance) {
        this.setFields(bean, instance);
        this.setProperties(bean, instance);
      },
      setFields : function(bean, instance) {
        var type = bean.type;
        var fields = type.fields;
        for ( var fieldName in fields) {
          if (fields.hasOwnProperty(fieldName)) {
            var fieldType = fields[fieldName];
            var fieldValue = bean.definition[fieldName];
            if (fieldValue) {
              var fieldInstance = this.convert(fieldType, fieldValue);
              instance[fieldName] = fieldInstance;
            }
          }
        }
      },
      setProperties : function(bean, instance) {
        var type = bean.type;
        var properties = type.properties;
        for ( var propertyName in properties) {
          if (properties.hasOwnProperty(propertyName)) {
            var property = properties[propertyName];
            var propertyType = property.type;
            var propertyValue = bean.definition[propertyName];
            if (propertyValue) {
              var propertyInstance = this.convert(propertyType, propertyValue);
              property.set(instance, propertyInstance);
            }
          }
        }
      },
      afterInstancesPopulated : function() {
        var names = this.beanNames;
        // Call afterInstancePopulated function
        for ( var index = 0; index < names.length; index++) {
          var name = names[index];
          var beanDefinition = this.beanDefinitions[name];
          var instance = this.instances[name];
          this.afterInstancePopulated(beanDefinition, instance);
        }
      },
      afterInstancePopulated : function(beanDefinition, instance) {
        var type = beanDefinition.type;
        if (type.afterInstancePopulated) {
          type.afterInstancePopulated(instance, this);
        }
        if (instance.afterInstancePopulated) {
          instance.afterInstancePopulated(this);
        }
      },
      convert : function(type, value) {
        if ((typeof value) == 'undefined') {
          return undefined;
        } else if (value === null) {
          return null;
        } else if (value instanceof Legato.Beans.BeanDefinition) {
          return this.retrieveBean(value);

        } else if ((typeof value) == 'function') {
          return value(this);
        } else if (value instanceof Legato.Beans.Reference) {
          return this.getBean(value.target);
        } else if (((typeof value) == 'object')
            && Legato.Lang.ObjectUtils.isFunction(value.getInstance)) {
          return value.getInstance(this);
        } else if ((typeof value) == 'string') {
          if (type.isInstance(value)) {
            // No conversion required
            return value;
          } else if (this.converters[type.typeName]) {
            var converter = this.converters[type.typeName];
            return converter.convert(value);
          } else if (type.fromString) {
            return type.fromString(value, this);
          } else {
            return value;
            // throw new Error(
            // 'Could not convert the string value ['
            // + value
            // + '] to the type ['
            // + type.typeName
            // + '].'
            // + 'The type does not support string conversion and there is no
            // registered converter for this type.');
          }
        } else if (((typeof value) == 'object') && (type.isList || type.isMap)) {
          var elementType = type.elementType;
          if (type.isList) {
            var listResult = [];

            if (Legato.Lang.ObjectUtils.isArray(value)) {
              for ( var arrayToArrayIndex = 0; arrayToArrayIndex < value.length; arrayToArrayIndex++) {
                listResult[arrayToArrayIndex] = this.convert(elementType,
                    value[arrayToArrayIndex]);
              }
            } else {
              var objectToArrayIndex = 0;
              for ( var objectToArrayPropertyName in value) {
                if (value.hasOwnProperty(objectToArrayPropertyName)) {
                  listResult[objectToArrayIndex] = this.convert(elementType,
                      value[objectToArrayPropertyName]);
                  objectToArrayIndex++;
                }
              }
            }

            return listResult;
          } else {
            var objectResult = {};

            if (Legato.Lang.ObjectUtils.isArray(value)) {
              for ( var arrayToObjectIndex = 0; arrayToObjectIndex < value.length; arrayToObjectIndex++) {
                objectResult[arrayToObjectIndex] = this.convert(elementType,
                    value[arrayToObjectIndex]);
              }
            } else {
              for ( var objectToObjectPropertyName in value) {
                if (value.hasOwnProperty(objectToObjectPropertyName)) {
                  objectResult[objectToObjectPropertyName] = this.convert(
                      elementType, value[objectToObjectPropertyName]);
                }
              }
            }
            return objectResult;
          }
        } else if (type.isInstance && type.isInstance(value)) {
          // No conversion required
          return value;
        } else {
          throw new Error('Conversion of [' + value + '] to [' + type.typeName
              + '] is not implemented.');
        }
      },
      /*
       * Function: getBean
       * 
       * Retrieves bean instance for the given name.
       * 
       * Parameters:
       * 
       * name - {String} name of the bean.
       */
      getBean : function(name) {
        Legato.Util.Ensure.ensureNotEmptyString(name);

        if (name.charAt(0) == '&') {
          var realName = name.substring(1);
          var factoryBean = this.instances[realName];
          if (Legato.Lang.ObjectUtils.exists(factoryBean)) {
            return factoryBean;
          } else {
            throw new Legato.Lang.Exception(
                'Bean [' + realName + '] could not be found.');
          }
        } else {
          var bean = this.instances[name];
          if (!Legato.Lang.ObjectUtils.exists(bean)) {
            throw new Legato.Lang.Exception(
                'Bean [' + name + '] could not be found.');
          } else if (Legato.Lang.ObjectUtils.isFunction(bean.getInstance)) {
            return bean.getInstance(this);
          } else {
            return bean;
          }
        }
      },
      /*
       * Function: putBean
       * 
       * Puts bean instance to the bean factory.
       * 
       * Parameters:
       * 
       * name - {String} name of the bean. instance - bean to be added to the
       * bean factory.
       */
      putBean : function(name, instance) {
        Legato.Util.Ensure.ensureString(name);
        Legato.Util.Ensure.ensureExists(instance);
        this.instances[name] = instance;
      },
      destroy : function() {
        // if unloadDestroy is null, we've already been destroyed
      if (!this.unloadDestroy) {
        return false;
      }

      // map has been destroyed. dont do it again!
      OpenLayers.Event.stopObserving(window, 'unload', this.unloadDestroy);
      this.unloadDestroy = null;

      for ( var instanceName in this.instances) {
        if (this.instances.hasOwnProperty(instanceName)) {
          var instance = this.instances[instanceName];
          // if (Legato.Lang.ObjectUtils.isFunction(instance.destroy)) {
          // instance.destroy();
          // }
          instance = null;
        }
      }
      this.instances = null;
      for ( var beanDefinitionName in this.beanDefinitions) {
        if (this.beanDefinitions.hasOwnProperty(beanDefinitionName)) {
          var beanDefinition = this.beanDefinitions[beanDefinitionName];
          if (Legato.Lang.ObjectUtils.isFunction(beanDefinition.destroy)) {
            beanDefinition.destroy();
          }
          beanDefinition = null;
        }
      }
      this.beanDefinitions = null;
      this.beanNames = null;
    },
    CLASS_NAME :'Legato.Beans.XMLBeanFactory'
    });