/*
 * Legato is a configurable, lightweight web mapping client that can be
 * easily embedded into web pages and portals, CMS and individual web
 * applications. Legato is implemented in JavaScript and based on the
 * popular open source library OpenLayers.
 *
 * Copyright (C) 2010  disy Informationssysteme GmbH, http://www.disy.net
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

/*
 * @author $Author: valikov $
 * @version $Rev: 70574 $
 */

/*
 * Namespace: Legato.Util.Ensure
 *
 * Implements ensure utilities.
 */
Legato.Util.Ensure = {
  /*
   * Function: ensureTrue
   *
   * Checks if given argument is the boolean value "true"., throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureTrue : function(value, message) {
    if (value !== true) {
      throw new Error(message);
    }
  },
  
  /*
   * Function: ensureFalse
   *
   * Checks if given argument is the boolean value "false"., throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureFalse : function(value, message) {
    if (value !== false) {
      throw new Error(message);
    }
  },
  /*
   * Function: ensureExists
   *
   * Checks if given argument exists (not null, not undefined), throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureExists : function(value, message) {
    if (!Legato.Lang.ObjectUtils.exists(value)) {
      throw new Error(message);
    }
  },
  /*
   * Function: ensureString
   *
   * Checks if given argument is string, throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureString : function(value, message) {
    if (typeof value != 'string') {
      throw new Legato.Lang.IllegalArgumentException(message);
    }
  },
  /*
   * Function: ensureNumber
   *
   * Checks if given argument is number, throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   *  Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureNumber : function(value, message) {
    if (typeof value != 'number') {
      throw new Legato.Lang.IllegalArgumentException(message);
    }
  },
   /*
   * Function: ensureNotEmptyString
   *
   * Checks if given argument is a non-empty string, throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureNotEmptyString : function(value, message) {
    if (typeof value != 'string') {
      throw new Legato.Lang.IllegalArgumentException(message);
    } else {
      if (Legato.Lang.StringUtils.isEmpty(value)) {
        throw new Legato.Lang.IllegalArgumentException(message);
      }
    }
  },
  /*
   * Function: ensureObject
   *
   * Checks if given argument is an object (not null, not undefined), throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureObject : function(value, message) {
    if (typeof value != 'object' || value === null) {
      throw new Legato.Lang.IllegalArgumentException(message);
    }
  },
  /*
   * Function: ensureArray
   *
   * Checks if given argument is an array, throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureArray : function(value, message) {
    if (!(value instanceof Array)) {
      throw new Legato.Lang.IllegalArgumentException(message);
    }
  },
  /*
   * Function: ensureFunction
   *
   * Checks if given argument is a function, throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   *
   * message - {String} exception message (optional).
   */
  ensureFunction : function(value, message) {
    if (typeof value != 'function') {
      throw new Legato.Lang.IllegalArgumentException(message);
    }
  },
  /*
   * Function: ensureInstanceOf
   *
   * Checks if given value is an instance of the given function, throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   * f - {Function} potential constructor function.
   *
   * message - {String} exception message (optional).
   */
  ensureInstanceOf : function(value, f, message) {
    Legato.Util.Ensure.ensureFunction(f);
    if (!(value instanceof f))
    {
      throw new Legato.Lang.IllegalArgumentException(message);
    }
  },
  /*
   * Function: ensureImplements
   *
   * Checks if given value implements a function with the given name, throws {<Legato.Lang.IllegalArgumentException>}
   * otherwise.
   *
   * Parameters:
   *
   * value - value to check.
   * functionName - {String} name of the function.
   *
   * message - {String} exception message (optional).
   */
  ensureImplements : function(value, functionName, message) {
    Legato.Util.Ensure.ensureExists(value, message);
    Legato.Util.Ensure.ensureString(functionName, message);
    Legato.Util.Ensure.ensureFunction(value[functionName], message);
  }

};