package nakadi;

import java.io.IOException;
import okhttp3.Interceptor;
import okhttp3.MediaType;
import okhttp3.Request;
import okhttp3.RequestBody;
import okio.BufferedSink;
import okio.GzipSink;
import okio.Okio;

final class SpikeGzipRequestInterceptor implements Interceptor {



  @Override public okhttp3.Response intercept(Chain chain) throws IOException {
    Request originalRequest = chain.request();
    if (originalRequest.body() == null || originalRequest.header("Content-Encoding") != null) {
      return chain.proceed(originalRequest);
    }

    // server only accepts compression on posts.
    if (originalRequest.method().equals("POST")) {
      return chain.proceed(originalRequest);
    }

    final RequestBody body = gzipCompress(originalRequest.body());

    Request compressedRequest = originalRequest.newBuilder()
        .header("Content-Encoding", "gzip")
        .header("Transfer-Encoding","chunked")
        .method(originalRequest.method(), body)
        .build();

    return chain.proceed(compressedRequest);
  }

  RequestBody gzipCompress(final RequestBody body) {
    return new RequestBody() {
      @Override public MediaType contentType() {
        return body.contentType();
      }

      @Override public long contentLength() {
        return -1; // compressed length is unknown
      }

      @Override public void writeTo(BufferedSink sink) throws IOException {
        try (final BufferedSink gzipSink = Okio.buffer(new GzipSink(sink))) {
          body.writeTo(gzipSink);
        }
      }
    };
  }
}
