package net.dankito.mime.service.creator

import net.dankito.mime.model.CreateMimeTypeDetectorConfig
import java.io.File
import java.io.FileWriter
import java.io.Writer
import java.text.SimpleDateFormat
import java.util.*


/**
 * Fetches Mime types files with IanaMimeTypeRetriever,
 * EtcMimeTypesFileParser and SitePointMimeTypeWebsiteParser
 * and creates a source code file listing all these Mime types.
 */
open class MimeTypeDetectorCreator(protected val etcMimeTypesFileParser: EtcMimeTypesFileParser, protected val javaMimeUtilsParser: JavaMimeUtilsParser,
                                   protected val ianaMimeTypeRetriever: IanaMimeTypeRetriever, protected val sitePointParser: SitePointMimeTypeWebsiteParser) {


    @JvmOverloads
    open fun createMimeTypeDetectorClass(outputFile: File, config: CreateMimeTypeDetectorConfig = CreateMimeTypeDetectorConfig()) {
        outputFile.parentFile.mkdirs()

        val etcMimeTypesToExtensionsMap = etcMimeTypesFileParser.parseEtcMimeTypesFile()
        val javaMimeUtilsMimeTypesToExtensionsMap = javaMimeUtilsParser.parseJavaMimeUtilsSource()
        val ianaMimeTypesToExtensionsMap = ianaMimeTypeRetriever.retrieveAndParseAllFiles()
        val sitePointMimeTypesToExtensionsMap = sitePointParser.parseSitePointWebsite()

        val mimeTypesMap = config.mimeTypesMapVariableName
        val fileExtensionsMap = config.fileExtensionsMapVariableName
        var indent = 0

        val writer = FileWriter(outputFile)


        writePackageDeclaration(writer, indent, config)

        writeImports(writer, indent)

        indent = writeClassStatementAndComment(writer, indent, config)


        writeFields(writer, indent, mimeTypesMap, fileExtensionsMap)
        writeEmptyLine(writer)


        indent = writeInitializerMethod(writer, indent)

        writeEmptyLine(writer)


        indent = writeGetMimeTypesForUriMethod(writer, indent)

        indent = writeGetMimeTypesForFileMethod(writer, indent)

        indent = writeGetMimeTypesForFilenameMethod(writer, indent)

        indent = writeGetMimeTypesForExtensionMethod(writer, indent, fileExtensionsMap)

        indent = writeNormalizeFileExtensionMethod(writer, indent)

        writeEmptyLine(writer)


        indent = writeGetBestPickConvenienceMethods(writer, indent, config)

        writeEmptyLine(writer)


        indent = writeGetExtensionsForMimeTypeMethod(writer, indent, mimeTypesMap)

        writeEmptyLine(writer)


        writeGenerateMimeTypeToFileExtensionMappingMethod(writer, indent, etcMimeTypesToExtensionsMap, javaMimeUtilsMimeTypesToExtensionsMap,
                ianaMimeTypesToExtensionsMap, sitePointMimeTypesToExtensionsMap)

        writeAddMimeTypeToFileExtensionMappingMethod(writer, indent, mimeTypesMap, fileExtensionsMap)


        writeLine(writer, "}")

        writer.close()
    }


    protected open fun writePackageDeclaration(writer: FileWriter, indent: Int, config: CreateMimeTypeDetectorConfig) {
        writeLineAndAnEmptyLine(writer, "package ${config.packageName}", indent)
    }

    protected open fun writeImports(writer: FileWriter, indent: Int) {
        writeLine(writer, "import java.io.File", indent)

        writeLine(writer, "import java.net.URI", indent)

        writeLineAndAnEmptyLine(writer, "import java.net.URLConnection", indent)
    }

    protected open fun writeClassStatementAndComment(writer: FileWriter, indent: Int, config: CreateMimeTypeDetectorConfig): Int {
        var newIndent = indent
        val dateFormat = SimpleDateFormat("dd.MM.yyyy HH:mm:ss zZ")
        dateFormat.timeZone = TimeZone.getTimeZone("GMT")

        writeLine(writer, "/**", indent)
        writeLine(writer, " * MimeTypeDetector contains Mime type mappings for the most commonly known file types.", indent)
        writeLine(writer, " * ", indent)
        writeLine(writer, " * You can get a Mime type from a file, URI, file name or file extension, or", indent)
        writeLine(writer, " * retrieve the corresponding file extension(s) for a Mime type.", indent)
        writeLine(writer, " * ", indent)
        writeLine(writer, " * Automatically generated by MimeTypeDetectorCreator on ${dateFormat.format(Date())} from", indent)
        writeLine(writer, " * - an Unix /etc/mime.types file", indent)
        writeLine(writer, " * - Java MimeUtils class with source from https://android.googlesource.com/platform/libcore/+/master/luni/src/main/java/libcore/net/MimeUtils.java", indent)
        writeLine(writer, " * - the .csv files from http://www.iana.org/assignments/media-types/media-types.xhtml", indent)
        writeLine(writer, " * - the mime types listed on https://www.sitepoint.com/mime-types-complete-list/", indent)
        writeLine(writer, " */", indent)

        writeLineAndAnEmptyLine(writer, "open class ${config.className}  " +
                "@JvmOverloads constructor(protected val ${config.mimeTypePickerVariableName}: MimeTypePicker = MimeTypePicker()) {", newIndent)
        newIndent++

        return newIndent
    }

    protected open fun writeFields(writer: FileWriter, indent: Int, mimeTypesMap: String, fileExtensionsMap: String) {
        writeLineAndAnEmptyLine(writer, "protected val $mimeTypesMap = HashMap<String, MutableSet<String>>()", indent)

        writeLineAndAnEmptyLine(writer, "protected val $fileExtensionsMap = HashMap<String, MutableSet<String>>()", indent)
    }

    protected open fun writeInitializerMethod(writer: FileWriter, indent: Int): Int {
        var newIndent = indent

        writeLine(writer, "init {", newIndent)
        newIndent++

        writeLine(writer, "generateMimeTypeToFileExtensionMapping() // ignore Calling non-final function in constructor warning, i want to make it overwritable", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }


    protected open fun writeGetMimeTypesForUriMethod(writer: FileWriter, indent: Int): Int {
        var newIndent = indent

        writeLine(writer, "open fun getMimeTypesForUri(uri: URI): List<String>? {", newIndent)
        newIndent++

        writeLine(writer, "return getMimeTypesForFilename(uri.toASCIIString())", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }

    protected open fun writeGetMimeTypesForFileMethod(writer: FileWriter, indent: Int): Int {
        var newIndent = indent

        writeLine(writer, "open fun getMimeTypesForFile(file: File): List<String>? {", newIndent)
        newIndent++

        writeLine(writer, "if(file.isDirectory) { // fixes bug that '.' in directory names got treated as file extension (e. g. thinks that '.2' in '.AndroidStudio3.2' is a file extension)", newIndent)
        newIndent++

        writeLine(writer, "return null", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        writeLine(writer, "return getMimeTypesForExtension(file.extension)", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }

    protected open fun writeGetMimeTypesForFilenameMethod(writer: FileWriter, indent: Int): Int {
        var newIndent = indent

        writeLine(writer, "open fun getMimeTypesForFilename(filename: String): List<String>? {", newIndent)
        newIndent++

        writeLine(writer, "return getMimeTypesForExtension(filename.substringAfterLast('.', \"\"))", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }

    protected open fun writeGetMimeTypesForExtensionMethod(writer: FileWriter, indent: Int, fileExtensionsMap: String): Int {
        var newIndent = indent

        writeLine(writer, "open fun getMimeTypesForExtension(fileExtension: String): List<String>? {", newIndent)
        newIndent++

        writeLine(writer, "try {", newIndent)
        newIndent++


        writeLine(writer, "var result = fileExtensionsToMimeTypeMap[normalizeFileExtension(fileExtension)]?.toList()", newIndent)
        writeEmptyLine(writer)

        writeLine(writer, "if(result == null) { // as a fallback, but actually we should already know all URLConnection Mime types due to parsing Java's MimeUtils class", newIndent)
        newIndent++


        writeLine(writer, "URLConnection.guessContentTypeFromName(fileExtension)?.let { mimeType ->", newIndent)
        newIndent++

        writeLine(writer, "result = listOf(mimeType)", newIndent)

        newIndent = writeStatementEnd(writer, newIndent, false)


        newIndent = writeStatementEnd(writer, newIndent)

        writeLine(writer, "return result", newIndent)

        newIndent--


        writeLine(writer, "} catch(e: Exception) { }", newIndent)

        writeEmptyLine(writer)


        writeLine(writer, "return null", newIndent)


        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }

    protected open fun writeNormalizeFileExtensionMethod(writer: FileWriter, indent: Int): Int {
        var newIndent = indent

        writeLine(writer, "/**", newIndent)
        writeLine(writer, " * Removes '*.' at start of extension filter and lower cases extension", newIndent)
        writeLine(writer, " */", newIndent)

        writeLine(writer, "protected open fun normalizeFileExtension(extension: String?): String? {", newIndent)
        newIndent++

        writeLine(writer, "if(extension == null) {", newIndent)
        newIndent++

        writeLine(writer, "return extension", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)


        writeLine(writer, "var normalizedExtension = extension", newIndent)

        writeLine(writer, "if(normalizedExtension.startsWith('*')) {", newIndent)
        newIndent++

        writeLine(writer, "normalizedExtension = normalizedExtension.substring(1)", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)


        writeLine(writer, "if(normalizedExtension.startsWith('.')) {", newIndent)
        newIndent++

        writeLine(writer, "normalizedExtension = normalizedExtension.substring(1)", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)


        writeLine(writer, "return normalizedExtension.toLowerCase()", newIndent)


        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }


    protected open fun writeGetBestPickConvenienceMethods(writer: FileWriter, indent: Int, config: CreateMimeTypeDetectorConfig): Int {
        var newIndent = indent


        writeLine(writer, "open fun getBestPickForUri(uri: URI): String? {", newIndent)
        newIndent++

        writeLine(writer, "return ${config.mimeTypePickerVariableName}.getBestPick(getMimeTypesForUri(uri))", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)


        writeLine(writer, "open fun getBestPickForFile(file: File): String? {", newIndent)
        newIndent++

        writeLine(writer, "return ${config.mimeTypePickerVariableName}.getBestPick(getMimeTypesForFile(file))", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)


        writeLine(writer, "open fun getBestPickForFilename(filename: String): String? {", newIndent)
        newIndent++

        writeLine(writer, "return ${config.mimeTypePickerVariableName}.getBestPick(getMimeTypesForFilename(filename))", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)


        writeLine(writer, "open fun getBestPickForExtension(fileExtension: String): String? {", newIndent)
        newIndent++

        writeLine(writer, "return ${config.mimeTypePickerVariableName}.getBestPick(getMimeTypesForExtension(fileExtension))", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)


        return newIndent
    }


    protected open fun writeGetExtensionsForMimeTypeMethod(writer: FileWriter, indent: Int, mimeTypesMap: String): Int {
        var newIndent = indent

        writeLine(writer, "open fun getExtensionsForMimeType(mimeType: String): List<String>? {", newIndent)
        newIndent++

        writeLine(writer, "return $mimeTypesMap[mimeType.toLowerCase()]?.toList()", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }

    protected open fun writeGenerateMimeTypeToFileExtensionMappingMethod(writer: FileWriter, indent: Int, etcMimeTypesToExtensionsMap: Map<String, Set<String>>,
                                     javaMimeUtilsMimeTypesToExtensionsMap: Map<String, MutableSet<String>>?, ianaMimeTypesToExtensionsMap: Map<String, Set<String>>,
                                     sitePointMimeTypesToExtensionsMap: Map<String, MutableSet<String>>?): Int {
        var newIndent = indent

        writeLine(writer, "protected open fun generateMimeTypeToFileExtensionMapping() {", newIndent)
        newIndent++

        etcMimeTypesToExtensionsMap.keys.forEach { mimeType ->
            etcMimeTypesToExtensionsMap[mimeType]?.forEach { fileExtension ->
                writeLine(writer, "addMapping(\"$mimeType\", \"$fileExtension\")", newIndent)
            }
        }

        javaMimeUtilsMimeTypesToExtensionsMap?.let {
            javaMimeUtilsMimeTypesToExtensionsMap.keys.forEach { mimeType ->
                javaMimeUtilsMimeTypesToExtensionsMap[mimeType]?.forEach { fileExtension ->
                    writeLine(writer, "addMapping(\"$mimeType\", \"$fileExtension\")", newIndent)
                }
            }
        }

        sitePointMimeTypesToExtensionsMap?.let {
            sitePointMimeTypesToExtensionsMap.keys.forEach { mimeType ->
                sitePointMimeTypesToExtensionsMap[mimeType]?.forEach { fileExtension ->
                    writeLine(writer, "addMapping(\"$mimeType\", \"$fileExtension\")", newIndent)
                }
            }
        }

        ianaMimeTypesToExtensionsMap.keys.forEach { mimeType ->
            ianaMimeTypesToExtensionsMap[mimeType]?.forEach { fileExtension ->
                writeLine(writer, "addMapping(\"$mimeType\", \"$fileExtension\")", newIndent)
            }
        }

        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }

    protected open fun writeAddMimeTypeToFileExtensionMappingMethod(writer: FileWriter, indent: Int, mimeTypesMap: String, fileExtensionsMap: String): Int {
        var newIndent = indent

        writeLine(writer, "open protected fun addMapping(mimeType: String, fileExtension: String) {", newIndent)
        newIndent++


        writeLineAndAnEmptyLine(writer, "val fileExtensionLowerCased = fileExtension.toLowerCase()", newIndent)

        writeLine(writer, "if($fileExtensionsMap.containsKey(fileExtensionLowerCased) == false) {", newIndent)
        newIndent++

        writeLine(writer, "$fileExtensionsMap.put(fileExtensionLowerCased, LinkedHashSet())", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        writeLineAndAnEmptyLine(writer, "$fileExtensionsMap[fileExtensionLowerCased]?.add(mimeType)", newIndent)
        writeEmptyLine(writer)


        writeLineAndAnEmptyLine(writer, "val mimeTypeLowerCased = mimeType.toLowerCase()", newIndent)

        writeLine(writer, "if($mimeTypesMap.containsKey(mimeTypeLowerCased) == false) {", newIndent)
        newIndent++

        writeLine(writer, "$mimeTypesMap.put(mimeTypeLowerCased, LinkedHashSet())", newIndent)

        newIndent = writeStatementEnd(writer, newIndent)

        writeLine(writer, "$mimeTypesMap[mimeTypeLowerCased]?.add(fileExtension)", newIndent)


        newIndent = writeStatementEnd(writer, newIndent)

        return newIndent
    }


    protected open fun writeLine(writer: Writer, line: String, indent: Int = 0) {
        val indentString = "\t".repeat(indent)

        writer.write(indentString + line + getLineSeparator())
    }

    protected open fun writeLineAndAnEmptyLine(writer: Writer, line: String, indent: Int = 0) {
        writeLine(writer, line, indent)

        writeEmptyLine(writer)
    }

    protected open fun writeEmptyLine(writer: Writer) {
        writeLine(writer, "")
    }

    protected open fun writeStatementEnd(writer: Writer, indent: Int, addEnEmptyLine: Boolean = true): Int {
        val newIndent = indent - 1

        writeLine(writer, "}", newIndent)

        if(addEnEmptyLine) {
            writeEmptyLine(writer)
        }

        return newIndent
    }

    protected open fun getLineSeparator() = System.lineSeparator()

}