package net.dankito.mime

import java.io.File
import java.net.URI
import java.net.URLConnection

/**
 * MimeTypeDetector contains Mime type mappings for the most commonly known file types.
 *
 * You can get a Mime type from a file, URI, file name or file extension, or
 * retrieve the corresponding file extension(s) for a Mime type.
 *
 * Automatically generated by MimeTypeDetectorCreator on 14.02.2018 15:26:57 GMT+0000 from
 * - an Unix /etc/mime.types file
 * - Java MimeUtils class with source from https://android.googlesource.com/platform/libcore/+/master/luni/src/main/java/libcore/net/MimeUtils.java
 * - the .csv files from http://www.iana.org/assignments/media-types/media-types.xhtml
 * - the mime types listed on https://www.sitepoint.com/mime-types-complete-list/
 */
open class MimeTypeDetector  @JvmOverloads constructor(protected val picker: MimeTypePicker = MimeTypePicker()) {

    protected val mimeTypesMap = HashMap<String, MutableSet<String>>()

    protected val fileExtensionsToMimeTypeMap = HashMap<String, MutableSet<String>>()


    init {
        generateMimeTypeToFileExtensionMapping() // ignore Calling non-final function in constructor warning, i want to make it overwritable
    }


    open fun getMimeTypesForUri(uri: URI): List<String>? {
        return getMimeTypesForFilename(uri.toASCIIString())
    }

    open fun getMimeTypesForFile(file: File): List<String>? {
        return getMimeTypesForExtension(file.extension)
    }

    open fun getMimeTypesForFilename(filename: String): List<String>? {
        return getMimeTypesForExtension(filename.substringAfterLast('.', ""))
    }

    open fun getMimeTypesForExtension(fileExtension: String): List<String>? {
        try {
            var result = fileExtensionsToMimeTypeMap[normalizeFileExtension(fileExtension)]?.toList()

            if(result == null) { // as a fallback, but actually we should already know all URLConnection Mime types due to parsing Java's MimeUtils class
                URLConnection.guessContentTypeFromName(fileExtension)?.let { mimeType ->
                    result = listOf(mimeType)
                }
            }

            return result
        } catch(e: Exception) { }

        return null
    }

    /**
     * Removes '*.' at start of extension filter and lower cases extension
     */
    open protected fun normalizeFileExtension(extension: String?): String? {
        if(extension == null) {
            return extension
        }

        var normalizedExtension = extension
        if(normalizedExtension.startsWith('*')) {
            normalizedExtension = normalizedExtension.substring(1)
        }

        if(normalizedExtension.startsWith('.')) {
            normalizedExtension = normalizedExtension.substring(1)
        }

        return normalizedExtension.toLowerCase()
    }


    open fun getBestPickForUri(uri: URI): String? {
        return picker.getBestPick(getMimeTypesForUri(uri))
    }

    open fun getBestPickForFile(file: File): String? {
        return picker.getBestPick(getMimeTypesForFile(file))
    }

    open fun getBestPickForFilename(filename: String): String? {
        return picker.getBestPick(getMimeTypesForFilename(filename))
    }

    open fun getBestPickForExtension(fileExtension: String): String? {
        return picker.getBestPick(getMimeTypesForExtension(fileExtension))
    }


    open fun getExtensionsForMimeType(mimeType: String): List<String>? {
        return mimeTypesMap[mimeType.toLowerCase()]?.toList()
    }


    open protected fun generateMimeTypeToFileExtensionMapping() {
        addMapping("application/vnd.ms-htmlhelp", "chm")
        addMapping("application/atom+xml", "atom")
        addMapping("application/vnd.3gpp.pic-bw-var", "pvb")
        addMapping("application/clue_info+xml", "clue")
        addMapping("application/vnd.kidspiration", "kia")
        addMapping("application/vnd.ufdl", "ufdl")
        addMapping("application/vnd.ufdl", "ufd")
        addMapping("application/vnd.ufdl", "frm")
        addMapping("video/vnd.dece.mp4", "uvu")
        addMapping("video/vnd.dece.mp4", "uvvu")
        addMapping("audio/dls", "dls")
        addMapping("audio/x-flac", "flac")
        addMapping("application/font-tdpfr", "pfr")
        addMapping("application/vnd.crick.clicker.wordbank", "clkw")
        addMapping("application/tamp-apex-update", "tau")
        addMapping("application/vnd.geocube+xml", "g3")
        addMapping("application/vnd.geocube+xml", "g³")
        addMapping("application/vnd.kenameaapp", "htke")
        addMapping("text/xml", "xml")
        addMapping("text/xml", "xsd")
        addMapping("text/xml", "rng")
        addMapping("application/rss+xml", "rss")
        addMapping("application/x-tcl", "tcl")
        addMapping("application/vnd.fujitsu.oasys", "oas")
        addMapping("audio/EVRCB", "evb")
        addMapping("application/vnd.visionary", "vis")
        addMapping("multipart/vnd.bint.med-plus", "bmed")
        addMapping("application/prs.rdf-xml-crypt", "rdf-crypt")
        addMapping("application/xcap-diff+xml", "xdf")
        addMapping("text/html", "html")
        addMapping("text/html", "htm")
        addMapping("application/vnd.lotus-freelance", "prz")
        addMapping("application/vnd.lotus-freelance", "pre")
        addMapping("application/vnd.ms-artgalry", "cil")
        addMapping("application/vnd.geometry-explorer", "gex")
        addMapping("application/vnd.geometry-explorer", "gre")
        addMapping("application/vnd.genomatix.tuxedo", "txd")
        addMapping("audio/x-ms-wma", "wma")
        addMapping("application/vnd.commonspace", "csp")
        addMapping("application/vnd.commonspace", "cst")
        addMapping("application/vnd.fujixerox.docuworks.binder", "xbd")
        addMapping("application/vnd.amazon.mobi8-ebook", "azw3")
        addMapping("application/scvp-cv-request", "scq")
        addMapping("application/x-wais-source", "src")
        addMapping("application/vnd.accpac.simply.imp", "imp")
        addMapping("application/vnd.lotus-wordpro", "lwp")
        addMapping("application/vnd.lotus-wordpro", "sam")
        addMapping("application/vnd.mophun.application", "mpn")
        addMapping("application/vnd.osgi.dp", "dp")
        addMapping("text/html-sandboxed", "sandboxed")
        addMapping("application/provenance+xml", "provx")
        addMapping("application/vnd.citationstyles.style+xml", "csl")
        addMapping("application/vnd.neurolanguage.nlu", "nlu")
        addMapping("application/vnd.wap.mms-message", "mms")
        addMapping("application/mac-compactpro", "cpt")
        addMapping("application/urc-uisocketdesc+xml", "uis")
        addMapping("application/vnd.hbci", "hbci")
        addMapping("application/vnd.hbci", "hbc")
        addMapping("application/vnd.hbci", "kom")
        addMapping("application/vnd.hbci", "upa")
        addMapping("application/vnd.hbci", "pkd")
        addMapping("application/vnd.hbci", "bpd")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.template", "xltx")
        addMapping("application/vnd.tcpdump.pcap", "pcap")
        addMapping("application/vnd.tcpdump.pcap", "cap")
        addMapping("application/vnd.tcpdump.pcap", "dmp")
        addMapping("application/x-tex", "tex")
        addMapping("application/vnd.multiad.creator.cif", "cif")
        addMapping("application/vnd.picsel", "efif")
        addMapping("application/tamp-sequence-adjust-confirm", "sac")
        addMapping("application/vnd.ms-cab-compressed", "cab")
        addMapping("application/tamp-update-confirm", "tuc")
        addMapping("audio/vnd.sealedmedia.softseal.mpeg", "smp3")
        addMapping("audio/vnd.sealedmedia.softseal.mpeg", "smp")
        addMapping("audio/vnd.sealedmedia.softseal.mpeg", "s1m")
        addMapping("application/n-quads", "nq")
        addMapping("application/vnd.software602.filler.form+xml", "fo")
        addMapping("application/vnd.anki", "apkg")
        addMapping("application/vnd.rainstor.data", "tree")
        addMapping("x-conference/x-cooltalk", "ice")
        addMapping("application/vnd.Mobius.DAF", "daf")
        addMapping("application/A2L", "a2l")
        addMapping("image/vnd.sealed.png", "spng")
        addMapping("image/vnd.sealed.png", "spn")
        addMapping("image/vnd.sealed.png", "s1n")
        addMapping("application/vnd.coreos.ignition+json", "ign")
        addMapping("application/vnd.coreos.ignition+json", "ignition")
        addMapping("image/webp", "webp")
        addMapping("audio/x-realaudio", "ra")
        addMapping("application/vnd.fdf", "fdf")
        addMapping("application/vnd.apple.mpegurl", "m3u8")
        addMapping("application/vnd.rig.cryptonote", "cryptonote")
        addMapping("application/x-killustrator", "kil")
        addMapping("application/vnd.nokia.radio-preset", "rpst")
        addMapping("application/vnd.syncml+xml", "xsm")
        addMapping("audio/x-aiff", "aif")
        addMapping("audio/x-aiff", "aiff")
        addMapping("audio/x-aiff", "aifc")
        addMapping("image/vnd.fpx", "fpx")
        addMapping("application/pkix-cert", "cer")
        addMapping("application/vnd.framemaker", "fm")
        addMapping("image/vnd.dece.graphic", "uvi")
        addMapping("image/vnd.dece.graphic", "uvvi")
        addMapping("image/vnd.dece.graphic", "uvg")
        addMapping("image/vnd.dece.graphic", "uvvg")
        addMapping("application/vnd.oasis.opendocument.image-template", "oti")
        addMapping("application/rtf", "rtf")
        addMapping("application/vnd.aristanetworks.swi", "swi")
        addMapping("image/t38", "t38")
        addMapping("application/vnd.imagemeter.folder+zip", "imf")
        addMapping("image/x-portable-pixmap", "ppm")
        addMapping("application/vnd.groove-help", "ghf")
        addMapping("application/vnd.vidsoft.vidconference", "vsc")
        addMapping("application/vnd.syncml.dm+wbxml", "bdm")
        addMapping("application/vnd.pg.osasli", "ei6")
        addMapping("application/vnd.ahead.space", "ahead")
        addMapping("application/vnd.unity", "unityweb")
        addMapping("application/vnd.panoply", "plp")
        addMapping("audio/AMR", "amr")
        addMapping("text/uri-list", "uris")
        addMapping("text/uri-list", "uri")
        addMapping("application/vnd.evolv.ecig.settings", "ecig")
        addMapping("application/vnd.adobe.formscentral.fcdt", "fcdt")
        addMapping("text/vnd.abc", "abc")
        addMapping("application/ccxml+xml", "ccxml")
        addMapping("application/cpl+xml", "cpl")
        addMapping("application/vnd.wolfram.mathematica.package", "m")
        addMapping("application/vnd.eszigno3+xml", "es3")
        addMapping("application/vnd.eszigno3+xml", "et3")
        addMapping("image/png", "png")
        addMapping("application/cccex", "c3ex")
        addMapping("application/vnd.onepagertamx", "tamx")
        addMapping("application/ATFX", "atfx")
        addMapping("application/vnd.musician", "mus")
        addMapping("application/lgr+xml", "lgr")
        addMapping("application/vnd.yamaha.smaf-audio", "saf")
        addMapping("application/vnd.ms-xpsdocument", "xps")
        addMapping("application/vnd.anser-web-certificate-issue-initiation", "cii")
        addMapping("application/vnd.onepagertamp", "tamp")
        addMapping("application/vnd.sealed.3df", "s3df")
        addMapping("audio/ogg", "oga")
        addMapping("audio/ogg", "ogg")
        addMapping("audio/ogg", "opus")
        addMapping("audio/ogg", "spx")
        addMapping("audio/32kadpcm", "726")
        addMapping("application/x-tar", "tar")
        addMapping("application/vnd.smart.notebook", "notebook")
        addMapping("application/vnd.ms-asf", "asf")
        addMapping("application/vnd.stepmania.package", "smzip")
        addMapping("application/vnd.medcalcdata", "mc1")
        addMapping("application/xcap-el+xml", "xel")
        addMapping("text/vnd.wap.sl", "sl")
        addMapping("application/vnd.hdt", "hdt")
        addMapping("text/x-pod", "pod")
        addMapping("text/vnd.wap.si", "si")
        addMapping("application/vnd.pg.format", "str")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.template", "dotx")
        addMapping("application/vnd.piaccess.application-license", "pil")
        addMapping("text/sgml", "sgml")
        addMapping("text/sgml", "sgm")
        addMapping("application/vnd.adobe.flash.movie", "swf")
        addMapping("application/vnd.noblenet-sealer", "nns")
        addMapping("application/vnd.shana.informed.formdata", "ifm")
        addMapping("application/vnd.trid.tpt", "tpt")
        addMapping("application/vnd.blueice.multipass", "mpm")
        addMapping("application/vnd.contact.cmsg", "cdbcmsg")
        addMapping("application/vnd.epson.quickanime", "qam")
        addMapping("application/vnd.ms-excel.sheet.binary.macroEnabled.12", "xlsb")
        addMapping("application/ocsp-request", "orq")
        addMapping("application/vnd.hp-hps", "hps")
        addMapping("audio/vnd.rip", "rip")
        addMapping("application/vnd.dreamfactory", "dfac")
        addMapping("application/vnd.ieee.1905", "1905.1")
        addMapping("application/epub+zip", "epub")
        addMapping("text/vnd.a", "a")
        addMapping("text/vnd.si.uricatalogue", "uric")
        addMapping("application/reginfo+xml", "rif")
        addMapping("application/vnd.crick.clicker", "clkx")
        addMapping("application/vnd.symbian.install", "sis")
        addMapping("application/exi", "exi")
        addMapping("application/marc", "mrc")
        addMapping("application/vnd.mediastation.cdkey", "cdkey")
        addMapping("application/vnd.oma.dd+xml", "dd")
        addMapping("application/vnd.3gpp2.tcap", "tcap")
        addMapping("image/vnd.sealedmedia.softseal.jpg", "sjpg")
        addMapping("image/vnd.sealedmedia.softseal.jpg", "sjp")
        addMapping("image/vnd.sealedmedia.softseal.jpg", "s1j")
        addMapping("model/vnd.parasolid.transmit.text", "x_t")
        addMapping("model/vnd.parasolid.transmit.text", "xmt_txt")
        addMapping("application/vnd.sun.xml.math", "sxm")
        addMapping("application/vnd.cluetrust.cartomobile-config", "c11amc")
        addMapping("application/vnd.spotfire.dxp", "dxp")
        addMapping("application/vnd.smaf", "mmf")
        addMapping("application/vnd.joost.joda-archive", "joda")
        addMapping("audio/x-annodex", "axa")
        addMapping("application/vnd.sun.xml.writer", "sxw")
        addMapping("application/vnd.3gpp2.sms", "sms")
        addMapping("image/vnd.fujixerox.edmics-mmr", "mmr")
        addMapping("application/vnd.theqvd", "qvd")
        addMapping("application/vnd.data-vision.rdz", "rdz")
        addMapping("application/vnd.pvi.ptid1", "ptid")
        addMapping("application/vnd.sus-calendar", "sus")
        addMapping("application/vnd.sus-calendar", "susp")
        addMapping("application/timestamped-data", "tsd")
        addMapping("application/vnd.acucobol", "acu")
        addMapping("application/vnd.olpc-sugar", "xo")
        addMapping("application/vnd.sealed.ppt", "sppt")
        addMapping("application/vnd.sealed.ppt", "s1p")
        addMapping("application/vnd.Mobius.MBK", "mbk")
        addMapping("application/vnd.Mobius.DIS", "dis")
        addMapping("application/vnd.lotus-1-2-3", "123")
        addMapping("application/vnd.lotus-1-2-3", "wk4")
        addMapping("application/vnd.lotus-1-2-3", "wk3")
        addMapping("application/vnd.lotus-1-2-3", "wk1")
        addMapping("application/vnd.novadigm.EXT", "ext")
        addMapping("application/vnd.clonk.c4group", "c4g")
        addMapping("application/vnd.clonk.c4group", "c4d")
        addMapping("application/vnd.clonk.c4group", "c4f")
        addMapping("application/vnd.clonk.c4group", "c4p")
        addMapping("application/vnd.clonk.c4group", "c4u")
        addMapping("application/vnd.geonext", "gxt")
        addMapping("application/srgs", "gram")
        addMapping("image/x-xbitmap", "xbm")
        addMapping("application/vnd.oasis.opendocument.graphics-template", "otg")
        addMapping("application/vnd.onepagertatx", "tatx")
        addMapping("model/vnd.parasolid.transmit.binary", "x_b")
        addMapping("model/vnd.parasolid.transmit.binary", "xmt_bin")
        addMapping("application/vnd.noblenet-directory", "nnd")
        addMapping("application/vnd.onepagertatp", "tatp")
        addMapping("audio/x-wav", "wav")
        addMapping("application/vnd.kahootz", "ktz")
        addMapping("application/vnd.kahootz", "ktr")
        addMapping("application/ccmp+xml", "ccmp")
        addMapping("font/ttf", "ttf")
        addMapping("application/cms", "cmsc")
        addMapping("image/emf", "emf")
        addMapping("audio/vnd.audikoz", "koz")
        addMapping("application/octet-stream", "bin")
        addMapping("application/octet-stream", "lha")
        addMapping("application/octet-stream", "lzh")
        addMapping("application/octet-stream", "exe")
        addMapping("application/octet-stream", "class")
        addMapping("application/octet-stream", "so")
        addMapping("application/octet-stream", "dll")
        addMapping("application/octet-stream", "img")
        addMapping("application/octet-stream", "iso")
        addMapping("video/x-ms-asf", "asx")
        addMapping("application/rdf+xml", "rdf")
        addMapping("application/x-sv4crc", "sv4crc")
        addMapping("application/vnd.google-earth.kml+xml", "kml")
        addMapping("video/3gpp2", "3g2")
        addMapping("video/3gpp2", "3gpp2")
        addMapping("application/vnd.dvb.ait", "ait")
        addMapping("audio/mpeg", "mp3")
        addMapping("audio/mpeg", "mpga")
        addMapping("audio/mpeg", "mp1")
        addMapping("audio/mpeg", "mp2")
        addMapping("application/davmount+xml", "davmount")
        addMapping("application/vnd.geoplan", "g2w")
        addMapping("application/vnd.palm", "prc")
        addMapping("application/vnd.palm", "pdb")
        addMapping("application/vnd.palm", "pqa")
        addMapping("application/vnd.palm", "oprc")
        addMapping("audio/x-mpegurl", "m3u")
        addMapping("application/tamp-community-update", "tcu")
        addMapping("application/xcap-caps+xml", "xca")
        addMapping("image/svg+xml", "svg")
        addMapping("image/svg+xml", "svgz")
        addMapping("model/vnd.dwf", "dwf")
        addMapping("application/vnd.hp-jlyt", "jlt")
        addMapping("application/vnd.llamagraphics.life-balance.exchange+xml", "lbe")
        addMapping("application/vnd.apple.installer+xml", "dist")
        addMapping("application/vnd.apple.installer+xml", "distz")
        addMapping("application/vnd.apple.installer+xml", "pkg")
        addMapping("application/vnd.apple.installer+xml", "mpkg")
        addMapping("application/vnd.osa.netdeploy", "ndc")
        addMapping("message/global-delivery-status", "u8dsn")
        addMapping("video/vnd.vivo", "viv")
        addMapping("application/vnd.jam", "jam")
        addMapping("image/fits", "fits")
        addMapping("image/fits", "fit")
        addMapping("image/fits", "fts")
        addMapping("application/mathematica", "nb")
        addMapping("application/mathematica", "ma")
        addMapping("application/mathematica", "mb")
        addMapping("audio/ATRAC3", "at3")
        addMapping("audio/ATRAC3", "aa3")
        addMapping("audio/ATRAC3", "omg")
        addMapping("application/mp21", "m21")
        addMapping("application/mp21", "mp21")
        addMapping("image/vnd.airzip.accelerator.azv", "azv")
        addMapping("application/MF4", "mf4")
        addMapping("model/x3d-vrml", "x3dv")
        addMapping("model/x3d-vrml", "x3dvz")
        addMapping("application/vnd.software602.filler.form-xml-zip", "zfo")
        addMapping("image/x-portable-anymap", "pnm")
        addMapping("application/yin+xml", "yin")
        addMapping("application/vnd.balsamiq.bmml+xml", "bmml")
        addMapping("application/cellml+xml", "cellml")
        addMapping("application/cellml+xml", "cml")
        addMapping("video/x-msvideo", "avi")
        addMapping("application/ecmascript", "es")
        addMapping("application/vnd.kde.kpresenter", "kpr")
        addMapping("application/vnd.kde.kpresenter", "kpt")
        addMapping("application/vnd.osgi.subsystem", "esa")
        addMapping("application/vnd.1000minds.decision-model+xml", "1km")
        addMapping("application/vnd.kde.kontour", "kon")
        addMapping("application/vnd.geospace", "g3w")
        addMapping("image/vnd.xiff", "xif")
        addMapping("application/vnd.uiq.theme", "utz")
        addMapping("video/x-annodex", "axv")
        addMapping("audio/prs.sid", "sid")
        addMapping("audio/prs.sid", "psid")
        addMapping("application/oebps-package+xml", "opf")
        addMapping("application/vnd.oxli.countgraph", "oxlicg")
        addMapping("application/vnd.syncml.dmddf+xml", "ddf")
        addMapping("application/PDX", "pdx")
        addMapping("application/vnd.comicbook+zip", "cbz")
        addMapping("application/dssc+der", "dssc")
        addMapping("text/csv", "csv")
        addMapping("application/pkcs10", "p10")
        addMapping("application/pkcs12", "p12")
        addMapping("application/pkcs12", "pfx")
        addMapping("application/vnd.lotus-notes", "nsf")
        addMapping("application/vnd.lotus-notes", "ntf")
        addMapping("application/vnd.lotus-notes", "ndl")
        addMapping("application/vnd.lotus-notes", "ns4")
        addMapping("application/vnd.lotus-notes", "ns3")
        addMapping("application/vnd.lotus-notes", "ns2")
        addMapping("application/vnd.lotus-notes", "nsh")
        addMapping("application/vnd.lotus-notes", "nsg")
        addMapping("application/vnd.is-xpr", "xpr")
        addMapping("text/css", "css")
        addMapping("text/troff", "t")
        addMapping("text/troff", "tr")
        addMapping("text/troff", "roff")
        addMapping("image/cgm", "cgm")
        addMapping("model/vnd.mts", "mts")
        addMapping("application/x-troff-ms", "ms")
        addMapping("application/sru+xml", "sru")
        addMapping("application/vnd.kde.kivio", "flw")
        addMapping("application/vnd.scribus", "scd")
        addMapping("application/vnd.scribus", "sla")
        addMapping("application/vnd.scribus", "slaz")
        addMapping("application/vnd.las.las+json", "lasjson")
        addMapping("application/sparql-query", "rq")
        addMapping("application/vnd.sun.xml.calc.template", "stc")
        addMapping("audio/EVRCWB", "evw")
        addMapping("application/x-troff-me", "me")
        addMapping("application/resource-lists+xml", "rl")
        addMapping("application/vnd.tmobile-livetv", "tmo")
        addMapping("application/x-dvi", "dvi")
        addMapping("application/x-latex", "latex")
        addMapping("application/vnd.oma.drm.message", "dm")
        addMapping("application/watcherinfo+xml", "wif")
        addMapping("application/vnd.frogans.ltf", "ltf")
        addMapping("model/x3d+xml", "x3db")
        addMapping("application/ttml+xml", "ttml")
        addMapping("video/mpeg", "mpeg")
        addMapping("video/mpeg", "mpg")
        addMapping("video/mpeg", "mpe")
        addMapping("video/mpeg", "m1v")
        addMapping("video/mpeg", "m2v")
        addMapping("application/vnd.mcd", "mcd")
        addMapping("audio/vnd.everad.plj", "plj")
        addMapping("application/oda", "oda")
        addMapping("application/vnd.oasis.opendocument.text-master", "odm")
        addMapping("application/vnd.bluetooth.ep.oob", "ep")
        addMapping("text/tab-separated-values", "tsv")
        addMapping("audio/vnd.ms-playready.media.pya", "pya")
        addMapping("application/vnd.xmpie.dpkg", "dpkg")
        addMapping("image/gif", "gif")
        addMapping("application/vnd.bmi", "bmi")
        addMapping("application/vnd.dtg.local.flash", "fla")
        addMapping("text/vnd.esmertec.theme-descriptor", "jtd")
        addMapping("application/vnd.oma.drm.dcf", "o4a")
        addMapping("application/vnd.oma.drm.dcf", "o4v")
        addMapping("audio/ATRAC-X", "atx")
        addMapping("image/vnd.fastbidsheet", "fbs")
        addMapping("application/vnd.epson.salt", "slt")
        addMapping("application/sdp", "sdp")
        addMapping("application/srgs+xml", "grxml")
        addMapping("application/vnd.pmi.widget", "wg")
        addMapping("application/vnd.triscape.mxs", "mxs")
        addMapping("audio/L16", "l16")
        addMapping("message/global", "u8msg")
        addMapping("application/vnd.sun.wadl+xml", "wadl")
        addMapping("application/link-format", "wlnk")
        addMapping("application/vnd.kde.kformula", "kfo")
        addMapping("application/vnd.quobject-quoxdocument", "quox")
        addMapping("application/vnd.quobject-quoxdocument", "quiz")
        addMapping("text/cache-manifest", "appcache")
        addMapping("text/cache-manifest", "manifest")
        addMapping("application/vnd.multiad.creator", "crtr")
        addMapping("audio/iLBC", "lbc")
        addMapping("application/vnd.oasis.opendocument.text", "odt")
        addMapping("application/x-bcpio", "bcpio")
        addMapping("application/atomdeleted+xml", "atomdeleted")
        addMapping("application/ocsp-response", "ors")
        addMapping("application/vnd.adobe.fxp", "fxp")
        addMapping("application/vnd.adobe.fxp", "fxpl")
        addMapping("model/vnd.gdl", "gdl")
        addMapping("model/vnd.gdl", "gsm")
        addMapping("model/vnd.gdl", "win")
        addMapping("model/vnd.gdl", "dor")
        addMapping("model/vnd.gdl", "lmp")
        addMapping("model/vnd.gdl", "rsm")
        addMapping("model/vnd.gdl", "msm")
        addMapping("model/vnd.gdl", "ism")
        addMapping("application/scvp-cv-response", "scs")
        addMapping("font/woff", "woff")
        addMapping("model/vnd.collada+xml", "dae")
        addMapping("model/vnd.valve.source.compiled-map", "bsp")
        addMapping("application/lostsync+xml", "lostsyncxml")
        addMapping("audio/vnd.dts", "dts")
        addMapping("image/vnd.tencent.tap", "tap")
        addMapping("application/vnd.businessobjects", "rep")
        addMapping("application/vnd.sigrok.session", "sr")
        addMapping("application/vnd.ms-works", "wcm")
        addMapping("application/vnd.ms-works", "wdb")
        addMapping("application/vnd.ms-works", "wks")
        addMapping("application/vnd.ms-works", "wps")
        addMapping("image/vnd.fujixerox.edmics-rlc", "rlc")
        addMapping("audio/x-stm", "stm")
        addMapping("application/vnd.dna", "dna")
        addMapping("application/vnd.lotus-screencam", "scm")
        addMapping("application/vnd.fsc.weblaunch", "fsc")
        addMapping("application/vnd.dynageo", "geo")
        addMapping("application/vnd.fluxtime.clip", "ftc")
        addMapping("image/ktx", "ktx")
        addMapping("application/vnd.sealed.csf", "scsf")
        addMapping("audio/x-s3m", "s3m")
        addMapping("application/mbox", "mbox")
        addMapping("application/prs.xsf+xml", "xsf")
        addMapping("application/vnd.accpac.simply.aso", "aso")
        addMapping("image/vnd.radiance", "hdr")
        addMapping("image/vnd.radiance", "rgbe")
        addMapping("image/vnd.radiance", "xyze")
        addMapping("application/vnd.xmpie.xlim", "xlim")
        addMapping("application/urc-grpsheet+xml", "gsheet")
        addMapping("application/vnd.fastcopy-disk-image", "dim")
        addMapping("application/vnd.ms-word.template.macroEnabled.12", "dotm")
        addMapping("application/vnd.seemail", "see")
        addMapping("application/scvp-vp-response", "spp")
        addMapping("application/vnd.nokia.n-gage.ac+xml", "ac")
        addMapping("application/vnd.novadigm.EDM", "edm")
        addMapping("application/vnd.ms-excel", "xls")
        addMapping("application/vnd.ms-excel", "xlm")
        addMapping("application/vnd.ms-excel", "xla")
        addMapping("application/vnd.ms-excel", "xlc")
        addMapping("application/vnd.ms-excel", "xlt")
        addMapping("application/vnd.ms-excel", "xlw")
        addMapping("application/vnd.noblenet-web", "nnw")
        addMapping("application/tamp-error", "ter")
        addMapping("application/pkcs8", "p8")
        addMapping("application/vnd.novadigm.EDX", "edx")
        addMapping("application/x-sv4cpio", "sv4cpio")
        addMapping("application/vnd.muvee.style", "msty")
        addMapping("application/xml-dtd", "dtd")
        addMapping("model/vnd.opengex", "ogex")
        addMapping("application/vnd.sealed.eml", "seml")
        addMapping("application/vnd.sealed.eml", "sem")
        addMapping("application/vnd.document+json", "docjson")
        addMapping("application/route-apd+xml", "rapd")
        addMapping("application/vnd.ims.imsccv1p1", "imscc")
        addMapping("application/metalink+xml", "metalink")
        addMapping("application/vnd.oasis.opendocument.database", "odb")
        addMapping("application/vnd.wqd", "wqd")
        addMapping("model/gltf+json", "gltf")
        addMapping("application/vnd.mapbox-vector-tile", "mvt")
        addMapping("application/scim+json", "scim")
        addMapping("application/vnd.Mobius.MQY", "mqy")
        addMapping("application/vnd.cinderella", "cdy")
        addMapping("application/postscript", "ps")
        addMapping("application/postscript", "eps")
        addMapping("application/postscript", "ai")
        addMapping("text/prs.fallenstein.rst", "rst")
        addMapping("application/vnd.kde.karbon", "karbon")
        addMapping("application/vnd.ms-ims", "ims")
        addMapping("application/timestamp-reply", "tsr")
        addMapping("application/vnd.ipunplugged.rcprofile", "rcprofile")
        addMapping("application/jrd+json", "jrd")
        addMapping("audio/EVRCNW", "enw")
        addMapping("application/vnd.yamaha.openscoreformat", "osf")
        addMapping("application/shf+xml", "shf")
        addMapping("application/vnd.smart.teacher", "teacher")
        addMapping("application/vnd.fujitsu.oasysprs", "bh2")
        addMapping("application/vnd.spotfire.sfs", "sfs")
        addMapping("audio/AMR-WB", "awb")
        addMapping("video/vnd.dvb.file", "dvb")
        addMapping("application/vnd.nintendo.snes.rom", "sfc")
        addMapping("application/vnd.nintendo.snes.rom", "smc")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.template", "potx")
        addMapping("video/vnd.sealed.mpeg4", "s14")
        addMapping("application/CEA", "cea")
        addMapping("application/tei+xml", "tei")
        addMapping("application/tei+xml", "teiCorpus")
        addMapping("application/tei+xml", "odd")
        addMapping("video/vnd.sealed.mpeg1", "smpg")
        addMapping("video/vnd.sealed.mpeg1", "s11")
        addMapping("video/vnd.sealedmedia.softseal.mov", "smov")
        addMapping("video/vnd.sealedmedia.softseal.mov", "smo")
        addMapping("video/vnd.sealedmedia.softseal.mov", "s1q")
        addMapping("application/x-gtar", "gtar")
        addMapping("video/vnd.dece.sd", "uvs")
        addMapping("video/vnd.dece.sd", "uvvs")
        addMapping("application/prs.hpub+zip", "hpub")
        addMapping("application/vnd.nokia.n-gage.data", "ngdat")
        addMapping("image/ief", "ief")
        addMapping("application/ld+json", "jsonld")
        addMapping("application/relax-ng-compact-syntax", "rnc")
        addMapping("application/vnd.ibm.modcap", "list3820")
        addMapping("application/vnd.ibm.modcap", "listafp")
        addMapping("application/vnd.ibm.modcap", "afp")
        addMapping("application/vnd.ibm.modcap", "pseg3820")
        addMapping("application/vnd.kde.kchart", "chrt")
        addMapping("application/vnd.yellowriver-custom-menu", "cmp")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.document", "docx")
        addMapping("video/vnd.dece.pd", "uvp")
        addMapping("video/vnd.dece.pd", "uvvp")
        addMapping("application/x-cdlink", "vcd")
        addMapping("application/csrattrs", "csrattrs")
        addMapping("application/vnd.crick.clicker.keyboard", "clkk")
        addMapping("text/vnd.fmi.flexstor", "flx")
        addMapping("application/n-triples", "nt")
        addMapping("audio/asc", "acn")
        addMapping("application/smil+xml", "smil")
        addMapping("application/smil+xml", "smi")
        addMapping("application/smil+xml", "sml")
        addMapping("application/vnd.epson.ssf", "ssf")
        addMapping("text/vnd.in3d.3dml", "3dml")
        addMapping("text/vnd.in3d.3dml", "3dm")
        addMapping("application/x-annodex", "anx")
        addMapping("application/vnd.cosmocaller", "cmc")
        addMapping("application/vnd.pocketlearn", "plf")
        addMapping("text/markdown", "markdown")
        addMapping("text/markdown", "md")
        addMapping("application/vnd.dvb.service", "svc")
        addMapping("application/dash+xml", "mpd")
        addMapping("application/calendar+xml", "xcs")
        addMapping("application/vnd.frogans.fnc", "fnc")
        addMapping("model/vnd.moml+xml", "moml")
        addMapping("application/vnd.wap.wmlscriptc", "wmlsc")
        addMapping("application/vnd.wmc", "wmc")
        addMapping("application/vnd.recordare.musicxml", "mxl")
        addMapping("model/iges", "igs")
        addMapping("model/iges", "iges")
        addMapping("text/richtext", "rtx")
        addMapping("image/x-targa", "tga")
        addMapping("application/xcap-att+xml", "xav")
        addMapping("application/vnd.igloader", "igl")
        addMapping("application/vnd.cups-ppd", "ppd")
        addMapping("application/x-shar", "shar")
        addMapping("application/tamp-update", "tur")
        addMapping("application/vnd.publishare-delta-tree", "qps")
        addMapping("application/vnd.qualcomm.brew-app-res", "bar")
        addMapping("application/vnd.crick.clicker.palette", "clkp")
        addMapping("application/gzip", "gz")
        addMapping("application/gzip", "tgz")
        addMapping("application/vnd.sun.xml.impress", "sxi")
        addMapping("application/vnd.yamaha.hv-voice", "hvp")
        addMapping("application/vnd.sealedmedia.softseal.pdf", "spdf")
        addMapping("application/vnd.sealedmedia.softseal.pdf", "spd")
        addMapping("application/vnd.sealedmedia.softseal.pdf", "s1a")
        addMapping("application/vnd.groove-tool-template", "tpl")
        addMapping("text/vnd.net2phone.commcenter.command", "ccc")
        addMapping("chemical/x-xyz", "xyz")
        addMapping("application/vnd.uri-map", "urim")
        addMapping("application/vnd.uri-map", "urimap")
        addMapping("application/vnd.Quark.QuarkXPress", "qxd")
        addMapping("application/vnd.Quark.QuarkXPress", "qxt")
        addMapping("application/vnd.Quark.QuarkXPress", "qwd")
        addMapping("application/vnd.Quark.QuarkXPress", "qwt")
        addMapping("application/vnd.Quark.QuarkXPress", "qxl")
        addMapping("application/vnd.Quark.QuarkXPress", "qxb")
        addMapping("application/vnd.imagemeter.image+zip", "imi")
        addMapping("application/vnd.ms-excel.template.macroEnabled.12", "xltm")
        addMapping("text/provenance-notation", "provn")
        addMapping("application/sgml-open-catalog", "soc")
        addMapping("application/vnd.onepager", "tam")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slideshow", "ppsx")
        addMapping("application/vnd.geogebra.file", "ggb")
        addMapping("application/voicexml+xml", "vxml")
        addMapping("video/x-javafx", "fxm")
        addMapping("application/vnd.xara", "xar")
        addMapping("audio/midi", "mid")
        addMapping("audio/midi", "midi")
        addMapping("audio/midi", "kar")
        addMapping("application/vnd.hal+xml", "hal")
        addMapping("application/vnd.wfa.p2p", "p2p")
        addMapping("application/x-perl", "pl")
        addMapping("application/vnd.ibm.electronic-media", "emm")
        addMapping("application/vnd.tao.intent-module-archive", "tao")
        addMapping("application/vnd.pawaafile", "paw")
        addMapping("application/wsdl+xml", "wsdl")
        addMapping("image/x-rgb", "rgb")
        addMapping("application/vnd.epson.msf", "msf")
        addMapping("application/vnd.oasis.opendocument.spreadsheet-template", "ots")
        addMapping("application/trig", "trig")
        addMapping("application/vnd.3lightssoftware.imagescal", "imgcal")
        addMapping("application/x-futuresplash", "spl")
        addMapping("text/jcr-cnd", "cnd")
        addMapping("application/mets+xml", "mets")
        addMapping("video/vnd.mpegurl", "mxu")
        addMapping("video/vnd.mpegurl", "m4u")
        addMapping("application/vnd.FloGraphIt", "gph")
        addMapping("application/x-netcdf", "nc")
        addMapping("application/x-netcdf", "cdf")
        addMapping("text/vcard", "vcf")
        addMapping("text/vcard", "vcard")
        addMapping("application/vnd.HandHeld-Entertainment+xml", "zmm")
        addMapping("application/vnd.intercon.formnet", "xpw")
        addMapping("application/vnd.intercon.formnet", "xpx")
        addMapping("video/x-matroska-3d", "mk3d")
        addMapping("application/vnd.kde.kspread", "ksp")
        addMapping("application/prs.cww", "cw")
        addMapping("application/prs.cww", "cww")
        addMapping("application/vnd.balsamiq.bmpr", "bmpr")
        addMapping("application/xop+xml", "xop")
        addMapping("image/x-xpixmap", "xpm")
        addMapping("application/geo+json", "geojson")
        addMapping("application/vnd.lotus-approach", "apr")
        addMapping("application/vnd.lotus-approach", "vew")
        addMapping("video/quicktime", "mov")
        addMapping("video/quicktime", "qt")
        addMapping("video/mp4", "mp4")
        addMapping("video/mp4", "mpg4")
        addMapping("video/mp4", "m4v")
        addMapping("application/x-sh", "sh")
        addMapping("video/webm", "webm")
        addMapping("image/vnd.djvu", "djvu")
        addMapping("image/vnd.djvu", "djv")
        addMapping("application/vnd.nokia.radio-presets", "rpss")
        addMapping("video/x-ms-wm", "wm")
        addMapping("audio/vnd.nortel.vbk", "vbk")
        addMapping("application/mads+xml", "mads")
        addMapping("image/vnd.valve.source.texture", "vtf")
        addMapping("application/vnd.font-fontforge-sfd", "sfd")
        addMapping("application/vnd.etsi.asic-s+zip", "asics")
        addMapping("application/p2p-overlay+xml", "relo")
        addMapping("application/vnd.fujixerox.docuworks", "xdw")
        addMapping("application/vnd.mophun.certificate", "mpc")
        addMapping("application/vnd.yamaha.hv-script", "hvs")
        addMapping("application/vnd.hhe.lesson-player", "les")
        addMapping("video/3gpp", "3gp")
        addMapping("video/3gpp", "3gpp")
        addMapping("x-epoc/x-sisx-app", "sisx")
        addMapping("application/vnd.claymore", "cla")
        addMapping("application/vnd.acucorp", "atc")
        addMapping("application/vnd.acucorp", "acutc")
        addMapping("application/vnd.denovo.fcselayout-link", "fe_launch")
        addMapping("application/vnd.chess-pgn", "pgn")
        addMapping("application/vnd.nervana", "entity")
        addMapping("application/vnd.nervana", "request")
        addMapping("application/vnd.nervana", "bkm")
        addMapping("application/vnd.nervana", "kcm")
        addMapping("application/zip", "zip")
        addMapping("application/rpki-roa", "roa")
        addMapping("application/xslt+xml", "xsl")
        addMapping("application/xslt+xml", "xslt")
        addMapping("application/vnd.groove-vcard", "vcg")
        addMapping("application/dvcs", "dvc")
        addMapping("image/vnd.adobe.photoshop", "psd")
        addMapping("application/vnd.audiograph", "aep")
        addMapping("application/vnd.Mobius.MSL", "msl")
        addMapping("video/vnd.dece.hd", "uvh")
        addMapping("video/vnd.dece.hd", "uvvh")
        addMapping("model/vnd.vtu", "vtu")
        addMapping("image/vnd.zbrush.pcx", "pcx")
        addMapping("application/vnd.oma.drm.rights+xml", "dr")
        addMapping("application/vnd.hydrostatix.sof-data", "sfd-hdstx")
        addMapping("application/vnd.ms-officetheme", "thmx")
        addMapping("application/vnd.google-earth.kmz", "kmz")
        addMapping("application/vnd.ericsson.quickcall", "qcall")
        addMapping("application/vnd.ericsson.quickcall", "qca")
        addMapping("application/vnd.marlin.drm.mdcf", "mdc")
        addMapping("application/tamp-apex-update-confirm", "auc")
        addMapping("application/vnd.syncml.dm+xml", "xdm")
        addMapping("application/vnd.vcx", "vcx")
        addMapping("application/gml+xml", "gml")
        addMapping("application/ATF", "atf")
        addMapping("text/vnd.fly", "fly")
        addMapping("application/vnd.curl", "curl")
        addMapping("audio/EVRC", "evc")
        addMapping("audio/SMV", "smv")
        addMapping("application/owl+xml", "owx")
        addMapping("application/vnd.ibm.MiniPay", "mpy")
        addMapping("application/x-java-pack200", "pack")
        addMapping("video/x-ms-wvx", "wvx")
        addMapping("application/vnd.proteus.magazine", "mgz")
        addMapping("application/inkml+xml", "ink")
        addMapping("application/inkml+xml", "inkml")
        addMapping("application/x-xz", "xz")
        addMapping("model/vnd.gtw", "gtw")
        addMapping("application/vnd.ms-powerpoint.slide.macroEnabled.12", "sldm")
        addMapping("application/vnd.cluetrust.cartomobile-config-pkg", "c11amz")
        addMapping("video/vnd.radgamettools.bink", "bik")
        addMapping("video/vnd.radgamettools.bink", "bk2")
        addMapping("application/vnd.3gpp.pic-bw-large", "plb")
        addMapping("application/x-director", "dcr")
        addMapping("application/x-director", "dir")
        addMapping("application/x-director", "dxr")
        addMapping("application/vnd.groove-injector", "grv")
        addMapping("application/vnd.ms-excel.addin.macroEnabled.12", "xlam")
        addMapping("font/otf", "otf")
        addMapping("text/vnd.trolltech.linguist", "ts")
        addMapping("video/vnd.radgamettools.smacker", "smk")
        addMapping("application/dashdelta", "mpdd")
        addMapping("audio/vnd.dolby.mlp", "mlp")
        addMapping("application/vnd.nintendo.nitro.rom", "nds")
        addMapping("application/vnd.maxmind.maxmind-db", "mmdb")
        addMapping("application/pkix-crl", "crl")
        addMapping("image/x-xwindowdump", "xwd")
        addMapping("image/vnd.ms-modi", "mdi")
        addMapping("application/x-cpio", "cpio")
        addMapping("application/vnd.espass-espass+zip", "espass")
        addMapping("video/x-sgi-movie", "movie")
        addMapping("image/prs.pti", "pti")
        addMapping("video/vnd.sealed.swf", "sswf")
        addMapping("video/vnd.sealed.swf", "ssw")
        addMapping("application/vnd.sailingtracker.track", "st")
        addMapping("application/vnd.criticaltools.wbs+xml", "wbs")
        addMapping("application/vnd.evolv.ecig.theme", "ecigtheme")
        addMapping("application/vnd.oasis.opendocument.presentation-template", "otp")
        addMapping("application/thraud+xml", "tfi")
        addMapping("application/vnd.sealed.xls", "sxls")
        addMapping("application/vnd.sealed.xls", "sxl")
        addMapping("application/vnd.sealed.xls", "s1e")
        addMapping("application/pkixcmp", "pki")
        addMapping("image/x-portable-graymap", "pgm")
        addMapping("application/vnd.ibm.secure-container", "sc")
        addMapping("application/vnd.ms-excel.sheet.macroEnabled.12", "xlsm")
        addMapping("text/vnd.debian.copyright", "copyright")
        addMapping("application/vnd.wap.sic", "sic")
        addMapping("application/vnd.jisp", "jisp")
        addMapping("application/xv+xml", "mxml")
        addMapping("application/xv+xml", "xhvml")
        addMapping("application/xv+xml", "xvml")
        addMapping("application/xv+xml", "xvm")
        addMapping("application/vnd.previewsystems.box", "box")
        addMapping("application/vnd.previewsystems.box", "vbox")
        addMapping("application/x-ustar", "ustar")
        addMapping("application/sql", "sql")
        addMapping("application/vnd.intergeo", "i2g")
        addMapping("application/wspolicy+xml", "wspolicy")
        addMapping("application/vnd.msa-disk-image", "msa")
        addMapping("application/vnd.realvnc.bed", "bed")
        addMapping("application/vnd.ms-project", "mpp")
        addMapping("application/vnd.ms-project", "mpt")
        addMapping("application/prs.nprend", "rnd")
        addMapping("application/prs.nprend", "rct")
        addMapping("application/vnd.adobe.xfdf", "xfdf")
        addMapping("application/CDFX+XML", "cdfx")
        addMapping("application/vnd.fujitsu.oasys2", "oa2")
        addMapping("application/vnd.oasis.opendocument.text-template", "ott")
        addMapping("audio/x-matroska", "mka")
        addMapping("application/vnd.fujitsu.oasys3", "oa3")
        addMapping("text/vnd.ms-mediapackage", "mpf")
        addMapping("application/x-hdf", "hdf")
        addMapping("application/x-texinfo", "texinfo")
        addMapping("application/x-texinfo", "texi")
        addMapping("application/pls+xml", "pls")
        addMapping("application/route-usd+xml", "rusd")
        addMapping("application/json-patch+json", "json-patch")
        addMapping("image/bmp", "bmp")
        addMapping("image/bmp", "dib")
        addMapping("application/sieve", "siv")
        addMapping("application/sieve", "sieve")
        addMapping("application/yang", "yang")
        addMapping("application/mods+xml", "mods")
        addMapping("audio/mobile-xmf", "mxmf")
        addMapping("application/scvp-vp-request", "spq")
        addMapping("application/vnd.3M.Post-it-Notes", "pwn")
        addMapping("application/timestamp-query", "tsq")
        addMapping("image/x-cmu-raster", "ras")
        addMapping("text/dns", "soa")
        addMapping("text/dns", "zone")
        addMapping("application/vnd.ms-word.document.macroEnabled.12", "docm")
        addMapping("application/vnd.ms-powerpoint.slideshow.macroEnabled.12", "ppsm")
        addMapping("application/cdmi-queue", "cdmiq")
        addMapping("audio/qcelp", "qcp")
        addMapping("video/iso.segment", "m4s")
        addMapping("application/vnd.xfdl", "xfdl")
        addMapping("application/vnd.xfdl", "xfd")
        addMapping("application/vnd.yaoweme", "yme")
        addMapping("application/vnd.nokia.n-gage.symbian.install", "n-gage")
        addMapping("application/vnd.groove-account", "gac")
        addMapping("application/vnd.ms-powerpoint", "ppt")
        addMapping("application/vnd.ms-powerpoint", "pps")
        addMapping("application/vnd.ms-powerpoint", "pot")
        addMapping("image/vnd.sealedmedia.softseal.gif", "sgif")
        addMapping("image/vnd.sealedmedia.softseal.gif", "sgi")
        addMapping("image/vnd.sealedmedia.softseal.gif", "s1g")
        addMapping("application/vnd.dzr", "dzr")
        addMapping("image/vnd.globalgraphics.pgb", "pgb")
        addMapping("image/tiff", "tiff")
        addMapping("image/tiff", "tif")
        addMapping("application/x-stuffit", "sit")
        addMapping("application/vnd.MFER", "mwf")
        addMapping("application/vnd.wap.wbxml", "wbxml")
        addMapping("application/vnd.dece.unspecified", "uvx")
        addMapping("application/vnd.dece.unspecified", "uvvx")
        addMapping("application/vnd.vectorworks", "vwx")
        addMapping("audio/vnd.dts.hd", "dtshd")
        addMapping("application/vnd.wt.stf", "stf")
        addMapping("application/vnd.etsi.asic-e+zip", "asice")
        addMapping("application/vnd.etsi.asic-e+zip", "sce")
        addMapping("application/vnd.SimTech-MindMapper", "twd")
        addMapping("application/vnd.SimTech-MindMapper", "twds")
        addMapping("application/widget", "wgt")
        addMapping("video/x-matroska", "mkv")
        addMapping("application/vnd.ezpix-album", "ez2")
        addMapping("application/vnd.sun.xml.calc", "sxc")
        addMapping("application/vnd.oma.dd2+xml", "dd2")
        addMapping("application/vnd.ms-tnef", "tnef")
        addMapping("application/vnd.ms-tnef", "tnf")
        addMapping("application/vnd.astraea-software.iota", "iota")
        addMapping("application/vnd.oasis.opendocument.text-web", "oth")
        addMapping("application/vnd.hp-hpid", "hpi")
        addMapping("application/vnd.hp-hpid", "hpid")
        addMapping("text/x-setext", "etx")
        addMapping("application/fdt+xml", "fdt")
        addMapping("application/vnd.airzip.filesecure.azs", "azs")
        addMapping("application/metalink4+xml", "meta4")
        addMapping("application/x-java-jnlp-file", "jnlp")
        addMapping("application/vnd.las.las+xml", "lasxml")
        addMapping("audio/vnd.dece.audio", "uva")
        addMapping("audio/vnd.dece.audio", "uvva")
        addMapping("text/vnd.dvb.subtitle", "sub")
        addMapping("application/vnd.isac.fcs", "fcs")
        addMapping("audio/x-mod", "mod")
        addMapping("audio/x-mod", "ult")
        addMapping("audio/x-mod", "uni")
        addMapping("audio/x-mod", "m15")
        addMapping("audio/x-mod", "mtm")
        addMapping("audio/x-mod", "669")
        addMapping("audio/x-mod", "med")
        addMapping("text/vnd.sun.j2me.app-descriptor", "jad")
        addMapping("application/vnd.stepmania.stepchart", "sm")
        addMapping("application/vnd.uoml+xml", "uoml")
        addMapping("application/vnd.uoml+xml", "uo")
        addMapping("application/vnd.airzip.filesecure.azf", "azf")
        addMapping("application/vnd.oasis.opendocument.presentation", "odp")
        addMapping("application/vnd.irepository.package+xml", "irp")
        addMapping("application/vnd.mif", "mif")
        addMapping("application/vnd.sealedmedia.softseal.html", "stml")
        addMapping("application/vnd.sealedmedia.softseal.html", "s1h")
        addMapping("audio/ac3", "ac3")
        addMapping("application/vnd.debian.binary-package", "deb")
        addMapping("application/vnd.debian.binary-package", "udeb")
        addMapping("application/vnd.doremir.scorecloud-binary-document", "scld")
        addMapping("video/mj2", "mj2")
        addMapping("video/mj2", "mjp2")
        addMapping("application/ogg", "ogx")
        addMapping("application/vnd.dart", "dart")
        addMapping("application/vnd.amiga.ami", "ami")
        addMapping("text/turtle", "ttl")
        addMapping("application/ATXML", "atxml")
        addMapping("text/vnd.ascii-art", "ascii")
        addMapping("video/vnd.fvt", "fvt")
        addMapping("application/vnd.hp-HPGL", "hpgl")
        addMapping("application/vnd.ibm.rights-management", "irm")
        addMapping("audio/vnd.digital-winds", "eol")
        addMapping("application/vnd.hzn-3d-crossword", "x3d")
        addMapping("video/vnd.dece.video", "uvv")
        addMapping("video/vnd.dece.video", "uvvv")
        addMapping("application/vnd.oma.drm.rights+wbxml", "drc")
        addMapping("application/vnd.fujitsu.oasysgp", "fg5")
        addMapping("application/vnd.anser-web-funds-transfer-initiation", "fti")
        addMapping("application/vnd.solent.sdkm+xml", "sdkm")
        addMapping("application/vnd.solent.sdkm+xml", "sdkd")
        addMapping("image/prs.btif", "btif")
        addMapping("image/prs.btif", "btf")
        addMapping("application/vnd.oasis.opendocument.image", "odi")
        addMapping("application/x-bzip2", "bz2")
        addMapping("application/atomcat+xml", "atomcat")
        addMapping("application/vnd.micrografx.flo", "flo")
        addMapping("application/vnd.wap.slc", "slc")
        addMapping("application/vnd.android.package-archive", "apk")
        addMapping("application/AML", "aml")
        addMapping("application/vnd.sema", "sema")
        addMapping("application/vnd.grafeq", "gqf")
        addMapping("application/vnd.grafeq", "gqs")
        addMapping("application/x-xpinstall", "xpi")
        addMapping("application/vnd.semd", "semd")
        addMapping("application/vnd.semf", "semf")
        addMapping("application/vnd.shana.informed.interchange", "iif")
        addMapping("audio/vnd.nuera.ecelp4800", "ecelp4800")
        addMapping("application/vnd.Kinar", "kne")
        addMapping("application/vnd.Kinar", "knp")
        addMapping("application/vnd.Kinar", "sdf")
        addMapping("application/vnd.openblox.game-binary", "obg")
        addMapping("image/vnd.dxf", "dxf")
        addMapping("message/rfc822", "eml")
        addMapping("message/rfc822", "mail")
        addMapping("message/rfc822", "art")
        addMapping("application/vnd.visio", "vsd")
        addMapping("application/vnd.visio", "vst")
        addMapping("application/vnd.visio", "vsw")
        addMapping("application/vnd.visio", "vss")
        addMapping("application/x-csh", "csh")
        addMapping("application/x-chrome-extension", "crx")
        addMapping("image/wmf", "wmf")
        addMapping("application/vnd.insors.igm", "igm")
        addMapping("application/vnd.jcp.javame.midlet-rms", "rms")
        addMapping("application/cdmi-capability", "cdmia")
        addMapping("application/vnd.epson.esf", "esf")
        addMapping("application/vnd.fuzzysheet", "fzs")
        addMapping("application/hyperstudio", "stk")
        addMapping("application/vnd.americandynamics.acc", "acc")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slide", "sldx")
        addMapping("application/vnd.etsi.timestamp-token", "tst")
        addMapping("application/vnd.dece.zip", "uvz")
        addMapping("application/vnd.dece.zip", "uvvz")
        addMapping("application/vnd.mseq", "mseq")
        addMapping("audio/mp4", "m4a")
        addMapping("application/pdf", "pdf")
        addMapping("application/pgp-encrypted", "pgp")
        addMapping("application/vnd.wap.wmlc", "wmlc")
        addMapping("application/x-bittorrent", "torrent")
        addMapping("application/sparql-results+xml", "srx")
        addMapping("multipart/voice-message", "vpm")
        addMapping("application/vnd.commerce-battelle", "ica")
        addMapping("application/vnd.commerce-battelle", "icf")
        addMapping("application/vnd.commerce-battelle", "icd")
        addMapping("application/vnd.commerce-battelle", "ic0")
        addMapping("application/vnd.commerce-battelle", "ic1")
        addMapping("application/vnd.commerce-battelle", "ic2")
        addMapping("application/vnd.commerce-battelle", "ic3")
        addMapping("application/vnd.commerce-battelle", "ic4")
        addMapping("application/vnd.commerce-battelle", "ic5")
        addMapping("application/vnd.commerce-battelle", "ic6")
        addMapping("application/vnd.commerce-battelle", "ic7")
        addMapping("application/vnd.commerce-battelle", "ic8")
        addMapping("application/mathml+xml", "mml")
        addMapping("application/vnd.kde.kword", "kwd")
        addMapping("application/vnd.kde.kword", "kwt")
        addMapping("application/vnd.ms-wpl", "wpl")
        addMapping("application/dssc+xml", "xdssc")
        addMapping("application/vnd.openstreetmap.data+xml", "osm")
        addMapping("application/vnd.xmpie.ppkg", "ppkg")
        addMapping("application/x-rpm", "rpm")
        addMapping("application/vnd.dece.data", "uvf")
        addMapping("application/vnd.dece.data", "uvvf")
        addMapping("application/vnd.dece.data", "uvd")
        addMapping("application/vnd.dece.data", "uvvd")
        addMapping("image/vnd.microsoft.icon", "ico")
        addMapping("application/vnd.sun.xml.writer.template", "stw")
        addMapping("application/vnd.wordperfect", "wpd")
        addMapping("application/vnd.umajin", "umj")
        addMapping("application/vnd.oma.drm.content", "dcf")
        addMapping("application/vnd.ms-powerpoint.presentation.macroEnabled.12", "pptm")
        addMapping("application/vnd.fujixerox.docuworks.container", "xct")
        addMapping("model/vrml", "wrl")
        addMapping("model/vrml", "vrml")
        addMapping("application/vnd.Mobius.TXF", "txf")
        addMapping("application/vnd.yamaha.smaf-phrase", "spf")
        addMapping("application/javascript", "js")
        addMapping("application/dskpp+xml", "xmls")
        addMapping("application/vnd.intu.qbo", "qbo")
        addMapping("application/ssml+xml", "ssml")
        addMapping("application/urc-ressheet+xml", "rsheet")
        addMapping("application/resource-lists-diff+xml", "rld")
        addMapping("application/vnd.ms-powerpoint.addin.macroEnabled.12", "ppam")
        addMapping("application/vnd.groove-tool-message", "gtm")
        addMapping("image/vnd.dwg", "dwg")
        addMapping("application/vnd.sun.xml.writer.global", "sxg")
        addMapping("application/vnd.iccprofile", "icc")
        addMapping("application/vnd.iccprofile", "icm")
        addMapping("application/oxps", "oxps")
        addMapping("application/vnd.mfmp", "mfm")
        addMapping("application/ODX", "odx")
        addMapping("application/vnd.oasis.opendocument.chart-template", "otc")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", "xlsx")
        addMapping("text/csv-schema", "csvs")
        addMapping("application/vnd.shana.informed.package", "ipk")
        addMapping("application/cdmi-object", "cdmio")
        addMapping("application/vnd.valve.source.material", "vmt")
        addMapping("application/vnd.cloanto.rp9", "rp9")
        addMapping("application/vnd.osgeo.mapguide.package", "mgp")
        addMapping("application/route-s-tsid+xml", "sls")
        addMapping("application/andrew-inset", "ez")
        addMapping("application/vnd.autopackage", "package")
        addMapping("video/ogg", "ogv")
        addMapping("application/atomsvc+xml", "atomsvc")
        addMapping("application/msword", "doc")
        addMapping("image/jpm", "jpm")
        addMapping("image/jpm", "jpgm")
        addMapping("application/vnd.ms-3mfdocument", "3mf")
        addMapping("application/DIT", "dit")
        addMapping("application/emotionml+xml", "emotionml")
        addMapping("image/jpx", "jpx")
        addMapping("image/jpx", "jpf")
        addMapping("text/vnd.graphviz", "gv")
        addMapping("text/vnd.graphviz", "dot")
        addMapping("application/vnd.dece.ttml+xml", "uvt")
        addMapping("application/vnd.dece.ttml+xml", "uvvt")
        addMapping("text/vnd.in3d.spot", "spot")
        addMapping("text/vnd.in3d.spot", "spo")
        addMapping("application/vnd.ezpix-package", "ez3")
        addMapping("application/DII", "dii")
        addMapping("application/vnd.route66.link66+xml", "link66")
        addMapping("application/vnd.sealed.mht", "smht")
        addMapping("application/vnd.sealed.mht", "smh")
        addMapping("application/vnd.vsf", "vsf")
        addMapping("application/cbor", "cbor")
        addMapping("application/vnd.openblox.game+xml", "obgx")
        addMapping("image/vnd.wap.wbmp", "wbmp")
        addMapping("application/vnd.ctc-posml", "pml")
        addMapping("application/pkcs7-mime", "p7m")
        addMapping("application/pkcs7-mime", "p7c")
        addMapping("application/cdmi-container", "cdmic")
        addMapping("application/vnd.powerbuilder6", "pbd")
        addMapping("application/vnd.intu.qfx", "qfx")
        addMapping("message/global-disposition-notification", "u8mdn")
        addMapping("application/rpki-manifest", "mft")
        addMapping("application/fastinfoset", "finf")
        addMapping("text/n3", "n3")
        addMapping("audio/x-pn-realaudio", "ram")
        addMapping("audio/x-pn-realaudio", "rm")
        addMapping("application/vnd.koan", "skp")
        addMapping("application/vnd.koan", "skd")
        addMapping("application/vnd.koan", "skm")
        addMapping("application/vnd.koan", "skt")
        addMapping("text/plain", "txt")
        addMapping("text/plain", "asc")
        addMapping("text/plain", "text")
        addMapping("text/plain", "pm")
        addMapping("text/plain", "el")
        addMapping("text/plain", "c")
        addMapping("text/plain", "h")
        addMapping("text/plain", "cc")
        addMapping("text/plain", "hh")
        addMapping("text/plain", "cxx")
        addMapping("text/plain", "hxx")
        addMapping("text/plain", "f90")
        addMapping("text/plain", "conf")
        addMapping("text/plain", "log")
        addMapping("application/ipfix", "ipfix")
        addMapping("application/vnd.ecowin.chart", "mag")
        addMapping("application/vnd.groove-identity-message", "gim")
        addMapping("application/vnd.tml", "vfr")
        addMapping("application/vnd.tml", "viaframe")
        addMapping("application/vnd.Mobius.PLC", "plc")
        addMapping("application/vnd.yamaha.hv-dic", "hvd")
        addMapping("application/vnd.hp-PCL", "pcl")
        addMapping("text/vnd.wap.wmlscript", "wmls")
        addMapping("application/vnd.sun.xml.impress.template", "sti")
        addMapping("application/vnd.oasis.opendocument.formula", "odf")
        addMapping("video/vnd.ms-playready.media.pyv", "pyv")
        addMapping("application/pgp-signature", "sig")
        addMapping("application/vnd.mozilla.xul+xml", "xul")
        addMapping("application/vnd.onepagertat", "tat")
        addMapping("application/vnd.webturbo", "wtb")
        addMapping("image/vnd.fst", "fst")
        addMapping("application/vnd.sun.xml.draw.template", "std")
        addMapping("application/vnd.filmit.zfc", "zfc")
        addMapping("application/vnd.vd-study", "mxi")
        addMapping("application/vnd.vd-study", "study-inter")
        addMapping("application/vnd.vd-study", "model-inter")
        addMapping("application/lost+xml", "lostxml")
        addMapping("application/vnd.shana.informed.formtemplate", "itp")
        addMapping("application/xcap-ns+xml", "xns")
        addMapping("image/x-portable-bitmap", "pbm")
        addMapping("application/vnd.lotus-organizer", "or3")
        addMapping("application/vnd.lotus-organizer", "or2")
        addMapping("application/vnd.lotus-organizer", "org")
        addMapping("audio/vnd.nuera.ecelp7470", "ecelp7470")
        addMapping("application/urc-targetdesc+xml", "td")
        addMapping("application/vnd.geogebra.tool", "ggt")
        addMapping("image/vnd.mozilla.apng", "apng")
        addMapping("text/prs.lines.tag", "tag")
        addMapping("text/prs.lines.tag", "dsc")
        addMapping("text/mizar", "miz")
        addMapping("font/woff2", "woff2")
        addMapping("application/json", "json")
        addMapping("audio/vnd.lucent.voice", "lvp")
        addMapping("application/x-troff-man", "man")
        addMapping("application/x-troff-man", "1")
        addMapping("application/x-troff-man", "2")
        addMapping("application/x-troff-man", "3")
        addMapping("application/x-troff-man", "4")
        addMapping("application/x-troff-man", "5")
        addMapping("application/x-troff-man", "6")
        addMapping("application/x-troff-man", "7")
        addMapping("application/x-troff-man", "8")
        addMapping("image/jls", "jls")
        addMapping("application/vnd.fdsn.mseed", "msd")
        addMapping("application/vnd.fdsn.mseed", "mseed")
        addMapping("application/vnd.fdsn.seed", "seed")
        addMapping("application/vnd.fdsn.seed", "dataless")
        addMapping("application/vnd.ms-lrm", "lrm")
        addMapping("application/auth-policy+xml", "apxml")
        addMapping("audio/vnd.presonus.multitrack", "multitrack")
        addMapping("application/efi", "efi")
        addMapping("application/vnd.kodak-descriptor", "sse")
        addMapping("audio/ATRAC-ADVANCED-LOSSLESS", "aal")
        addMapping("text/calendar", "ics")
        addMapping("text/calendar", "ifb")
        addMapping("application/vnd.macports.portpkg", "portpkg")
        addMapping("application/x-xspf+xml", "xspf")
        addMapping("application/tamp-sequence-adjust", "tsa")
        addMapping("video/vnd.dece.mobile", "uvm")
        addMapping("video/vnd.dece.mobile", "uvvm")
        addMapping("application/mxf", "mxf")
        addMapping("application/vnd.evolv.ecig.profile", "ecigprofile")
        addMapping("application/xhtml+xml", "xhtml")
        addMapping("application/xhtml+xml", "xhtm")
        addMapping("application/xhtml+xml", "xht")
        addMapping("application/pkcs7-signature", "p7s")
        addMapping("application/vnd.enliven", "nml")
        addMapping("application/vnd.micrografx.igx", "igx")
        addMapping("application/vnd.immervision-ivu", "ivu")
        addMapping("application/vnd.bluetooth.le.oob", "le")
        addMapping("application/vnd.chipnuts.karaoke-mmd", "mmd")
        addMapping("application/vnd.3gpp.pic-bw-small", "psb")
        addMapping("application/vnd.immervision-ivp", "ivp")
        addMapping("audio/basic", "au")
        addMapping("audio/basic", "snd")
        addMapping("application/pkix-pkipath", "pkipath")
        addMapping("application/vnd.zul", "zir")
        addMapping("application/vnd.zul", "zirz")
        addMapping("application/vnd.xmpie.cpkg", "cpkg")
        addMapping("application/LXF", "lxf")
        addMapping("application/mac-binhex40", "hqx")
        addMapping("image/jp2", "jp2")
        addMapping("image/jp2", "jpg2")
        addMapping("application/vnd.llamagraphics.life-balance.desktop", "lbd")
        addMapping("application/vnd.sealed.tiff", "stif")
        addMapping("application/vnd.sun.xml.draw", "sxd")
        addMapping("application/vnd.cendio.thinlinc.clientconf", "tlclient")
        addMapping("application/rpki-ghostbusters", "gbr")
        addMapping("application/vnd.wolfram.player", "nbp")
        addMapping("application/xcap-error+xml", "xer")
        addMapping("video/vnd.nokia.interleaved-multimedia", "nim")
        addMapping("application/tamp-community-update-confirm", "cuc")
        addMapping("application/vnd.fujixerox.ddd", "ddd")
        addMapping("application/simple-filter+xml", "cl")
        addMapping("application/vnd.oasis.opendocument.graphics", "odg")
        addMapping("application/pskc+xml", "pskcxml")
        addMapping("audio/x-ms-wax", "wax")
        addMapping("image/dicom-rle", "drle")
        addMapping("application/vnd.gmx", "gmx")
        addMapping("application/its+xml", "its")
        addMapping("application/vnd.trueapp", "tra")
        addMapping("application/vnd.dpgraph", "dpg")
        addMapping("application/vnd.dpgraph", "mwc")
        addMapping("application/vnd.dpgraph", "dpgraph")
        addMapping("application/bacnet-xdd+zip", "xdd")
        addMapping("application/vnd.sealed.doc", "sdoc")
        addMapping("application/vnd.sealed.doc", "sdo")
        addMapping("application/vnd.sealed.doc", "s1w")
        addMapping("video/x-ms-wmv", "wmv")
        addMapping("video/x-ms-wmx", "wmx")
        addMapping("application/rls-services+xml", "rs")
        addMapping("application/vnd.mynfc", "taglet")
        addMapping("text/vnd.wap.wml", "wml")
        addMapping("text/vnd.DMClientScript", "dms")
        addMapping("application/vnd.desmume.movie", "dsm")
        addMapping("model/mesh", "msh")
        addMapping("model/mesh", "mesh")
        addMapping("model/mesh", "silo")
        addMapping("application/vnd.oasis.opendocument.chart", "odc")
        addMapping("image/jpeg", "jpg")
        addMapping("image/jpeg", "jpeg")
        addMapping("image/jpeg", "jpe")
        addMapping("image/jpeg", "jfif")
        addMapping("application/mmt-usd+xml", "musd")
        addMapping("font/collection", "ttc")
        addMapping("application/vnd.coffeescript", "coffee")
        addMapping("image/tiff-fx", "tfx")
        addMapping("application/vnd.wv.csp+wbxml", "wv")
        addMapping("video/x-flv", "flv")
        addMapping("application/vnd.openofficeorg.extension", "oxt")
        addMapping("application/vnd.rar", "rar")
        addMapping("application/rfc+xml", "rfcxml")
        addMapping("message/global-headers", "u8hdr")
        addMapping("application/vnd.oasis.opendocument.spreadsheet", "ods")
        addMapping("application/x-java-archive", "jar")
        addMapping("application/vnd.adobe.xdp+xml", "xdp")
        addMapping("application/vnd.nitf", "nitf")
        addMapping("application/vnd.chemdraw+xml", "cdxml")
        addMapping("application/vnd.openeye.oeb", "oeb")
        addMapping("application/DCD", "dcd")
        addMapping("text/xml-external-parsed-entity", "ent")
        addMapping("application/vnd.crick.clicker.template", "clkt")
        addMapping("application/cdmi-domain", "cdmid")
        addMapping("application/vnd.wfa.wsc", "wsc")
        addMapping("application/marcxml+xml", "mrcx")
        addMapping("application/vnd.ms-powerpoint.template.macroEnabled.12", "potm")
        addMapping("application/emma+xml", "emma")
        addMapping("audio/vnd.nuera.ecelp9600", "ecelp9600")
        addMapping("application/dicom", "dcm")
        addMapping("application/vnd.zzazz.deck+xml", "zaz")
        addMapping("application/vnd.preminet", "preminet")
        addMapping("application/vnd.ms-fontobject", "eot")
        addMapping("application/hta", "hta")
        addMapping("application/vnd.visio", "vsd")
        addMapping("video/dv", "dif")
        addMapping("video/dv", "dv")
        addMapping("image/x-ms-bmp", "bmp")
        addMapping("audio/midi", "mid")
        addMapping("audio/midi", "midi")
        addMapping("audio/midi", "ota")
        addMapping("audio/midi", "kar")
        addMapping("audio/midi", "rtttl")
        addMapping("audio/midi", "xmf")
        addMapping("application/x-iso9660-image", "iso")
        addMapping("application/x-kpresenter", "kpr")
        addMapping("application/x-kpresenter", "kpt")
        addMapping("image/x-rgb", "rgb")
        addMapping("application/vnd.oasis.opendocument.spreadsheet-template", "ots")
        addMapping("audio/imelody", "imy")
        addMapping("application/x-futuresplash", "spl")
        addMapping("video/avi", "avi")
        addMapping("application/pdf", "pdf")
        addMapping("application/x-quicktimeplayer", "qtl")
        addMapping("audio/prs.sid", "sid")
        addMapping("application/x-bittorrent", "torrent")
        addMapping("video/vnd.mpegurl", "mxu")
        addMapping("text/xml", "xml")
        addMapping("application/rss+xml", "rss")
        addMapping("application/x-abiword", "abw")
        addMapping("text/iuls", "uls")
        addMapping("application/x-pkcs12", "p12")
        addMapping("application/x-pkcs12", "pfx")
        addMapping("application/vnd.stardivision.draw", "sda")
        addMapping("image/x-coreldrawpattern", "pat")
        addMapping("text/html", "htm")
        addMapping("text/html", "html")
        addMapping("image/x-xpixmap", "xpm")
        addMapping("video/dl", "dl")
        addMapping("text/css", "css")
        addMapping("text/x-dsrc", "d")
        addMapping("video/quicktime", "qt")
        addMapping("video/quicktime", "mov")
        addMapping("video/mp4", "mp4")
        addMapping("application/vnd.sun.xml.writer.template", "stw")
        addMapping("application/pgp-keys", "key")
        addMapping("text/x-c++hdr", "hpp")
        addMapping("text/x-c++hdr", "h++")
        addMapping("text/x-c++hdr", "hxx")
        addMapping("text/x-c++hdr", "hh")
        addMapping("video/webm", "webm")
        addMapping("audio/x-ms-wma", "wma")
        addMapping("image/vnd.djvu", "djvu")
        addMapping("image/vnd.djvu", "djv")
        addMapping("application/x-wais-source", "src")
        addMapping("image/x-photoshop", "psd")
        addMapping("video/x-ms-wm", "wm")
        addMapping("application/x-mif", "mif")
        addMapping("text/x-bibtex", "bib")
        addMapping("application/vnd.sun.xml.calc.template", "stc")
        addMapping("audio/x-scpls", "pls")
        addMapping("application/msaccess", "mdb")
        addMapping("application/x-dvi", "dvi")
        addMapping("application/x-latex", "latex")
        addMapping("video/3gpp", "3gpp")
        addMapping("video/3gpp", "3gp")
        addMapping("application/x-chess-pgn", "pgn")
        addMapping("x-epoc/x-sisx-app", "sisx")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.template", "xltx")
        addMapping("application/vnd.sun.xml.writer.global", "sxg")
        addMapping("application/x-cdf", "cdf")
        addMapping("video/mpeg", "mpeg")
        addMapping("video/mpeg", "mpg")
        addMapping("video/mpeg", "mpe")
        addMapping("video/mpeg", "VOB")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", "xlsx")
        addMapping("image/ico", "cur")
        addMapping("image/ico", "ico")
        addMapping("image/x-panasonic-rw2", "rw2")
        addMapping("application/vnd.stardivision.impress", "sdd")
        addMapping("application/vnd.stardivision.impress", "sdp")
        addMapping("application/oda", "oda")
        addMapping("application/zip", "zip")
        addMapping("application/vnd.oasis.opendocument.text-master", "odm")
        addMapping("text/x-boo", "boo")
        addMapping("text/tab-separated-values", "tsv")
        addMapping("x-conference/x-cooltalk", "ice")
        addMapping("image/gif", "gif")
        addMapping("application/x-msi", "msi")
        addMapping("image/webp", "webp")
        addMapping("audio/x-realaudio", "ra")
        addMapping("text/h323", "323")
        addMapping("application/andrew-inset", "ez")
        addMapping("application/x-x509-server-cert", "crt")
        addMapping("application/vnd.stardivision.math", "smf")
        addMapping("application/x-killustrator", "kil")
        addMapping("application/vnd.google-earth.kmz", "kmz")
        addMapping("image/x-samsung-srw", "srw")
        addMapping("video/ogg", "ogv")
        addMapping("audio/x-aiff", "aif")
        addMapping("audio/x-aiff", "aiff")
        addMapping("audio/x-aiff", "aifc")
        addMapping("text/rtf", "rtf")
        addMapping("application/msword", "doc")
        addMapping("application/msword", "dot")
        addMapping("audio/mpegurl", "m3u")
        addMapping("application/vnd.stardivision.writer-global", "sgl")
        addMapping("application/pkix-cert", "cer")
        addMapping("application/x-apple-diskimage", "dmg")
        addMapping("video/x-ms-wvx", "wvx")
        addMapping("text/x-tcl", "tcl")
        addMapping("audio/amr-wb", "awb")
        addMapping("application/vnd.oasis.opendocument.text", "odt")
        addMapping("application/x-bcpio", "bcpio")
        addMapping("application/x-koan", "skp")
        addMapping("application/x-koan", "skd")
        addMapping("application/x-koan", "skt")
        addMapping("application/x-koan", "skm")
        addMapping("audio/x-sd2", "sd2")
        addMapping("image/x-portable-pixmap", "ppm")
        addMapping("text/x-pcs-gcd", "gcd")
        addMapping("application/x-xfig", "fig")
        addMapping("application/x-pkcs7-certreqresp", "p7r")
        addMapping("image/vnd.wap.wbmp", "wbmp")
        addMapping("application/x-nwc", "nwc")
        addMapping("application/x-pkcs7-crl", "crl")
        addMapping("text/x-csh", "csh")
        addMapping("application/x-director", "dcr")
        addMapping("application/x-director", "dir")
        addMapping("application/x-director", "dxr")
        addMapping("application/x-kspread", "ksp")
        addMapping("audio/x-pn-realaudio", "ra")
        addMapping("audio/x-pn-realaudio", "rm")
        addMapping("audio/x-pn-realaudio", "ram")
        addMapping("text/plain", "txt")
        addMapping("text/plain", "asc")
        addMapping("text/plain", "text")
        addMapping("text/plain", "diff")
        addMapping("text/plain", "po")
        addMapping("image/png", "png")
        addMapping("application/x-jmol", "jmz")
        addMapping("image/x-corelphotopaint", "cpt")
        addMapping("image/x-xwindowdump", "xwd")
        addMapping("application/x-cpio", "cpio")
        addMapping("video/x-sgi-movie", "movie")
        addMapping("application/x-font", "pfa")
        addMapping("application/x-font", "pfb")
        addMapping("application/x-font", "gsf")
        addMapping("application/x-font", "pcf")
        addMapping("application/x-font", "pcf.Z")
        addMapping("audio/aac-adts", "aac")
        addMapping("audio/ogg", "oga")
        addMapping("audio/ogg", "ogg")
        addMapping("audio/ogg", "spx")
        addMapping("application/vnd.oasis.opendocument.presentation-template", "otp")
        addMapping("application/x-tar", "tar")
        addMapping("application/vnd.sun.xml.impress.template", "sti")
        addMapping("text/x-tex", "tex")
        addMapping("text/x-tex", "ltx")
        addMapping("text/x-tex", "sty")
        addMapping("text/x-tex", "cls")
        addMapping("application/vnd.oasis.opendocument.formula", "odf")
        addMapping("application/pgp-signature", "pgp")
        addMapping("image/x-coreldraw", "cdr")
        addMapping("image/x-portable-graymap", "pgm")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.template", "dotx")
        addMapping("image/x-coreldrawtemplate", "cdt")
        addMapping("application/vnd.youtube.yt", "yt")
        addMapping("application/x-troff", "t")
        addMapping("application/x-troff", "roff")
        addMapping("application/x-webarchive-xml", "webarchivexml")
        addMapping("application/vnd.sun.xml.draw.template", "std")
        addMapping("application/x-shockwave-flash", "swf")
        addMapping("image/x-portable-bitmap", "pbm")
        addMapping("text/x-java", "java")
        addMapping("application/x-dms", "dms")
        addMapping("application/x-ustar", "ustar")
        addMapping("application/vnd.stardivision.calc", "sdc")
        addMapping("application/vnd.ms-excel", "xls")
        addMapping("application/vnd.ms-excel", "xlt")
        addMapping("text/x-vcalendar", "vcs")
        addMapping("application/x-sv4cpio", "sv4cpio")
        addMapping("application/x-troff-man", "man")
        addMapping("application/x-xcf", "xcf")
        addMapping("application/epub+zip", "epub")
        addMapping("audio/amr", "amr")
        addMapping("application/vnd.oasis.opendocument.text-template", "ott")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.presentation", "pptx")
        addMapping("audio/x-matroska", "mka")
        addMapping("application/x-hdf", "hdf")
        addMapping("application/x-texinfo", "texinfo")
        addMapping("application/x-texinfo", "texi")
        addMapping("application/vnd.oasis.opendocument.database", "odb")
        addMapping("image/x-fuji-raf", "raf")
        addMapping("video/m4v", "m4v")
        addMapping("application/x-oz-application", "oza")
        addMapping("text/x-literate-haskell", "lhs")
        addMapping("text/calendar", "ics")
        addMapping("text/calendar", "icz")
        addMapping("text/mathml", "mml")
        addMapping("application/vnd.sun.xml.math", "sxm")
        addMapping("image/bmp", "bmp")
        addMapping("image/x-jng", "jng")
        addMapping("audio/mobile-xmf", "mxmf")
        addMapping("text/text", "phps")
        addMapping("application/vnd.cinderella", "cdy")
        addMapping("video/mp2ts", "ts")
        addMapping("application/x-hwp", "hwp")
        addMapping("application/xhtml+xml", "xhtml")
        addMapping("text/x-component", "htc")
        addMapping("application/vnd.smaf", "mmf")
        addMapping("image/x-cmu-raster", "ras")
        addMapping("application/vnd.rim.cod", "cod")
        addMapping("text/comma-separated-values", "csv")
        addMapping("application/x-iphone", "iii")
        addMapping("application/vnd.sun.xml.writer", "sxw")
        addMapping("audio/basic", "snd")
        addMapping("image/x-sony-arw", "arw")
        addMapping("image/x-adobe-dng", "dng")
        addMapping("text/x-chdr", "h")
        addMapping("application/x-internet-signup", "ins")
        addMapping("application/x-internet-signup", "isp")
        addMapping("image/x-canon-cr2", "cr2")
        addMapping("application/vnd.ms-powerpoint", "ppt")
        addMapping("application/vnd.ms-powerpoint", "pot")
        addMapping("application/vnd.ms-powerpoint", "pps")
        addMapping("image/x-olympus-orf", "orf")
        addMapping("application/mac-binhex40", "hqx")
        addMapping("text/x-c++src", "cpp")
        addMapping("text/x-c++src", "c++")
        addMapping("text/x-c++src", "cc")
        addMapping("text/x-c++src", "cxx")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.template", "potx")
        addMapping("application/rar", "rar")
        addMapping("image/tiff", "tiff")
        addMapping("image/tiff", "tif")
        addMapping("image/x-pentax-pef", "pef")
        addMapping("application/x-stuffit", "sit")
        addMapping("application/vnd.sun.xml.draw", "sxd")
        addMapping("application/x-gtar", "tgz")
        addMapping("application/x-gtar", "gtar")
        addMapping("application/x-gtar", "taz")
        addMapping("application/x-ms-wmd", "wmd")
        addMapping("image/ief", "ief")
        addMapping("application/vnd.stardivision.writer", "sdw")
        addMapping("application/vnd.stardivision.writer", "vor")
        addMapping("video/x-matroska", "mkv")
        addMapping("application/x-wingz", "wz")
        addMapping("application/x-object", "o")
        addMapping("application/vnd.sun.xml.calc", "sxc")
        addMapping("image/x-xbitmap", "xbm")
        addMapping("application/vnd.oasis.opendocument.graphics-template", "otg")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.document", "docx")
        addMapping("application/x-ica", "ica")
        addMapping("application/x-cdlink", "vcd")
        addMapping("application/x-go-sgf", "sgf")
        addMapping("image/x-nikon-nrw", "nrw")
        addMapping("text/x-csrc", "c")
        addMapping("application/vnd.oasis.opendocument.text-web", "oth")
        addMapping("application/x-ms-wmz", "wmz")
        addMapping("video/x-la-asf", "lsf")
        addMapping("video/x-la-asf", "lsx")
        addMapping("audio/x-wav", "wav")
        addMapping("application/x-maker", "frm")
        addMapping("application/x-maker", "maker")
        addMapping("application/x-maker", "frame")
        addMapping("application/x-maker", "fb")
        addMapping("application/x-maker", "book")
        addMapping("application/x-maker", "fbdoc")
        addMapping("application/x-x509-user-cert", "crt")
        addMapping("text/x-setext", "etx")
        addMapping("application/vnd.oasis.opendocument.graphics", "odg")
        addMapping("application/x-webarchive", "webarchive")
        addMapping("audio/x-ms-wax", "wax")
        addMapping("image/pcx", "pcx")
        addMapping("video/x-ms-asf", "asf")
        addMapping("video/x-ms-asf", "asx")
        addMapping("application/rdf+xml", "rdf")
        addMapping("application/x-sv4crc", "sv4crc")
        addMapping("application/vnd.google-earth.kml+xml", "kml")
        addMapping("video/3gpp2", "3gpp2")
        addMapping("video/3gpp2", "3g2")
        addMapping("video/x-mng", "mng")
        addMapping("application/dsptype", "tsp")
        addMapping("audio/aac", "aac")
        addMapping("video/fli", "fli")
        addMapping("audio/flac", "flac")
        addMapping("audio/mpeg", "mp3")
        addMapping("audio/mpeg", "mpga")
        addMapping("audio/mpeg", "mpega")
        addMapping("audio/mpeg", "mp2")
        addMapping("audio/mpeg", "m4a")
        addMapping("application/x-lha", "lha")
        addMapping("text/x-pascal", "p")
        addMapping("text/x-pascal", "pas")
        addMapping("video/x-ms-wmv", "wmv")
        addMapping("application/x-doom", "wad")
        addMapping("audio/x-gsm", "gsm")
        addMapping("video/x-ms-wmx", "wmx")
        addMapping("application/vnd.oasis.opendocument.presentation", "odp")
        addMapping("application/x-x509-ca-cert", "crt")
        addMapping("audio/3gpp", "3gpp")
        addMapping("image/x-jg", "art")
        addMapping("model/iges", "igs")
        addMapping("model/iges", "iges")
        addMapping("text/richtext", "rtx")
        addMapping("text/x-moc", "moc")
        addMapping("audio/x-mpegurl", "m3u")
        addMapping("model/mesh", "msh")
        addMapping("model/mesh", "mesh")
        addMapping("model/mesh", "silo")
        addMapping("image/jpeg", "jpg")
        addMapping("image/jpeg", "jpeg")
        addMapping("image/jpeg", "jpe")
        addMapping("application/ogg", "ogx")
        addMapping("image/x-icon", "ico")
        addMapping("application/x-ns-proxy-autoconfig", "pac")
        addMapping("image/svg+xml", "svg")
        addMapping("image/svg+xml", "svgz")
        addMapping("application/x-shar", "shar")
        addMapping("application/x-debian-package", "deb")
        addMapping("application/x-debian-package", "udeb")
        addMapping("application/vnd.sun.xml.impress", "sxi")
        addMapping("video/x-webex", "wrf")
        addMapping("application/vnd.ms-pki.stl", "stl")
        addMapping("application/vnd.oasis.opendocument.spreadsheet", "ods")
        addMapping("application/pics-rules", "prf")
        addMapping("application/mathematica", "nb")
        addMapping("application/vnd.oasis.opendocument.image", "odi")
        addMapping("application/x-lzh", "lzh")
        addMapping("application/x-pem-file", "pem")
        addMapping("text/x-haskell", "hs")
        addMapping("text/x-vcard", "vcf")
        addMapping("application/vnd.android.package-archive", "apk")
        addMapping("application/x-kchart", "chrt")
        addMapping("application/x-flac", "flac")
        addMapping("application/futuresplash", "spl")
        addMapping("application/x-gnumeric", "gnumeric")
        addMapping("application/x-freemind", "mm")
        addMapping("application/x-lzx", "lzx")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slideshow", "ppsx")
        addMapping("image/x-nikon-nef", "nef")
        addMapping("application/x-graphing-calculator", "gcf")
        addMapping("image/x-portable-anymap", "pnm")
        addMapping("application/x-kword", "kwd")
        addMapping("application/x-kword", "kwt")
        addMapping("application/powerpoint", "ppt")
        addMapping("application/freeloader", "frl")
        addMapping("audio/midi", "kar")
        addMapping("audio/midi", "mid")
        addMapping("audio/midi", "midi")
        addMapping("application/vnd.rn-realplayer", "rnx")
        addMapping("application/vnd.xara", "web")
        addMapping("application/x-ksh", "ksh")
        addMapping("image/x-pcx", "pcx")
        addMapping("image/x-rgb", "rgb")
        addMapping("video/x-mpeq2a", "mp2")
        addMapping("application/x-livescreen", "ivy")
        addMapping("video/x-ms-asf-plugin", "asx")
        addMapping("application/x-pagemaker", "pm4")
        addMapping("application/x-pagemaker", "pm5")
        addMapping("application/x-visio", "vsd")
        addMapping("application/x-visio", "vst")
        addMapping("application/x-visio", "vsw")
        addMapping("video/avi", "avi")
        addMapping("application/x-midi", "mid")
        addMapping("application/x-midi", "midi")
        addMapping("text/xml", "xml")
        addMapping("application/x-netcdf", "cdf")
        addMapping("application/x-netcdf", "nc")
        addMapping("application/x-tcl", "tcl")
        addMapping("audio/aiff", "aif")
        addMapping("audio/aiff", "aifc")
        addMapping("audio/aiff", "aiff")
        addMapping("audio/it", "it")
        addMapping("application/x-pkcs7-mime", "p7c")
        addMapping("application/x-pkcs7-mime", "p7m")
        addMapping("application/x-freelance", "pre")
        addMapping("text/html", "acgi")
        addMapping("text/html", "htm")
        addMapping("text/html", "html")
        addMapping("text/html", "htmls")
        addMapping("text/html", "htx")
        addMapping("text/html", "shtml")
        addMapping("application/x-ip2", "ip")
        addMapping("audio/voc", "voc")
        addMapping("image/x-xpixmap", "pm")
        addMapping("image/x-xpixmap", "xpm")
        addMapping("video/quicktime", "moov")
        addMapping("video/quicktime", "mov")
        addMapping("video/quicktime", "qt")
        addMapping("application/x-lotusscreencam", "scm")
        addMapping("application/x-navimap", "map")
        addMapping("application/x-sh", "sh")
        addMapping("application/x-sdp", "sdp")
        addMapping("application/x-wais-source", "src")
        addMapping("application/x-wais-source", "wsrc")
        addMapping("text/x-script.ksh", "ksh")
        addMapping("text/x-script.perl", "pl")
        addMapping("application/binhex4", "hqx")
        addMapping("application/x-sea", "sea")
        addMapping("audio/x-twinvq-plugin", "vqe")
        addMapping("audio/x-twinvq-plugin", "vql")
        addMapping("audio/x-midi", "mid")
        addMapping("audio/x-midi", "midi")
        addMapping("application/mac-compactpro", "cpt")
        addMapping("text/x-fortran", "f")
        addMapping("text/x-fortran", "f77")
        addMapping("text/x-fortran", "f90")
        addMapping("text/x-fortran", "for")
        addMapping("text/x-speech", "spc")
        addMapping("text/x-speech", "talk")
        addMapping("application/x-pointplus", "css")
        addMapping("music/x-karaoke", "kar")
        addMapping("text/x-m", "m")
        addMapping("text/x-h", "h")
        addMapping("text/x-h", "hh")
        addMapping("application/x-tex", "tex")
        addMapping("text/x-c", "c")
        addMapping("text/x-c", "cc")
        addMapping("text/x-c", "cpp")
        addMapping("application/x-cdf", "cdf")
        addMapping("application/vnd.ms-pki.pko", "pko")
        addMapping("video/x-mpeg", "mp2")
        addMapping("video/x-mpeg", "mp3")
        addMapping("text/x-script.tcl", "tcl")
        addMapping("application/acad", "dwg")
        addMapping("application/zip", "zip")
        addMapping("application/dxf", "dxf")
        addMapping("x-conference/x-cooltalk", "ice")
        addMapping("application/x-project", "mpc")
        addMapping("application/x-project", "mpt")
        addMapping("application/x-project", "mpv")
        addMapping("application/x-project", "mpx")
        addMapping("application/arj", "arj")
        addMapping("text/x-uuencode", "uu")
        addMapping("text/x-uuencode", "uue")
        addMapping("audio/x-realaudio", "ra")
        addMapping("video/vdo", "vdo")
        addMapping("application/vnd.fdf", "fdf")
        addMapping("application/x-vnd.audioexplosion.mzz", "mzz")
        addMapping("text/x-script.guile", "scm")
        addMapping("application/x-macbinary", "bin")
        addMapping("text/javascript", "js")
        addMapping("application/mac-binhex", "hqx")
        addMapping("audio/x-aiff", "aif")
        addMapping("audio/x-aiff", "aifc")
        addMapping("audio/x-aiff", "aiff")
        addMapping("video/x-amt-demorun", "xdr")
        addMapping("application/x-navidoc", "nvd")
        addMapping("image/vnd.fpx", "fpx")
        addMapping("video/x-isvideo", "isu")
        addMapping("application/x-javascript", "js")
        addMapping("application/pkix-cert", "cer")
        addMapping("application/pkix-cert", "crt")
        addMapping("application/vnd.ms-pki.certstore", "sst")
        addMapping("application/envoy", "evy")
        addMapping("application/x-sprite", "spr")
        addMapping("application/x-sprite", "sprite")
        addMapping("application/x-authorware-bin", "aab")
        addMapping("application/lha", "lha")
        addMapping("application/rtf", "rtf")
        addMapping("application/rtf", "rtx")
        addMapping("application/java-byte-code", "class")
        addMapping("image/x-portable-pixmap", "ppm")
        addMapping("application/x-msexcel", "xla")
        addMapping("application/x-msexcel", "xls")
        addMapping("application/x-msexcel", "xlw")
        addMapping("chemical/x-pdb", "pdb")
        addMapping("chemical/x-pdb", "xyz")
        addMapping("application/x-pixclscript", "plx")
        addMapping("multipart/x-zip", "zip")
        addMapping("application/x-pkcs7-certreqresp", "p7r")
        addMapping("application/binhex", "hqx")
        addMapping("application/x-troff-msvideo", "avi")
        addMapping("application/base64", "mm")
        addMapping("application/base64", "mme")
        addMapping("application/x-ima", "ima")
        addMapping("text/uri-list", "uni")
        addMapping("text/uri-list", "unis")
        addMapping("text/uri-list", "uri")
        addMapping("text/uri-list", "uris")
        addMapping("image/vnd.rn-realflash", "rf")
        addMapping("application/x-authorware-map", "aam")
        addMapping("text/vnd.abc", "abc")
        addMapping("application/x-director", "dcr")
        addMapping("application/x-director", "dir")
        addMapping("application/x-director", "dxr")
        addMapping("application/vocaltec-media-desc", "vmd")
        addMapping("image/pjpeg", "jfif")
        addMapping("image/pjpeg", "jpe")
        addMapping("image/pjpeg", "jpeg")
        addMapping("image/pjpeg", "jpg")
        addMapping("audio/nspaudio", "la")
        addMapping("audio/nspaudio", "lma")
        addMapping("application/x-inventor", "iv")
        addMapping("image/png", "png")
        addMapping("image/png", "x-png")
        addMapping("application/x-pkcs7-certificates", "spc")
        addMapping("audio/x-liveaudio", "lam")
        addMapping("text/x-script.zsh", "zsh")
        addMapping("application/x-omcdatamaker", "omcd")
        addMapping("application/pkix-crl", "crl")
        addMapping("image/x-xwd", "xwd")
        addMapping("image/x-xwindowdump", "xwd")
        addMapping("application/x-cpio", "cpio")
        addMapping("application/drafting", "drw")
        addMapping("video/x-sgi-movie", "movie")
        addMapping("video/x-sgi-movie", "mv")
        addMapping("application/x-tar", "tar")
        addMapping("application/mime", "aps")
        addMapping("application/streamingmedia", "ssm")
        addMapping("application/mcad", "mcd")
        addMapping("text/x-script.sh", "sh")
        addMapping("application/x-compactpro", "cpt")
        addMapping("audio/tsplayer", "tsp")
        addMapping("application/x-binary", "bin")
        addMapping("image/x-portable-graymap", "pgm")
        addMapping("application/x-mspowerpoint", "ppt")
        addMapping("image/x-niff", "nif")
        addMapping("image/x-niff", "niff")
        addMapping("application/x-shockwave-flash", "swf")
        addMapping("application/x-wpwin", "wpd")
        addMapping("x-world/x-svr", "svr")
        addMapping("text/sgml", "sgm")
        addMapping("text/sgml", "sgml")
        addMapping("application/x-ustar", "ustar")
        addMapping("application/book", "boo")
        addMapping("application/book", "book")
        addMapping("audio/x-psid", "sid")
        addMapping("application/vnd.ms-project", "mpp")
        addMapping("i-world/i-vrml", "ivr")
        addMapping("audio/x-vnd.audioexplosion.mjuicemediafile", "mjf")
        addMapping("video/vivo", "viv")
        addMapping("video/vivo", "vivo")
        addMapping("image/vasa", "mcf")
        addMapping("application/x-hdf", "hdf")
        addMapping("audio/make", "funk")
        addMapping("audio/make", "my")
        addMapping("audio/make", "pfunk")
        addMapping("application/x-texinfo", "texi")
        addMapping("application/x-texinfo", "texinfo")
        addMapping("application/marc", "mrc")
        addMapping("application/x-envoy", "env")
        addMapping("application/x-envoy", "evy")
        addMapping("application/netmc", "mcp")
        addMapping("image/bmp", "bm")
        addMapping("image/bmp", "bmp")
        addMapping("text/x-script.phyton", "py")
        addMapping("application/x-esrehber", "es")
        addMapping("application/step", "step")
        addMapping("application/step", "stp")
        addMapping("text/x-asm", "asm")
        addMapping("text/x-asm", "s")
        addMapping("image/x-cmu-raster", "ras")
        addMapping("application/x-deepv", "deepv")
        addMapping("text/vnd.rn-realtext", "rt")
        addMapping("application/vnd.ms-powerpoint", "pot")
        addMapping("application/vnd.ms-powerpoint", "ppa")
        addMapping("application/vnd.ms-powerpoint", "pps")
        addMapping("application/vnd.ms-powerpoint", "ppt")
        addMapping("application/vnd.ms-powerpoint", "pwz")
        addMapping("application/gnutar", "tgz")
        addMapping("image/tiff", "tif")
        addMapping("image/tiff", "tiff")
        addMapping("application/x-stuffit", "sit")
        addMapping("video/animaflex", "afl")
        addMapping("text/x-script.perl-module", "pm")
        addMapping("application/fractals", "fif")
        addMapping("audio/x-twinvq", "vqf")
        addMapping("image/x-xbitmap", "xbm")
        addMapping("application/x-java-class", "class")
        addMapping("application/vnd.rn-realmedia", "rm")
        addMapping("audio/x-wav", "wav")
        addMapping("application/toolbook", "tbk")
        addMapping("image/florian", "flo")
        addMapping("image/florian", "turbot")
        addMapping("application/x-bzip", "bz")
        addMapping("application/x-x509-user-cert", "crt")
        addMapping("text/x-setext", "etx")
        addMapping("application/octet-stream", "a")
        addMapping("application/octet-stream", "arc")
        addMapping("application/octet-stream", "arj")
        addMapping("application/octet-stream", "bin")
        addMapping("application/octet-stream", "com")
        addMapping("application/octet-stream", "dump")
        addMapping("application/octet-stream", "exe")
        addMapping("application/octet-stream", "lha")
        addMapping("application/octet-stream", "lhx")
        addMapping("application/octet-stream", "lzh")
        addMapping("application/octet-stream", "lzx")
        addMapping("application/octet-stream", "o")
        addMapping("application/octet-stream", "psd")
        addMapping("application/octet-stream", "saveme")
        addMapping("application/octet-stream", "uu")
        addMapping("application/octet-stream", "zoo")
        addMapping("video/x-ms-asf", "asf")
        addMapping("video/x-ms-asf", "asx")
        addMapping("application/x-sv4crc", "sv4crc")
        addMapping("application/dsptype", "tsp")
        addMapping("audio/mpeg", "m2a")
        addMapping("audio/mpeg", "mp2")
        addMapping("audio/mpeg", "mpa")
        addMapping("audio/mpeg", "mpg")
        addMapping("audio/mpeg", "mpga")
        addMapping("audio/make.my.funk", "pfunk")
        addMapping("application/x-lha", "lha")
        addMapping("text/mcf", "mcf")
        addMapping("text/x-pascal", "p")
        addMapping("audio/x-mod", "mod")
        addMapping("application/x-rtf", "rtf")
        addMapping("application/x-mathcad", "mcd")
        addMapping("image/x-pict", "pct")
        addMapping("image/x-jg", "art")
        addMapping("application/vnd.hp-hpgl", "hgl")
        addMapping("application/vnd.hp-hpgl", "hpg")
        addMapping("application/vnd.hp-hpgl", "hpgl")
        addMapping("application/x-world", "svr")
        addMapping("application/x-world", "wrl")
        addMapping("audio/x-nspaudio", "la")
        addMapping("audio/x-nspaudio", "lma")
        addMapping("image/x-icon", "ico")
        addMapping("model/vnd.dwf", "dwf")
        addMapping("application/x-compressed", "gz")
        addMapping("application/x-compressed", "tgz")
        addMapping("application/x-compressed", "z")
        addMapping("application/x-compressed", "zip")
        addMapping("audio/vnd.qcelp", "qcp")
        addMapping("application/mac-binary", "bin")
        addMapping("image/jutvision", "jut")
        addMapping("application/x-portable-anymap", "pnm")
        addMapping("application/x-authorware-seg", "aas")
        addMapping("application/x-omcregerator", "omcr")
        addMapping("video/vnd.vivo", "viv")
        addMapping("video/vnd.vivo", "vivo")
        addMapping("application/x-chat", "cha")
        addMapping("application/x-chat", "chat")
        addMapping("audio/x-mpeg", "mp2")
        addMapping("application/x-bzip2", "boz")
        addMapping("application/x-bzip2", "bz2")
        addMapping("video/x-qtc", "qtc")
        addMapping("audio/xm", "xm")
        addMapping("application/vocaltec-media-file", "vmf")
        addMapping("application/x-winhelp", "hlp")
        addMapping("audio/x-pn-realaudio-plugin", "ra")
        addMapping("audio/x-pn-realaudio-plugin", "rmp")
        addMapping("audio/x-pn-realaudio-plugin", "rpm")
        addMapping("audio/x-mpequrl", "m3u")
        addMapping("audio/x-au", "au")
        addMapping("image/x-dwg", "dwg")
        addMapping("image/x-dwg", "dxf")
        addMapping("image/x-dwg", "svf")
        addMapping("image/x-portable-anymap", "pnm")
        addMapping("application/wordperfect6.0", "w60")
        addMapping("application/wordperfect6.0", "wp5")
        addMapping("application/wordperfect6.1", "w61")
        addMapping("application/hta", "hta")
        addMapping("text/x-script.lisp", "lsp")
        addMapping("message/rfc822", "mht")
        addMapping("message/rfc822", "mhtml")
        addMapping("message/rfc822", "mime")
        addMapping("application/x-csh", "csh")
        addMapping("video/x-msvideo", "avi")
        addMapping("application/x-helpfile", "help")
        addMapping("application/x-helpfile", "hlp")
        addMapping("application/x-pkcs10", "p10")
        addMapping("audio/wav", "wav")
        addMapping("application/ecmascript", "js")
        addMapping("image/vnd.rn-realpix", "rp")
        addMapping("image/vnd.xiff", "xif")
        addMapping("audio/x-mpeg-3", "mp3")
        addMapping("application/pdf", "pdf")
        addMapping("text/webviewhtml", "htt")
        addMapping("application/x-internett-signup", "ins")
        addMapping("application/vnd.wap.wmlc", "wmlc")
        addMapping("application/x-pkcs12", "p12")
        addMapping("text/x-script.elisp", "el")
        addMapping("video/x-fli", "fli")
        addMapping("application/pkcs-crl", "crl")
        addMapping("application/pkcs10", "p10")
        addMapping("text/x-sgml", "sgm")
        addMapping("text/x-sgml", "sgml")
        addMapping("video/dl", "dl")
        addMapping("text/css", "css")
        addMapping("application/sounder", "sdr")
        addMapping("application/x-bsh", "bsh")
        addMapping("application/x-bsh", "sh")
        addMapping("application/x-bsh", "shar")
        addMapping("application/pkcs-12", "p12")
        addMapping("application/x-troff-ms", "ms")
        addMapping("text/x-script.csh", "csh")
        addMapping("application/x-omc", "omc")
        addMapping("model/vrml", "vrml")
        addMapping("model/vrml", "wrl")
        addMapping("model/vrml", "wrz")
        addMapping("application/x-mif", "mif")
        addMapping("image/xpm", "xpm")
        addMapping("application/javascript", "js")
        addMapping("windows/metafile", "wmf")
        addMapping("application/x-troff-me", "me")
        addMapping("image/x-quicktime", "qif")
        addMapping("image/x-quicktime", "qti")
        addMapping("image/x-quicktime", "qtif")
        addMapping("video/gl", "gl")
        addMapping("image/g3fax", "g3")
        addMapping("application/x-dvi", "dvi")
        addMapping("application/x-latex", "latex")
        addMapping("application/x-latex", "ltx")
        addMapping("video/x-scm", "scm")
        addMapping("text/scriplet", "wsc")
        addMapping("application/groupwise", "vew")
        addMapping("application/x-excel", "xla")
        addMapping("application/x-excel", "xlb")
        addMapping("application/x-excel", "xlc")
        addMapping("application/x-excel", "xld")
        addMapping("application/x-excel", "xlk")
        addMapping("application/x-excel", "xll")
        addMapping("application/x-excel", "xlm")
        addMapping("application/x-excel", "xls")
        addMapping("application/x-excel", "xlt")
        addMapping("application/x-excel", "xlv")
        addMapping("application/x-excel", "xlw")
        addMapping("application/x-compress", "z")
        addMapping("image/vnd.dwg", "dwg")
        addMapping("image/vnd.dwg", "dxf")
        addMapping("image/vnd.dwg", "svf")
        addMapping("application/x-seelogo", "sl")
        addMapping("image/x-xbm", "xbm")
        addMapping("audio/mod", "mod")
        addMapping("video/mpeg", "m1v")
        addMapping("video/mpeg", "m2v")
        addMapping("video/mpeg", "mp2")
        addMapping("video/mpeg", "mp3")
        addMapping("video/mpeg", "mpa")
        addMapping("video/mpeg", "mpe")
        addMapping("video/mpeg", "mpeg")
        addMapping("video/mpeg", "mpg")
        addMapping("application/x-navistyle", "stl")
        addMapping("application/oda", "oda")
        addMapping("application/x-123", "wk1")
        addMapping("application/sla", "stl")
        addMapping("text/asp", "asp")
        addMapping("text/tab-separated-values", "tsv")
        addMapping("video/x-amt-showrun", "xsr")
        addMapping("audio/x-adpcm", "snd")
        addMapping("image/gif", "gif")
        addMapping("application/sea", "sea")
        addMapping("video/msvideo", "avi")
        addMapping("text/x-la-asf", "lsx")
        addMapping("application/mspowerpoint", "pot")
        addMapping("application/mspowerpoint", "pps")
        addMapping("application/mspowerpoint", "ppt")
        addMapping("application/mspowerpoint", "ppz")
        addMapping("xgl/drawing", "xgz")
        addMapping("x-world/x-3dmf", "3dm")
        addMapping("x-world/x-3dmf", "3dmf")
        addMapping("x-world/x-3dmf", "qd3")
        addMapping("x-world/x-3dmf", "qd3d")
        addMapping("image/vnd.net-fpx", "fpx")
        addMapping("application/sdp", "sdp")
        addMapping("application/pro_eng", "part")
        addMapping("application/pro_eng", "prt")
        addMapping("video/avs-video", "avs")
        addMapping("application/x-qpro", "wb1")
        addMapping("text/x-audiosoft-intra", "aip")
        addMapping("application/msword", "doc")
        addMapping("application/msword", "dot")
        addMapping("application/msword", "w6w")
        addMapping("application/msword", "wiz")
        addMapping("application/msword", "word")
        addMapping("image/x-windows-bmp", "bmp")
        addMapping("video/vnd.rn-realvideo", "rv")
        addMapping("text/pascal", "pas")
        addMapping("application/commonground", "dp")
        addMapping("application/x-cmu-raster", "ras")
        addMapping("application/x-bcpio", "bcpio")
        addMapping("application/set", "set")
        addMapping("application/x-koan", "skd")
        addMapping("application/x-koan", "skm")
        addMapping("application/x-koan", "skp")
        addMapping("application/x-koan", "skt")
        addMapping("x-music/x-midi", "mid")
        addMapping("x-music/x-midi", "midi")
        addMapping("application/x-mplayer2", "asx")
        addMapping("application/x-lisp", "lsp")
        addMapping("image/vnd.wap.wbmp", "wbmp")
        addMapping("application/x-newton-compatible-pkg", "pkg")
        addMapping("application/x-navi-animation", "ani")
        addMapping("application/x-mac-binhex40", "hqx")
        addMapping("application/x-pkcs7-signature", "p7a")
        addMapping("application/pkcs7-mime", "p7c")
        addMapping("application/pkcs7-mime", "p7m")
        addMapping("xgl/movie", "xmz")
        addMapping("application/iges", "iges")
        addMapping("application/iges", "igs")
        addMapping("model/x-pov", "pov")
        addMapping("audio/x-jam", "jam")
        addMapping("application/x-cocoa", "cco")
        addMapping("application/vnd.hp-pcl", "pcl")
        addMapping("text/x-script.rexx", "rexx")
        addMapping("image/fif", "fif")
        addMapping("multipart/x-ustar", "ustar")
        addMapping("audio/x-pn-realaudio", "ra")
        addMapping("audio/x-pn-realaudio", "ram")
        addMapping("audio/x-pn-realaudio", "rm")
        addMapping("audio/x-pn-realaudio", "rmm")
        addMapping("audio/x-pn-realaudio", "rmp")
        addMapping("text/plain", "c")
        addMapping("text/plain", "c++")
        addMapping("text/plain", "cc")
        addMapping("text/plain", "com")
        addMapping("text/plain", "conf")
        addMapping("text/plain", "cxx")
        addMapping("text/plain", "def")
        addMapping("text/plain", "f")
        addMapping("text/plain", "f90")
        addMapping("text/plain", "for")
        addMapping("text/plain", "g")
        addMapping("text/plain", "h")
        addMapping("text/plain", "hh")
        addMapping("text/plain", "idc")
        addMapping("text/plain", "jav")
        addMapping("text/plain", "java")
        addMapping("text/plain", "list")
        addMapping("text/plain", "log")
        addMapping("text/plain", "lst")
        addMapping("text/plain", "m")
        addMapping("text/plain", "mar")
        addMapping("text/plain", "pl")
        addMapping("text/plain", "sdml")
        addMapping("text/plain", "text")
        addMapping("text/plain", "txt")
        addMapping("x-world/x-vrml", "vrml")
        addMapping("x-world/x-vrml", "wrl")
        addMapping("x-world/x-vrml", "wrz")
        addMapping("application/java", "class")
        addMapping("text/x-server-parsed-html", "shtml")
        addMapping("text/x-server-parsed-html", "ssi")
        addMapping("application/x-cpt", "cpt")
        addMapping("application/x-conference", "nsc")
        addMapping("application/vnd.nokia.configuration-message", "ncm")
        addMapping("text/vnd.wap.wmlscript", "wmls")
        addMapping("application/lzx", "lzx")
        addMapping("audio/x-mid", "mid")
        addMapping("audio/x-mid", "midi")
        addMapping("application/x-wintalk", "wtk")
        addMapping("image/pict", "pic")
        addMapping("image/pict", "pict")
        addMapping("application/plain", "text")
        addMapping("application/x-frame", "mif")
        addMapping("text/x-script", "hlb")
        addMapping("audio/voxware", "vox")
        addMapping("application/x-troff", "roff")
        addMapping("application/x-troff", "t")
        addMapping("application/x-troff", "tr")
        addMapping("application/clariscad", "ccad")
        addMapping("application/x-gss", "gss")
        addMapping("application/x-gsp", "gsp")
        addMapping("image/x-portable-bitmap", "pbm")
        addMapping("multipart/x-gzip", "gzip")
        addMapping("application/vnd.ms-excel", "xlb")
        addMapping("application/vnd.ms-excel", "xlc")
        addMapping("application/vnd.ms-excel", "xll")
        addMapping("application/vnd.ms-excel", "xlm")
        addMapping("application/vnd.ms-excel", "xls")
        addMapping("application/vnd.ms-excel", "xlw")
        addMapping("text/x-vcalendar", "vcs")
        addMapping("application/x-sv4cpio", "sv4cpio")
        addMapping("application/x-troff-man", "man")
        addMapping("video/x-motion-jpeg", "mjpg")
        addMapping("application/x-pcl", "pcl")
        addMapping("www/mime", "mime")
        addMapping("application/x-sit", "sit")
        addMapping("text/x-uil", "uil")
        addMapping("application/solids", "sol")
        addMapping("video/x-atomic3d-feature", "fmf")
        addMapping("application/inf", "inf")
        addMapping("image/x-jps", "jps")
        addMapping("paleovu/x-pv", "pvu")
        addMapping("application/x-wri", "wri")
        addMapping("application/hlp", "hlp")
        addMapping("application/ringing-tones", "rng")
        addMapping("application/smil", "smi")
        addMapping("application/smil", "smil")
        addMapping("application/x-lotus", "wq1")
        addMapping("application/x-magic-cap-package-1.0", "mc$")
        addMapping("application/xml", "xml")
        addMapping("application/x-java-commerce", "jcm")
        addMapping("application/postscript", "ai")
        addMapping("application/postscript", "eps")
        addMapping("application/postscript", "ps")
        addMapping("text/x-component", "htc")
        addMapping("application/x-binhex40", "hqx")
        addMapping("application/x-aim", "aim")
        addMapping("application/pkcs7-signature", "p7s")
        addMapping("text/ecmascript", "js")
        addMapping("image/x-portable-greymap", "pgm")
        addMapping("text/x-java-source", "jav")
        addMapping("text/x-java-source", "java")
        addMapping("application/x-mix-transfer", "nix")
        addMapping("audio/basic", "au")
        addMapping("audio/basic", "snd")
        addMapping("audio/mpeg3", "mp3")
        addMapping("application/mbedlet", "mbd")
        addMapping("image/cmu-raster", "ras")
        addMapping("image/cmu-raster", "rast")
        addMapping("application/mac-binhex40", "hqx")
        addMapping("application/cdf", "cdf")
        addMapping("image/x-tiff", "tif")
        addMapping("image/x-tiff", "tiff")
        addMapping("audio/x-voc", "voc")
        addMapping("application/x-gtar", "gtar")
        addMapping("video/x-gl", "gl")
        addMapping("image/ief", "ief")
        addMapping("image/ief", "iefs")
        addMapping("application/excel", "xl")
        addMapping("application/excel", "xla")
        addMapping("application/excel", "xlb")
        addMapping("application/excel", "xlc")
        addMapping("application/excel", "xld")
        addMapping("application/excel", "xlk")
        addMapping("application/excel", "xll")
        addMapping("application/excel", "xlm")
        addMapping("application/excel", "xls")
        addMapping("application/excel", "xlt")
        addMapping("application/excel", "xlv")
        addMapping("application/excel", "xlw")
        addMapping("audio/tsp-audio", "tsi")
        addMapping("application/x-cdlink", "vcd")
        addMapping("application/x-bytecode.elisp (compiled elisp)", "elc")
        addMapping("application/vnd.ms-pki.seccat", "cat")
        addMapping("image/naplps", "nap")
        addMapping("image/naplps", "naplps")
        addMapping("text/vnd.fmi.flexstor", "flx")
        addMapping("image/xbm", "xbm")
        addMapping("audio/mid", "rmi")
        addMapping("application/vda", "vda")
        addMapping("video/fli", "fli")
        addMapping("audio/s3m", "s3m")
        addMapping("audio/x-gsm", "gsd")
        addMapping("audio/x-gsm", "gsm")
        addMapping("application/mswrite", "wri")
        addMapping("text/vnd.wap.wml", "wml")
        addMapping("application/vnd.wap.wmlscriptc", "wmlsc")
        addMapping("application/x-x509-ca-cert", "cer")
        addMapping("application/x-x509-ca-cert", "crt")
        addMapping("application/x-x509-ca-cert", "der")
        addMapping("text/x-script.tcsh", "tcsh")
        addMapping("model/iges", "iges")
        addMapping("model/iges", "igs")
        addMapping("text/richtext", "rt")
        addMapping("text/richtext", "rtf")
        addMapping("text/richtext", "rtx")
        addMapping("video/vosaic", "vos")
        addMapping("application/i-deas", "unv")
        addMapping("image/jpeg", "jfif")
        addMapping("image/jpeg", "jfif-tbnl")
        addMapping("image/jpeg", "jpe")
        addMapping("image/jpeg", "jpeg")
        addMapping("image/jpeg", "jpg")
        addMapping("drawing/x-dwf (old)", "dwf")
        addMapping("application/x-tbook", "sbk")
        addMapping("application/x-tbook", "tbk")
        addMapping("x-world/x-vrt", "vrt")
        addMapping("application/x-shar", "sh")
        addMapping("application/x-shar", "shar")
        addMapping("music/crescendo", "mid")
        addMapping("music/crescendo", "midi")
        addMapping("application/vnd.nokia.ringing-tone", "rng")
        addMapping("application/x-zip-compressed", "zip")
        addMapping("application/x-meme", "mm")
        addMapping("application/macbinary", "bin")
        addMapping("application/vnd.ms-pki.stl", "stl")
        addMapping("application/x-gzip", "gz")
        addMapping("application/x-gzip", "gzip")
        addMapping("application/x-lzh", "lzh")
        addMapping("application/x-elc", "elc")
        addMapping("application/futuresplash", "spl")
        addMapping("application/x-httpd-imap", "imap")
        addMapping("video/x-dv", "dif")
        addMapping("video/x-dv", "dv")
        addMapping("application/x-lzx", "lzx")
        addMapping("text/x-script.scheme", "scm")
        addMapping("application/wordperfect", "wp")
        addMapping("application/wordperfect", "wp5")
        addMapping("application/wordperfect", "wp6")
        addMapping("application/wordperfect", "wpd")
        addMapping("application/x-vnd.ls-xpix", "xpix")
        addMapping("application/x-bytecode.python", "pyc")
        addMapping("application/x-vrml", "vrml")
        addMapping("application/x-nokia-9000-communicator-add-on-software", "aos")
        addMapping("video/x-dl", "dl")
        addMapping("", "mpeg")
        addMapping("", "parityfec")
        addMapping("", "raw")
        addMapping("", "alternative")
        addMapping("", "digest")
        addMapping("", "mixed")
        addMapping("", "parallel")
        addMapping("", "gif")
        addMapping("", "ief")
        addMapping("", "jpeg")
        addMapping("", "ktx")
        addMapping("", "svg+xml")
        addMapping("", "mesh")
        addMapping("", "vrml")
        addMapping("", "G7291")
        addMapping("", "QCELP")
        addMapping("", "enriched")
        addMapping("", "plain")
        addMapping("", "richtext")
        addMapping("", "external-body")
        addMapping("", "partial")
        addMapping("", "rfc822")
        addMapping("", "emma+xml")
        addMapping("", "exi")
        addMapping("", "mathml-content+xml")
        addMapping("", "mathml-presentation+xml")
        addMapping("", "mathml+xml")
        addMapping("", "pgp-keys")
        addMapping("", "sparql-query")
        addMapping("", "sparql-results+xml")
        addMapping("", "xslt+xml")
        addMapping("application/vnd.motorola.flexsuite.fis", "vnd.motorola.flexsuite.fis")
        addMapping("application/vnd.osgi.bundle", "vnd.osgi.bundle")
        addMapping("application/vnd.ms-htmlhelp", "vnd.ms-htmlhelp")
        addMapping("application/vnd.kidspiration", "vnd.kidspiration")
        addMapping("application/vnd.dvb_service", "vnd.dvb.service")
        addMapping("application/vnd.iptc.g2.knowledgeitem+xml", "vnd.iptc.g2.knowledgeitem+xml")
        addMapping("application/smpte336m", "smpte336m")
        addMapping("application/kpml-request+xml", "kpml-request+xml")
        addMapping("application/ibe-key-request+xml", "ibe-key-request+xml")
        addMapping("application/vnd.crick.clicker.wordbank", "vnd.crick.clicker.wordbank")
        addMapping("text/xml", "xml")
        addMapping("application/vnd.microsoft.portable-executable", "vnd.microsoft.portable-executable")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheRecords+xml", "vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheRecords+xml")
        addMapping("multipart/vnd.bint.med-plus", "vnd.bint.med-plus")
        addMapping("application/prs.rdf-xml-crypt", "prs.rdf-xml-crypt")
        addMapping("application/vnd.ms-windows.printerpairing", "vnd.ms-windows.printerpairing")
        addMapping("application/vnd.iptc.g2.packageitem+xml", "vnd.iptc.g2.packageitem+xml")
        addMapping("application/vnd.genomatix.tuxedo", "vnd.genomatix.tuxedo")
        addMapping("application/cose", "cose")
        addMapping("application/macwriteii", "macwriteii")
        addMapping("application/scvp-cv-request", "scvp-cv-request")
        addMapping("application/vnd.accpac.simply.imp", "vnd.accpac.simply.imp")
        addMapping("video/vnd.sealed-swf", "vnd.sealed.swf")
        addMapping("application/vnd.osgi.dp", "vnd.osgi.dp")
        addMapping("application/provenance+xml", "provenance+xml")
        addMapping("application/vnd.citationstyles.style+xml", "vnd.citationstyles.style+xml")
        addMapping("application/vnd.dvb.ipdcdftnotifaccess", "vnd.dvb.ipdcdftnotifaccess")
        addMapping("application/vnd.ms-wmdrm.lic-resp", "vnd.ms-wmdrm.lic-resp")
        addMapping("application/vnd.hbci", "vnd.hbci")
        addMapping("application/vnd.ims.lis.v2.result+json", "vnd.ims.lis.v2.result+json")
        addMapping("application/vnd.tcpdump.pcap", "vnd.tcpdump.pcap")
        addMapping("application/vnd.tableschema+json", "vnd.tableschema+json")
        addMapping("application/vnd.multiad.creator.cif", "vnd.multiad.creator.cif")
        addMapping("application/index.vnd", "index.vnd")
        addMapping("application/vnd.picsel", "vnd.picsel")
        addMapping("application/tamp-sequence-adjust-confirm", "tamp-sequence-adjust-confirm")
        addMapping("application/tamp-update-confirm", "tamp-update-confirm")
        addMapping("application/vnd.sealed-eml", "vnd.sealed.eml")
        addMapping("application/n-quads", "n-quads")
        addMapping("application/xcon-conference-info-diff+xml", "xcon-conference-info-diff+xml")
        addMapping("application/vcard+xml", "vcard+xml")
        addMapping("application/vnd.anki", "vnd.anki")
        addMapping("audio/rtploopback", "rtploopback")
        addMapping("application/vnd.ntt-local.ogw_remote-access", "vnd.ntt-local.ogw_remote-access")
        addMapping("application/vnd.3gpp.bsf+xml", "vnd.3gpp.bsf+xml")
        addMapping("font/sfnt", "sfnt")
        addMapping("application/A2L", "A2L")
        addMapping("application/fhir+xml", "fhir+xml")
        addMapping("application/vnd.ims.lti.v2.toolproxy.id+json", "vnd.ims.lti.v2.toolproxy.id+json")
        addMapping("application/vnd.marlin.drm.actiontoken+xml", "vnd.marlin.drm.actiontoken+xml")
        addMapping("application/index-obj", "index.obj")
        addMapping("application/vnd.ibm.afplinedata", "vnd.ibm.afplinedata")
        addMapping("application/vnd.syncml+xml", "vnd.syncml+xml")
        addMapping("video/vnd.iptvforum.1dparityfec-2005", "vnd.iptvforum.1dparityfec-2005")
        addMapping("image/vnd.fpx", "vnd.fpx")
        addMapping("application/rtx", "rtx")
        addMapping("application/yang-patch+json", "yang-patch+json")
        addMapping("application/mbms-msk+xml", "mbms-msk+xml")
        addMapping("image/t38", "t38")
        addMapping("application/rtf", "rtf")
        addMapping("application/vnd.imagemeter.folder+zip", "vnd.imagemeter.folder+zip")
        addMapping("application/EmergencyCallData.ServiceInfo+xml", "EmergencyCallData.ServiceInfo+xml")
        addMapping("application/vnd.ahead.space", "vnd.ahead.space")
        addMapping("application/vnd.oma-scws-http-request", "vnd.oma-scws-http-request")
        addMapping("application/vnd.unity", "vnd.unity")
        addMapping("application/vnd.globalplatform.card-content-mgt", "vnd.globalplatform.card-content-mgt")
        addMapping("application/vnd.panoply", "vnd.panoply")
        addMapping("audio/AMR", "AMR")
        addMapping("text/uri-list", "uri-list")
        addMapping("message/sipfrag", "sipfrag")
        addMapping("application/vnd.adobe.formscentral.fcdt", "vnd.adobe.formscentral.fcdt")
        addMapping("application/vnd.motorola.flexsuite.wem", "vnd.motorola.flexsuite.wem")
        addMapping("application/ccxml+xml", "ccxml+xml")
        addMapping("application/prs.alvestrand.titrax-sheet", "prs.alvestrand.titrax-sheet")
        addMapping("application/vnd.wolfram.mathematica.package", "vnd.wolfram.mathematica.package")
        addMapping("audio/vnd.nokia.mobile-xmf", "vnd.nokia.mobile-xmf")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.chartsheet+xml", "vnd.openxmlformats-officedocument.spreadsheetml.chartsheet+xml")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.footnotes+xml", "vnd.openxmlformats-officedocument.wordprocessingml.footnotes+xml")
        addMapping("video/vnd-mpegurl", "vnd.mpegurl")
        addMapping("application/ATFX", "ATFX")
        addMapping("application/lgr+xml", "lgr+xml")
        addMapping("application/vnd.yamaha.smaf-audio", "vnd.yamaha.smaf-audio")
        addMapping("audio/ogg", "ogg")
        addMapping("audio/32kadpcm", "32kadpcm")
        addMapping("application/vnd.smart.notebook", "vnd.smart.notebook")
        addMapping("application/vnd.medcalcdata", "vnd.medcalcdata")
        addMapping("application/xcap-el+xml", "xcap-el+xml")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.styles+xml", "vnd.openxmlformats-officedocument.wordprocessingml.styles+xml")
        addMapping("application/vnd.infotech.project+xml", "vnd.infotech.project+xml")
        addMapping("application/SGML", "sgml")
        addMapping("application/vnd.pg.format", "vnd.pg.format")
        addMapping("application/vnd.wv.ssp+xml", "vnd.wv.ssp+xml")
        addMapping("audio/vnd.hns.audio", "vnd.hns.audio")
        addMapping("application/vnd.radisys.msml-dialog-speech+xml", "vnd.radisys.msml-dialog-speech+xml")
        addMapping("application/vnd.noblenet-sealer", "vnd.noblenet-sealer")
        addMapping("application/vnd.ms-wmdrm.meter-chlg-req", "vnd.ms-wmdrm.meter-chlg-req")
        addMapping("application/vnd.trid.tpt", "vnd.trid.tpt")
        addMapping("message/tracking-status", "tracking-status")
        addMapping("application/vnd.blueice.multipass", "vnd.blueice.multipass")
        addMapping("application/vnd.epson.quickanime", "vnd.epson.quickanime")
        addMapping("application/vnd.ms-excel.sheet.binary.macroEnabled.12", "vnd.ms-excel.sheet.binary.macroEnabled.12")
        addMapping("application/vnd.bbf.usp.msg", "vnd.bbf.usp.msg")
        addMapping("application/ocsp-request", "ocsp-request")
        addMapping("model/x3d+fastinfoset", "x3d+fastinfoset")
        addMapping("application/epub+zip", "epub+zip")
        addMapping("application/vnd.syncml.dmtnds+wbxml", "vnd.syncml.dmtnds+wbxml")
        addMapping("application/vnd.ocf+cbor", "vnd.ocf+cbor")
        addMapping("model/3mf", "3mf")
        addMapping("application/vnd.nokia.landmark+wbxml", "vnd.nokia.landmark+wbxml")
        addMapping("application/pkix-attr-cert", "pkix-attr-cert")
        addMapping("text/directory", "directory - DEPRECATED by RFC6350")
        addMapping("application/reginfo+xml", "reginfo+xml")
        addMapping("application/vnd.iptc.g2.newsitem+xml", "vnd.iptc.g2.newsitem+xml")
        addMapping("application/vnd.mediastation.cdkey", "vnd.mediastation.cdkey")
        addMapping("application/mosskey-request", "mosskey-request")
        addMapping("application/vnd.3gpp2.tcap", "vnd.3gpp2.tcap")
        addMapping("application/vnd.intertrust.nncp", "vnd.intertrust.nncp")
        addMapping("application/vnd.cluetrust.cartomobile-config", "vnd.cluetrust.cartomobile-config")
        addMapping("application/vnd.uplanet.cacheop", "vnd.uplanet.cacheop")
        addMapping("application/vnd.spotfire.dxp", "vnd.spotfire.dxp")
        addMapping("application/vnd.joost.joda-archive", "vnd.joost.joda-archive")
        addMapping("application/vnd.ecowin.fileupdate", "vnd.ecowin.fileupdate")
        addMapping("application/vnd.oma.lwm2m+tlv", "vnd.oma.lwm2m+tlv")
        addMapping("application/vnd.3gpp2.sms", "vnd.3gpp2.sms")
        addMapping("video/vnd.uvvu-mp4", "vnd.uvvu.mp4")
        addMapping("application/vnd.bekitzur-stech+json", "vnd.bekitzur-stech+json")
        addMapping("application/vnd.pvi.ptid1", "vnd.pvi.ptid1")
        addMapping("application/timestamped-data", "timestamped-data")
        addMapping("text/ulpfec", "ulpfec")
        addMapping("application/vnd.rs-274x", "vnd.rs-274x")
        addMapping("application/vnd.geonext", "vnd.geonext")
        addMapping("application/srgs", "srgs")
        addMapping("application/vnd.cups-raw", "vnd.cups-raw")
        addMapping("application/font-sfnt", "font-sfnt - DEPRECATED in favor of font/sfnt")
        addMapping("application/rlmi+xml", "rlmi+xml")
        addMapping("application/vnd.dvb.pfr", "vnd.dvb.pfr")
        addMapping("application/vnd.kahootz", "vnd.kahootz")
        addMapping("application/ccmp+xml", "ccmp+xml")
        addMapping("image/emf", "emf")
        addMapping("image/emf", "x-emf - DEPRECATED in favor of image/emf")
        addMapping("audio/RED", "RED")
        addMapping("application/vnd.oma.bcast.ltkm", "vnd.oma.bcast.ltkm")
        addMapping("application/vnd.ubisoft.webplayer", "vnd.ubisoft.webplayer")
        addMapping("application/vnd.google-earth.kml+xml", "vnd.google-earth.kml+xml")
        addMapping("audio/mpeg", "mpeg")
        addMapping("application/csvm+json", "csvm+json")
        addMapping("application/vnd.etsi.iptvueprofile+xml", "vnd.etsi.iptvueprofile+xml")
        addMapping("application/vnd.geoplan", "vnd.geoplan")
        addMapping("application/vnd.oma.bcast.associated-procedure-parameter+xml", "vnd.oma.bcast.associated-procedure-parameter+xml")
        addMapping("application/vnd.palm", "vnd.palm")
        addMapping("application/vnd.japannet-jpnstore-wakeup", "vnd.japannet-jpnstore-wakeup")
        addMapping("text/rtp-enc-aescm128", "rtp-enc-aescm128")
        addMapping("video/vnd.directv.mpeg-tts", "vnd.directv.mpeg-tts")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.styles+xml", "vnd.openxmlformats-officedocument.spreadsheetml.styles+xml")
        addMapping("image/vnd.sealedmedia.softseal-gif", "vnd.sealedmedia.softseal.gif")
        addMapping("audio/vnd.qcelp", "vnd.qcelp - DEPRECATED in favor of audio/qcelp")
        addMapping("application/vnd.hp-jlyt", "vnd.hp-jlyt")
        addMapping("application/vnd.apple.installer+xml", "vnd.apple.installer+xml")
        addMapping("message/global-delivery-status", "global-delivery-status")
        addMapping("application/vnd.radisys.msml-dialog+xml", "vnd.radisys.msml-dialog+xml")
        addMapping("application/vnd.jam", "vnd.jam")
        addMapping("text/SGML", "sgml")
        addMapping("application/mathematica", "mathematica")
        addMapping("audio/vnd.dolby.pl2", "vnd.dolby.pl2")
        addMapping("image/vnd.airzip.accelerator.azv", "vnd.airzip.accelerator.azv")
        addMapping("application/eshop", "eshop")
        addMapping("application/MF4", "MF4")
        addMapping("model/x3d-vrml", "x3d-vrml")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml", "vnd.openxmlformats-officedocument.wordprocessingml.document.main+xml")
        addMapping("application/CALS-1840", "CALS-1840")
        addMapping("application/yang-data+json", "yang-data+json")
        addMapping("application/yin+xml", "yin+xml")
        addMapping("application/vnd.balsamiq.bmml+xml", "vnd.balsamiq.bmml+xml")
        addMapping("application/cellml+xml", "cellml+xml")
        addMapping("video/vnd.iptvforum.ttsavc", "vnd.iptvforum.ttsavc")
        addMapping("application/alto-endpointcostparams+json", "alto-endpointcostparams+json")
        addMapping("application/vnd.iptc.g2.conceptitem+xml", "vnd.iptc.g2.conceptitem+xml")
        addMapping("audio/VMR-WB", "VMR-WB")
        addMapping("application/vnd.osgi.subsystem", "vnd.osgi.subsystem")
        addMapping("application/vnd.1000minds.decision-model+xml", "vnd.1000minds.decision-model+xml")
        addMapping("video/pointer", "pointer")
        addMapping("application/vnd.geospace", "vnd.geospace")
        addMapping("application/vnd.sealed-mht", "vnd.sealed.mht")
        addMapping("application/jwt", "jwt")
        addMapping("application/vnd.uiq.theme", "vnd.uiq.theme")
        addMapping("audio/prs.sid", "prs.sid")
        addMapping("application/oebps-package+xml", "oebps-package+xml")
        addMapping("application/vnd.datapackage+json", "vnd.datapackage+json")
        addMapping("application/mediaservercontrol+xml", "mediaservercontrol+xml")
        addMapping("text/prs.prop.logic", "prs.prop.logic")
        addMapping("application/vnd.omaloc-supl-init", "vnd.omaloc-supl-init")
        addMapping("text/csv", "csv")
        addMapping("application/vnd.lotus-notes", "vnd.lotus-notes")
        addMapping("text/css", "css")
        addMapping("application/vnd.is-xpr", "vnd.is-xpr")
        addMapping("application/vnd.ecdis-update", "vnd.ecdis-update")
        addMapping("audio/t140c", "t140c")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.tableStyles+xml", "vnd.openxmlformats-officedocument.presentationml.tableStyles+xml")
        addMapping("model/vnd.mts", "vnd.mts")
        addMapping("application/sru+xml", "sru+xml")
        addMapping("application/vnd.kde.kivio", "vnd.kde.kivio")
        addMapping("application/vnd.scribus", "vnd.scribus")
        addMapping("application/vnd.las.las+json", "vnd.las.las+json")
        addMapping("image/vnd-svf", "vnd.svf")
        addMapping("video/raptorfec", "raptorfec")
        addMapping("audio/EVRCWB", "EVRCWB")
        addMapping("audio/vnd.cmles.radio-events", "vnd.cmles.radio-events")
        addMapping("application/vnd.tmobile-livetv", "vnd.tmobile-livetv")
        addMapping("application/vnd.oma.bcast.simple-symbol-container", "vnd.oma.bcast.simple-symbol-container")
        addMapping("application/vnd.avistar+xml", "vnd.avistar+xml")
        addMapping("application/watcherinfo+xml", "watcherinfo+xml")
        addMapping("audio/MPA", "MPA")
        addMapping("application/yang-data+xml", "yang-data+xml")
        addMapping("application/vnd.dvb.iptv.alfec-enhancement", "vnd.dvb.iptv.alfec-enhancement")
        addMapping("application/vnd.frogans.ltf", "vnd.frogans.ltf")
        addMapping("application/vnd.ffsns", "vnd.ffsns")
        addMapping("model/x3d+xml", "x3d+xml")
        addMapping("application/vnd.hc+json", "vnd.hc+json")
        addMapping("application/ttml+xml", "ttml+xml")
        addMapping("text/vnd.motorola.reflex", "vnd.motorola.reflex")
        addMapping("text/tab-separated-values", "tab-separated-values")
        addMapping("audio/vnd.ms-playready.media.pya", "vnd.ms-playready.media.pya")
        addMapping("application/jwk+json", "jwk+json")
        addMapping("application/vnd.bmi", "vnd.bmi")
        addMapping("image/vnd.fastbidsheet", "vnd.fastbidsheet")
        addMapping("application/vnd.etsi.aoc+xml", "vnd.etsi.aoc+xml")
        addMapping("application/vnd.epson.salt", "vnd.epson.salt")
        addMapping("application/vnd.amundsen.maze+xml", "vnd.amundsen.maze+xml")
        addMapping("application/sdp", "sdp")
        addMapping("audio/encaprtp", "encaprtp")
        addMapping("application/vnd.triscape.mxs", "vnd.triscape.mxs")
        addMapping("application/vnd.gov.sk.e-form+xml", "vnd.gov.sk.e-form+xml")
        addMapping("message/global", "global")
        addMapping("application/link-format", "link-format")
        addMapping("application/vnd.kde.kformula", "vnd.kde.kformula")
        addMapping("audio/vnd.dolby.pulse.1", "vnd.dolby.pulse.1")
        addMapping("audio/vnd.4SB", "vnd.4SB")
        addMapping("audio/iLBC", "iLBC")
        addMapping("application/vnd.multiad.creator", "vnd.multiad.creator")
        addMapping("application/vnd.oasis.opendocument.text", "vnd.oasis.opendocument.text")
        addMapping("video/CelB", "CelB")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.webSettings+xml", "vnd.openxmlformats-officedocument.wordprocessingml.webSettings+xml")
        addMapping("application/ocsp-response", "ocsp-response")
        addMapping("text/vnd-curl", "vnd.curl")
        addMapping("text/vnd.radisys.msml-basic-layout", "vnd.radisys.msml-basic-layout")
        addMapping("font/woff", "woff")
        addMapping("application/xacml+xml", "xacml+xml")
        addMapping("image/vnd.tencent.tap", "vnd.tencent.tap")
        addMapping("application/IOTP", "iotp")
        addMapping("application/vnd.sigrok.session", "vnd.sigrok.session")
        addMapping("application/vnd.radisys.msml-audit-dialog+xml", "vnd.radisys.msml-audit-dialog+xml")
        addMapping("message/vnd.si.simp", "vnd.si.simp - OBSOLETED by request")
        addMapping("application/im-iscomposing+xml", "im-iscomposing+xml")
        addMapping("application/vnd.syncml.ds.notification", "vnd.syncml.ds.notification")
        addMapping("application/vnd.nokia.pcd+xml", "vnd.nokia.pcd+xml")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.tags+xml", "vnd.openxmlformats-officedocument.presentationml.tags+xml")
        addMapping("video/H261", "H261")
        addMapping("video/H263", "H263")
        addMapping("video/H264", "H264")
        addMapping("video/H265", "H265")
        addMapping("application/ulpfec", "ulpfec")
        addMapping("application/sep-exi", "sep-exi")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slide+xml", "vnd.openxmlformats-officedocument.presentationml.slide+xml")
        addMapping("application/passport", "passport")
        addMapping("application/vnd.fsc.weblaunch", "vnd.fsc.weblaunch")
        addMapping("audio/3gpp2", "3gpp2")
        addMapping("application/vnd.radisys.msml-conf+xml", "vnd.radisys.msml-conf+xml")
        addMapping("application/vnd.sealed.csf", "vnd.sealed.csf")
        addMapping("application/rpki-updown", "rpki-updown")
        addMapping("application/vnd.dece-zip", "vnd.dece.zip")
        addMapping("application/vnd.powerbuilder75-s", "vnd.powerbuilder75-s")
        addMapping("application/vnd.japannet-setstore-wakeup", "vnd.japannet-setstore-wakeup")
        addMapping("image/vnd.radiance", "vnd.radiance")
        addMapping("application/vnd.accpac.simply.aso", "vnd.accpac.simply.aso")
        addMapping("application/vnd.xmpie.xlim", "vnd.xmpie.xlim")
        addMapping("audio/t38", "t38")
        addMapping("application/urc-grpsheet+xml", "urc-grpsheet+xml")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.presentation.main+xml", "vnd.openxmlformats-officedocument.presentationml.presentation.main+xml")
        addMapping("application/alto-networkmapfilter+json", "alto-networkmapfilter+json")
        addMapping("application/vnd.cab-jscript", "vnd.cab-jscript")
        addMapping("application/jf2feed+json", "jf2feed+json")
        addMapping("application/vnd.ms-word.template.macroEnabled.12", "vnd.ms-word.template.macroEnabled.12")
        addMapping("application/vnd.obn", "vnd.obn")
        addMapping("application/EmergencyCallData.SubscriberInfo+xml", "EmergencyCallData.SubscriberInfo+xml")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.notesMaster+xml", "vnd.openxmlformats-officedocument.presentationml.notesMaster+xml")
        addMapping("application/vnd.nokia.n-gage.ac+xml", "vnd.nokia.n-gage.ac+xml")
        addMapping("application/vnd.ms-excel", "vnd.ms-excel")
        addMapping("application/vnd.noblenet-web", "vnd.noblenet-web")
        addMapping("application/vnd.sealed-doc", "vnd.sealed.doc")
        addMapping("application/index.response", "index.response")
        addMapping("application/tamp-error", "tamp-error")
        addMapping("application/pkcs8", "pkcs8")
        addMapping("application/vnd.eudora.data", "vnd.eudora.data")
        addMapping("model/vnd.opengex", "vnd.opengex")
        addMapping("application/vnd.ims.imsccv1p2", "vnd.ims.imsccv1p2")
        addMapping("application/vnd.orange.indata", "vnd.orange.indata")
        addMapping("application/vnd.ims.imsccv1p3", "vnd.ims.imsccv1p3")
        addMapping("application/vnd.quarantainenet", "vnd.quarantainenet")
        addMapping("application/vnd.document+json", "vnd.document+json")
        addMapping("application/vnd.ncd.reference", "vnd.ncd.reference")
        addMapping("application/route-apd+xml", "route-apd+xml")
        addMapping("application/vnd.ims.imsccv1p1", "vnd.ims.imsccv1p1")
        addMapping("application/vnd.ms-wmdrm.meter-resp", "vnd.ms-wmdrm.meter-resp")
        addMapping("application/vnd.oasis.opendocument.database", "vnd.oasis.opendocument.database")
        addMapping("application/vnd.wqd", "vnd.wqd")
        addMapping("model/gltf+json", "gltf+json")
        addMapping("application/EmergencyCallData.Control+xml", "EmergencyCallData.Control+xml")
        addMapping("application/vnd.cinderella", "vnd.cinderella")
        addMapping("application/vnd.kde.karbon", "vnd.kde.karbon")
        addMapping("application/timestamp-reply", "timestamp-reply")
        addMapping("application/vnd.microsoft.windows.thumbnail-cache", "vnd.microsoft.windows.thumbnail-cache")
        addMapping("application/vnd.iptc.g2.catalogitem+xml", "vnd.iptc.g2.catalogitem+xml")
        addMapping("application/vnd.openxmlformats-package.relationships+xml", "vnd.openxmlformats-package.relationships+xml")
        addMapping("audio/EVRCNW", "EVRCNW")
        addMapping("application/jrd+json", "jrd+json")
        addMapping("application/vnd.yamaha.openscoreformat", "vnd.yamaha.openscoreformat")
        addMapping("application/xml-external-parsed-entity", "xml-external-parsed-entity")
        addMapping("application/vnd.omads-email+xml", "vnd.omads-email+xml")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.pivotTable+xml", "vnd.openxmlformats-officedocument.spreadsheetml.pivotTable+xml")
        addMapping("audio/AMR-WB", "AMR-WB")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.connections+xml", "vnd.openxmlformats-officedocument.spreadsheetml.connections+xml")
        addMapping("application/vnd.enphase.envoy", "vnd.enphase.envoy")
        addMapping("video/vnd.iptvforum.2dparityfec-2005", "vnd.iptvforum.2dparityfec-2005")
        addMapping("application/wita", "wita")
        addMapping("application/tei+xml", "tei+xml")
        addMapping("application/pidf+xml", "pidf+xml")
        addMapping("video/vnd.dece.sd", "vnd.dece.sd")
        addMapping("application/vnd.httphone", "vnd.httphone")
        addMapping("application/vnd.nokia.n-gage.data", "vnd.nokia.n-gage.data")
        addMapping("application/relax-ng-compact-syntax", "relax-ng-compact-syntax")
        addMapping("model/gltf-binary", "gltf-binary")
        addMapping("application/vnd.ibm.modcap", "vnd.ibm.modcap")
        addMapping("application/conference-info+xml", "conference-info+xml")
        addMapping("application/vnd.kde.kchart", "vnd.kde.kchart")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.document", "vnd.openxmlformats-officedocument.wordprocessingml.document")
        addMapping("video/vnd.dece.pd", "vnd.dece.pd")
        addMapping("image/naplps", "naplps")
        addMapping("application/vnd.crick.clicker.keyboard", "vnd.crick.clicker.keyboard")
        addMapping("text/vnd.fmi.flexstor", "vnd.fmi.flexstor")
        addMapping("audio/GSM-HR-08", "GSM-HR-08")
        addMapping("application/alto-endpointcost+json", "alto-endpointcost+json")
        addMapping("application/vnd.nokia.conml+xml", "vnd.nokia.conml+xml")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.numbering+xml", "vnd.openxmlformats-officedocument.wordprocessingml.numbering+xml")
        addMapping("application/samlmetadata+xml", "samlmetadata+xml")
        addMapping("application/nss", "nss")
        addMapping("audio/rtx", "rtx")
        addMapping("audio/asc", "asc")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.handoutMaster+xml", "vnd.openxmlformats-officedocument.presentationml.handoutMaster+xml")
        addMapping("text/vnd.in3d.3dml", "vnd.in3d.3dml")
        addMapping("application/vnd.epson.ssf", "vnd.epson.ssf")
        addMapping("application/vnd.cosmocaller", "vnd.cosmocaller")
        addMapping("audio/PCMA-WB", "PCMA-WB")
        addMapping("application/vnd.pocketlearn", "vnd.pocketlearn")
        addMapping("application/vnd.uplanet.channel-wbxml", "vnd.uplanet.channel-wbxml")
        addMapping("application/cose-key-set", "cose-key-set")
        addMapping("application/mbms-schedule+xml", "mbms-schedule+xml")
        addMapping("application/media-policy-dataset+xml", "media-policy-dataset+xml")
        addMapping("application/vnd.wmc", "vnd.wmc")
        addMapping("audio/3gpp", "3gpp")
        addMapping("application/vnd.recordare.musicxml", "vnd.recordare.musicxml")
        addMapping("application/vnd.japannet-verification-wakeup", "vnd.japannet-verification-wakeup")
        addMapping("application/vnd.japannet-registration-wakeup", "vnd.japannet-registration-wakeup")
        addMapping("application/vnd.igloader", "vnd.igloader")
        addMapping("application/vnd.radisys.msml-dialog-group+xml", "vnd.radisys.msml-dialog-group+xml")
        addMapping("application/tamp-update", "tamp-update")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.template.main+xml", "vnd.openxmlformats-officedocument.spreadsheetml.template.main+xml")
        addMapping("application/alto-endpointpropparams+json", "alto-endpointpropparams+json")
        addMapping("application/vnd.crick.clicker.palette", "vnd.crick.clicker.palette")
        addMapping("application/vnd.uplanet.bearer-choice-wbxml", "vnd.uplanet.bearer-choice-wbxml")
        addMapping("application/vnd.yamaha.hv-voice", "vnd.yamaha.hv-voice")
        addMapping("application/font-woff", "font-woff - DEPRECATED in favor of font/woff")
        addMapping("application/alto-costmapfilter+json", "alto-costmapfilter+json")
        addMapping("application/vnd.groove-tool-template", "vnd.groove-tool-template")
        addMapping("image/vnd-djvu", "vnd.djvu")
        addMapping("application/vnd.hcl-bireports", "vnd.hcl-bireports")
        addMapping("application/vnd.dvb.ipdcesgaccess2", "vnd.dvb.ipdcesgaccess2")
        addMapping("application/vnd.uri-map", "vnd.uri-map")
        addMapping("application/vnd.comicbook-rar", "vnd.comicbook-rar")
        addMapping("application/vnd.Quark.QuarkXPress", "vnd.Quark.QuarkXPress")
        addMapping("application/vnd.apache.thrift.json", "vnd.apache.thrift.json")
        addMapping("application/vnd.imagemeter.image+zip", "vnd.imagemeter.image+zip")
        addMapping("video/vnd.directv-mpeg", "vnd.directv.mpeg")
        addMapping("application/raptorfec", "raptorfec")
        addMapping("audio/vnd.audiokoz", "vnd.audiokoz")
        addMapping("text/provenance-notation", "provenance-notation")
        addMapping("application/sgml-open-catalog", "sgml-open-catalog")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slideshow", "vnd.openxmlformats-officedocument.presentationml.slideshow")
        addMapping("application/ISUP", "isup")
        addMapping("application/vnd.fujixerox.ART4", "vnd.fujixerox.ART4")
        addMapping("video/vnd.CCTV", "vnd.CCTV")
        addMapping("application/vnd.3gpp-prose+xml", "vnd.3gpp-prose+xml")
        addMapping("application/vnd.geogebra.file", "vnd.geogebra.file")
        addMapping("application/vnd.oma.pal+xml", "vnd.oma.pal+xml")
        addMapping("application/msc-ivr+xml", "msc-ivr+xml")
        addMapping("application/example", "example")
        addMapping("application/voicexml+xml", "voicexml+xml")
        addMapping("application/vnd.restful+json", "vnd.restful+json")
        addMapping("application/vnd.ntt-local.file-transfer", "vnd.ntt-local.file-transfer")
        addMapping("application/set-registration", "set-registration")
        addMapping("application/vnd.tao.intent-module-archive", "vnd.tao.intent-module-archive")
        addMapping("application/vnd-curl", "vnd.curl")
        addMapping("application/spirits-event+xml", "spirits-event+xml")
        addMapping("text/jcr-cnd", "jcr-cnd")
        addMapping("application/mets+xml", "mets+xml")
        addMapping("application/mp4", "mp4")
        addMapping("application/vnd.powerbuilder7-s", "vnd.powerbuilder7-s")
        addMapping("video/ulpfec", "ulpfec")
        addMapping("text/vcard", "vcard")
        addMapping("application/vnd.kde.kspread", "vnd.kde.kspread")
        addMapping("application/prs.cww", "prs.cww")
        addMapping("application/vnd.balsamiq.bmpr", "vnd.balsamiq.bmpr")
        addMapping("application/vnd.ms-PrintSchemaTicket+xml", "vnd.ms-PrintSchemaTicket+xml")
        addMapping("application/geo+json", "geo+json")
        addMapping("audio/speex", "speex")
        addMapping("application/vnd.lotus-approach", "vnd.lotus-approach")
        addMapping("application/vnd.oma.poc.optimized-progress-report+xml", "vnd.oma.poc.optimized-progress-report+xml")
        addMapping("application/vnd.f-secure.mobile", "vnd.f-secure.mobile")
        addMapping("application/rtploopback", "rtploopback")
        addMapping("application/whoispp-response", "whoispp-response")
        addMapping("application/vnd.s3sms", "vnd.s3sms")
        addMapping("application/vnd.ncd.control", "vnd.ncd.control")
        addMapping("application/vnd.nokia.radio-presets", "vnd.nokia.radio-presets")
        addMapping("application/vnd.iptc.g2.planningitem+xml", "vnd.iptc.g2.planningitem+xml")
        addMapping("application/vnd.font-fontforge-sfd", "vnd.font-fontforge-sfd")
        addMapping("application/vnd.oma.bcast.sgdd+xml", "vnd.oma.bcast.sgdd+xml")
        addMapping("application/vnd.etsi.asic-s+zip", "vnd.etsi.asic-s+zip")
        addMapping("application/vnd.omads-folder+xml", "vnd.omads-folder+xml")
        addMapping("application/vnd.uplanet.list-wbxml", "vnd.uplanet.list-wbxml")
        addMapping("application/vnd.fujixerox.docuworks", "vnd.fujixerox.docuworks")
        addMapping("application/vnd.mophun.certificate", "vnd.mophun.certificate")
        addMapping("application/vnd.yamaha.hv-script", "vnd.yamaha.hv-script")
        addMapping("application/held+xml", "held+xml")
        addMapping("application/vnd.japannet-registration", "vnd.japannet-registration")
        addMapping("application/vnd-wap-wmlc", "vnd.wap.wmlc")
        addMapping("application/vnd.denovo.fcselayout-link", "vnd.denovo.fcselayout-link")
        addMapping("application/zip", "zip")
        addMapping("application/rpki-roa", "rpki-roa")
        addMapping("application/vnd.ims.lti.v2.toolsettings.simple+json", "vnd.ims.lti.v2.toolsettings.simple+json")
        addMapping("application/vnd.groove-vcard", "vnd.groove-vcard")
        addMapping("application/dvcs", "dvcs")
        addMapping("image/vnd.adobe.photoshop", "vnd.adobe.photoshop")
        addMapping("application/vnd.audiograph", "vnd.audiograph")
        addMapping("application/vnd.ms-PrintDeviceCapabilities+xml", "vnd.ms-PrintDeviceCapabilities+xml")
        addMapping("application/vnd.easykaraoke.cdgdownload", "vnd.easykaraoke.cdgdownload")
        addMapping("video/vnd.dece.hd", "vnd.dece.hd")
        addMapping("application/news-checkgroups", "news-checkgroups")
        addMapping("model/vnd.vtu", "vnd.vtu")
        addMapping("image/vnd.zbrush.pcx", "vnd.zbrush.pcx")
        addMapping("application/vnd.truedoc", "vnd.truedoc")
        addMapping("text/RED", "RED")
        addMapping("application/vnd.google-earth.kmz", "vnd.google-earth.kmz")
        addMapping("application/tamp-apex-update-confirm", "tamp-apex-update-confirm")
        addMapping("application/vnd.3gpp.mcptt-affiliation-command+xml", "vnd.3gpp.mcptt-affiliation-command+xml")
        addMapping("application/vnd.vividence.scriptfile", "vnd.vividence.scriptfile")
        addMapping("application/vnd.powerbuilder75", "vnd.powerbuilder75")
        addMapping("application/vnd.syncml.dm+xml", "vnd.syncml.dm+xml")
        addMapping("application/vnd.wv.csp+xml", "vnd.wv.csp+xml")
        addMapping("application/gml+xml", "gml+xml")
        addMapping("application/set-payment-initiation", "set-payment-initiation")
        addMapping("application/vnd.nokia.ncd", "vnd.nokia.ncd")
        addMapping("audio/SMV", "SMV")
        addMapping("application/vnd.ibm.MiniPay", "vnd.ibm.MiniPay")
        addMapping("application/xmpp+xml", "xmpp+xml")
        addMapping("application/vnd.3gpp.access-transfer-events+xml", "vnd.3gpp.access-transfer-events+xml")
        addMapping("application/vnd.etsi.iptvprofile+xml", "vnd.etsi.iptvprofile+xml")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.calcChain+xml", "vnd.openxmlformats-officedocument.spreadsheetml.calcChain+xml")
        addMapping("application/vnd.proteus.magazine", "vnd.proteus.magazine")
        addMapping("application/vnd.ms-powerpoint.slide.macroEnabled.12", "vnd.ms-powerpoint.slide.macroEnabled.12")
        addMapping("audio/PCMA", "PCMA")
        addMapping("text/grammar-ref-list", "grammar-ref-list")
        addMapping("application/vnd.informedcontrol.rms+xml", "vnd.informedcontrol.rms+xml")
        addMapping("audio/PCMU", "PCMU")
        addMapping("video/vnd.iptvforum.1dparityfec-1010", "vnd.iptvforum.1dparityfec-1010")
        addMapping("application/vnd.groove-injector", "vnd.groove-injector")
        addMapping("application/vnd.ms-excel.addin.macroEnabled.12", "vnd.ms-excel.addin.macroEnabled.12")
        addMapping("text/vnd.trolltech.linguist", "vnd.trolltech.linguist")
        addMapping("video/vnd.radgamettools.smacker", "vnd.radgamettools.smacker")
        addMapping("application/dashdelta", "dashdelta")
        addMapping("audio/vnd.dolby.mlp", "vnd.dolby.mlp")
        addMapping("application/vnd.nintendo.nitro.rom", "vnd.nintendo.nitro.rom")
        addMapping("application/vnd.maxmind.maxmind-db", "vnd.maxmind.maxmind-db")
        addMapping("video/SMPTE292M", "SMPTE292M")
        addMapping("application/vnd.espass-espass+zip", "vnd.espass-espass+zip")
        addMapping("image/prs.pti", "prs.pti")
        addMapping("audio/vnd.octel.sbc", "vnd.octel.sbc")
        addMapping("application/mbms-envelope+xml", "mbms-envelope+xml")
        addMapping("application/vnd.sailingtracker.track", "vnd.sailingtracker.track")
        addMapping("application/mosskey-data", "mosskey-data")
        addMapping("application/vnd.criticaltools.wbs+xml", "vnd.criticaltools.wbs+xml")
        addMapping("application/vnd.evolv.ecig.theme", "vnd.evolv.ecig.theme")
        addMapping("application/vnd.uplanet.signal", "vnd.uplanet.signal")
        addMapping("application/vnd.oma.poc.detailed-progress-report+xml", "vnd.oma.poc.detailed-progress-report+xml")
        addMapping("application/vnd.nokia.conml+wbxml", "vnd.nokia.conml+wbxml")
        addMapping("application/vnd.dvb.notif-generic+xml", "vnd.dvb.notif-generic+xml")
        addMapping("application/vnd.ms-excel.sheet.macroEnabled.12", "vnd.ms-excel.sheet.macroEnabled.12")
        addMapping("audio/vnd.dlna.adts", "vnd.dlna.adts")
        addMapping("application/nlsml+xml", "nlsml+xml")
        addMapping("application/xv+xml", "xv+xml")
        addMapping("application/vnd.eprints.data+xml", "vnd.eprints.data+xml")
        addMapping("application/vnd.previewsystems.box", "vnd.previewsystems.box")
        addMapping("application/wspolicy+xml", "wspolicy+xml")
        addMapping("application/vnd.realvnc.bed", "vnd.realvnc.bed")
        addMapping("image/vnd.sealedmedia.softseal-jpg", "vnd.sealedmedia.softseal.jpg")
        addMapping("application/prs.nprend", "prs.nprend")
        addMapping("application/vq-rtcpxr", "vq-rtcpxr")
        addMapping("model/vnd.rosette.annotated-data-model", "vnd.rosette.annotated-data-model")
        addMapping("application/vnd.adobe.xfdf", "vnd.adobe.xfdf")
        addMapping("application/CDFX+XML", "CDFX+XML")
        addMapping("application/vnd.fujitsu.oasys2", "vnd.fujitsu.oasys2")
        addMapping("text/vnd.ms-mediapackage", "vnd.ms-mediapackage")
        addMapping("application/vnd.fujitsu.oasys3", "vnd.fujitsu.oasys3")
        addMapping("message/sip", "sip")
        addMapping("application/mbms-deregister+xml", "mbms-deregister+xml")
        addMapping("application/news-groupinfo", "news-groupinfo")
        addMapping("application/pls+xml", "pls+xml")
        addMapping("message/http", "http")
        addMapping("application/json-patch+json", "json-patch+json")
        addMapping("image/bmp", "bmp")
        addMapping("application/yang", "yang")
        addMapping("application/vnd.dvb.ipdcesgaccess", "vnd.dvb.ipdcesgaccess")
        addMapping("application/scvp-vp-request", "scvp-vp-request")
        addMapping("application/vnd.3M.Post-it-Notes", "vnd.3M.Post-it-Notes")
        addMapping("application/timestamp-query", "timestamp-query")
        addMapping("application/vnd.collection.doc+json", "vnd.collection.doc+json")
        addMapping("application/moss-keys", "moss-keys")
        addMapping("application/vnd.oipf.contentaccessstreaming+xml", "vnd.oipf.contentaccessstreaming+xml")
        addMapping("application/vnd.street-stream", "vnd.street-stream")
        addMapping("application/vnd.ms-word.document.macroEnabled.12", "vnd.ms-word.document.macroEnabled.12")
        addMapping("application/cdmi-queue", "cdmi-queue")
        addMapping("application/vnd.etsi.iptvsad-npvr+xml", "vnd.etsi.iptvsad-npvr+xml")
        addMapping("video/iso.segment", "iso.segment")
        addMapping("application/vnd.nokia.n-gage.symbian.install", "vnd.nokia.n-gage.symbian.install - OBSOLETE; no replacement given")
        addMapping("application/vnd.desmume-movie", "vnd.desmume.movie")
        addMapping("application/moss-signature", "moss-signature")
        addMapping("video/vnd.nokia.videovoip", "vnd.nokia.videovoip")
        addMapping("image/vnd.globalgraphics.pgb", "vnd.globalgraphics.pgb")
        addMapping("application/vnd.MFER", "vnd.MFER")
        addMapping("application/vnd.vectorworks", "vnd.vectorworks")
        addMapping("audio/vnd.dts.hd", "vnd.dts.hd")
        addMapping("application/dialog-info+xml", "dialog-info+xml")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slideshow.main+xml", "vnd.openxmlformats-officedocument.presentationml.slideshow.main+xml")
        addMapping("application/vnd.ezpix-album", "vnd.ezpix-album")
        addMapping("application/vnd.oipf.dae.svg+xml", "vnd.oipf.dae.svg+xml")
        addMapping("application/vnd.micro+json", "vnd.micro+json")
        addMapping("text/raptorfec", "raptorfec")
        addMapping("application/vnd.ms-tnef", "vnd.ms-tnef")
        addMapping("application/vnd.astraea-software.iota", "vnd.astraea-software.iota")
        addMapping("application/vnd.oasis.opendocument.text-web", "vnd.oasis.opendocument.text-web")
        addMapping("application/vnd.svd", "vnd.svd")
        addMapping("video/vnd.motorola.video", "vnd.motorola.video")
        addMapping("application/fdt+xml", "fdt+xml")
        addMapping("video/JPEG", "JPEG")
        addMapping("application/alto-networkmap+json", "alto-networkmap+json")
        addMapping("application/vnd.airzip.filesecure.azs", "vnd.airzip.filesecure.azs")
        addMapping("application/metalink4+xml", "metalink4+xml")
        addMapping("video/vnd.hns.video", "vnd.hns.video")
        addMapping("application/x400-bp", "x400-bp")
        addMapping("audio/vnd.dece.audio", "vnd.dece.audio")
        addMapping("application/vnd.biopax.rdf+xml", "vnd.biopax.rdf+xml")
        addMapping("application/vnd.pwg-xhtml-print+xml", "vnd.pwg-xhtml-print+xml")
        addMapping("model/vnd.gs-gdl", "vnd.gs-gdl")
        addMapping("audio/EVRCB1", "EVRCB1")
        addMapping("text/vnd.sun.j2me.app-descriptor", "vnd.sun.j2me.app-descriptor")
        addMapping("audio/EVRCB0", "EVRCB0")
        addMapping("application/vnd.airzip.filesecure.azf", "vnd.airzip.filesecure.azf")
        addMapping("application/vnd.oasis.opendocument.presentation", "vnd.oasis.opendocument.presentation")
        addMapping("application/vnd.debian.binary-package", "vnd.debian.binary-package")
        addMapping("audio/vorbis", "vorbis")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml-template", "vnd.openxmlformats-officedocument.wordprocessingml.template")
        addMapping("text/turtle", "turtle")
        addMapping("audio/eac3", "eac3")
        addMapping("application/ATXML", "ATXML")
        addMapping("application/vnd.hp-HPGL", "vnd.hp-HPGL")
        addMapping("application/vnd.hzn-3d-crossword", "vnd.hzn-3d-crossword")
        addMapping("application/vnd.fujitsu.oasysgp", "vnd.fujitsu.oasysgp")
        addMapping("audio/mpa-robust", "mpa-robust")
        addMapping("audio/VDVI", "VDVI")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.comments+xml", "vnd.openxmlformats-officedocument.spreadsheetml.comments+xml")
        addMapping("application/vnd.oasis.opendocument.image", "vnd.oasis.opendocument.image")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slideMaster+xml", "vnd.openxmlformats-officedocument.presentationml.slideMaster+xml")
        addMapping("application/vnd.micrografx.flo", "vnd.micrografx.flo")
        addMapping("video/vnd.dece-mp4", "vnd.dece.mp4")
        addMapping("application/vnd.ms-playready.initiator+xml", "vnd.ms-playready.initiator+xml")
        addMapping("audio/G7221", "G7221")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.settings+xml", "vnd.openxmlformats-officedocument.wordprocessingml.settings+xml")
        addMapping("application/vnd.oftn.l10n+json", "vnd.oftn.l10n+json")
        addMapping("application/vnd.grafeq", "vnd.grafeq")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.revisionHeaders+xml", "vnd.openxmlformats-officedocument.spreadsheetml.revisionHeaders+xml")
        addMapping("application/vnd.semd", "vnd.semd")
        addMapping("application/vnd.semf", "vnd.semf")
        addMapping("application/vnd.shana.informed.interchange", "vnd.shana.informed.interchange")
        addMapping("application/prs.plucker", "prs.plucker")
        addMapping("application/vnd.apache.thrift.binary", "vnd.apache.thrift.binary")
        addMapping("audio/vnd.nuera.ecelp4800", "vnd.nuera.ecelp4800")
        addMapping("application/pkcs8-encrypted", "pkcs8-encrypted")
        addMapping("application/node", "node")
        addMapping("application/vnd.visio", "vnd.visio")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.queryTable+xml", "vnd.openxmlformats-officedocument.spreadsheetml.queryTable+xml")
        addMapping("application/vnd-dxr", "vnd.dxr")
        addMapping("image/wmf", "wmf")
        addMapping("image/wmf", "x-wmf - DEPRECATED in favor of image/wmf")
        addMapping("application/vnd.jcp.javame.midlet-rms", "vnd.jcp.javame.midlet-rms")
        addMapping("application/cdmi-capability", "cdmi-capability")
        addMapping("application/vnd.etsi.timestamp-token", "vnd.etsi.timestamp-token")
        addMapping("application/vnd.ms-wmdrm.lic-chlg-req", "vnd.ms-wmdrm.lic-chlg-req")
        addMapping("application/3gpdash-qoe-report+xml", "3gpdash-qoe-report+xml")
        addMapping("application/vnd.fujixerox.HBPL", "vnd.fujixerox.HBPL")
        addMapping("application/vnd.iptc.g2.newsmessage+xml", "vnd.iptc.g2.newsmessage+xml")
        addMapping("application/vnd-mif", "vnd.mif")
        addMapping("application/pgp-encrypted", "pgp-encrypted")
        addMapping("application/vnd.wap-wbxml", "vnd.wap.wbxml")
        addMapping("audio/DVI4", "DVI4")
        addMapping("video/jpeg2000", "jpeg2000")
        addMapping("application/vnd.oma.drm.risd+xml", "vnd.oma.drm.risd+xml")
        addMapping("application/vnd.commerce-battelle", "vnd.commerce-battelle")
        addMapping("application/vnd.ruckus.download", "vnd.ruckus.download")
        addMapping("application/vnd.kde.kword", "vnd.kde.kword")
        addMapping("application/vnd.oma.poc.groups+xml", "vnd.oma.poc.groups+xml")
        addMapping("application/vnd.ms-wpl", "vnd.ms-wpl")
        addMapping("application/dssc+xml", "dssc+xml")
        addMapping("application/vnd.xmpie.ppkg", "vnd.xmpie.ppkg")
        addMapping("application/vnd.dece.data", "vnd.dece.data")
        addMapping("application/vnd.dolby.mobile.1", "vnd.dolby.mobile.1")
        addMapping("application/vnd.dolby.mobile.2", "vnd.dolby.mobile.2")
        addMapping("image/vnd.microsoft.icon", "vnd.microsoft.icon")
        addMapping("application/vnd.otps.ct-kip+xml", "vnd.otps.ct-kip+xml")
        addMapping("multipart/x-mixed-replace", "x-mixed-replace")
        addMapping("application/vnd.wordperfect", "vnd.wordperfect")
        addMapping("video/VP8", "VP8")
        addMapping("application/vnd.ms-powerpoint.presentation.macroEnabled.12", "vnd.ms-powerpoint.presentation.macroEnabled.12")
        addMapping("application/epp+xml", "epp+xml")
        addMapping("video/vnd-vivo", "vnd.vivo")
        addMapping("application/mbms-reception-report+xml", "mbms-reception-report+xml")
        addMapping("application/vnd.fujixerox.docuworks.container", "vnd.fujixerox.docuworks.container")
        addMapping("application/vnd.Mobius.TXF", "vnd.Mobius.TXF")
        addMapping("text/rtploopback", "rtploopback")
        addMapping("application/javascript", "javascript")
        addMapping("application/dskpp+xml", "dskpp+xml")
        addMapping("application/vnd.intu.qbo", "vnd.intu.qbo")
        addMapping("application/ssml+xml", "ssml+xml")
        addMapping("application/vnd.oipf.ueprofile+xml", "vnd.oipf.ueprofile+xml")
        addMapping("application/urc-ressheet+xml", "urc-ressheet+xml")
        addMapping("application/vnd.ms-powerpoint.addin.macroEnabled.12", "vnd.ms-powerpoint.addin.macroEnabled.12")
        addMapping("application/vnd.adobe.flash-movie", "vnd.adobe.flash.movie")
        addMapping("application/problem+xml", "problem+xml")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.template.main+xml", "vnd.openxmlformats-officedocument.wordprocessingml.template.main+xml")
        addMapping("application/vnd.iccprofile", "vnd.iccprofile")
        addMapping("application/nasdata", "nasdata")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet", "vnd.openxmlformats-officedocument.spreadsheetml.sheet")
        addMapping("text/csv-schema", "csv-schema")
        addMapping("application/cdmi-object", "cdmi-object")
        addMapping("application/vnd.oipf.contentaccessdownload+xml", "vnd.oipf.contentaccessdownload+xml")
        addMapping("application/vnd.valve.source.material", "vnd.valve.source.material")
        addMapping("application/vnd.cloanto.rp9", "vnd.cloanto.rp9")
        addMapping("application/vnd.tmd.mediaflex.api+xml", "vnd.tmd.mediaflex.api+xml")
        addMapping("application/vnd.nokia.catalogs", "vnd.nokia.catalogs")
        addMapping("application/EmergencyCallData.VEDS+xml", "EmergencyCallData.VEDS+xml")
        addMapping("application/andrew-inset", "andrew-inset")
        addMapping("application/vnd.autopackage", "vnd.autopackage")
        addMapping("video/ogg", "ogg")
        addMapping("text/t140", "t140")
        addMapping("application/vnd.dm.delegation+xml", "vnd.dm.delegation+xml")
        addMapping("application/vnd.oma.bcast.drm-trigger+xml", "vnd.oma.bcast.drm-trigger+xml")
        addMapping("application/atomsvc+xml", "atomsvc+xml")
        addMapping("image/jpm", "jpm")
        addMapping("application/msword", "msword")
        addMapping("application/vnd.etsi.mcid+xml", "vnd.etsi.mcid+xml")
        addMapping("application/DIT", "DIT")
        addMapping("application/emotionml+xml", "emotionml+xml")
        addMapping("image/jpx", "jpx")
        addMapping("application/vnd.openxmlformats-package.core-properties+xml", "vnd.openxmlformats-package.core-properties+xml")
        addMapping("application/vnd.dece.ttml+xml", "vnd.dece.ttml+xml")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.tableSingleCells+xml", "vnd.openxmlformats-officedocument.spreadsheetml.tableSingleCells+xml")
        addMapping("application/vnd.informix-visionary", "vnd.informix-visionary - OBSOLETED in favor of application/vnd.visionary")
        addMapping("audio/GSM-EFR", "GSM-EFR")
        addMapping("application/vnd.ezpix-package", "vnd.ezpix-package")
        addMapping("application/DII", "DII")
        addMapping("application/vnd.radisys.msml-audit-stream+xml", "vnd.radisys.msml-audit-stream+xml")
        addMapping("application/vnd.route66.link66+xml", "vnd.route66.link66+xml")
        addMapping("application/load-control+xml", "load-control+xml")
        addMapping("application/cbor", "cbor")
        addMapping("text/vnd.IPTC.NewsML", "vnd.IPTC.NewsML")
        addMapping("application/vnd.openblox.game+xml", "vnd.openblox.game+xml")
        addMapping("application/vnd.ims.lti.v2.toolconsumerprofile+json", "vnd.ims.lti.v2.toolconsumerprofile+json")
        addMapping("multipart/multilingual", "multilingual")
        addMapping("application/pkcs7-mime", "pkcs7-mime")
        addMapping("application/vnd.oracle.resource+json", "vnd.oracle.resource+json")
        addMapping("application/vnd.drive+json", "vnd.drive+json")
        addMapping("application/vnd.openxmlformats-package.digital-signature-xmlsignature+xml", "vnd.openxmlformats-package.digital-signature-xmlsignature+xml")
        addMapping("application/vnd.xmpie.plan", "vnd.xmpie.plan")
        addMapping("application/ppsp-tracker+json", "ppsp-tracker+json")
        addMapping("application/vnd.ecowin.seriesupdate", "vnd.ecowin.seriesupdate")
        addMapping("application/vnd.intu.qfx", "vnd.intu.qfx")
        addMapping("application/jose", "jose")
        addMapping("application/rpki-manifest", "rpki-manifest")
        addMapping("application/alto-error+json", "alto-error+json")
        addMapping("text/n3", "n3")
        addMapping("application/vnd.koan", "vnd.koan")
        addMapping("application/ipfix", "ipfix")
        addMapping("application/vnd.renlearn.rlprint", "vnd.RenLearn.rlprint")
        addMapping("application/vnd.ecowin.chart", "vnd.ecowin.chart")
        addMapping("audio/vnd.cns.inf1", "vnd.cns.inf1")
        addMapping("application/mikey", "mikey")
        addMapping("application/vnd.groove-identity-message", "vnd.groove-identity-message")
        addMapping("application/vnd.tml", "vnd.tml")
        addMapping("application/vnd.Mobius.PLC", "vnd.Mobius.PLC")
        addMapping("application/vnd.etsi.sci+xml", "vnd.etsi.sci+xml")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.volatileDependencies+xml", "vnd.openxmlformats-officedocument.spreadsheetml.volatileDependencies+xml")
        addMapping("message/imdn+xml", "imdn+xml")
        addMapping("application/vnd.hp-PCL", "vnd.hp-PCL")
        addMapping("application/vnd.oasis.opendocument.formula", "vnd.oasis.opendocument.formula")
        addMapping("video/rtploopback", "rtploopback")
        addMapping("application/pgp-signature", "pgp-signature")
        addMapping("application/vnd.dvb.notif-container+xml", "vnd.dvb.notif-container+xml")
        addMapping("message/vnd.wfa.wsc", "vnd.wfa.wsc")
        addMapping("image/vnd.fst", "vnd.fst")
        addMapping("application/vnd.mozilla.xul+xml", "vnd.mozilla.xul+xml")
        addMapping("application/vnd.filmit.zfc", "vnd.filmit.zfc")
        addMapping("application/vnd.3gpp-prose-pc3ch+xml", "vnd.3gpp-prose-pc3ch+xml")
        addMapping("application/vnd.vd-study", "vnd.vd-study")
        addMapping("application/vnd.shana.informed.formtemplate", "vnd.shana.informed.formtemplate")
        addMapping("application/xcap-ns+xml", "xcap-ns+xml")
        addMapping("application/vnd.lotus-organizer", "vnd.lotus-organizer")
        addMapping("audio/vnd.nuera.ecelp7470", "vnd.nuera.ecelp7470")
        addMapping("application/vnd.geogebra.tool", "vnd.geogebra.tool")
        addMapping("image/vnd.mozilla.apng", "vnd.mozilla.apng")
        addMapping("text/prs.lines.tag", "prs.lines.tag")
        addMapping("text/mizar", "mizar")
        addMapping("font/woff2", "woff2")
        addMapping("application/json", "json")
        addMapping("application/sep+xml", "sep+xml")
        addMapping("image/vnd.sealed-png", "vnd.sealed.png")
        addMapping("audio/vnd.dolby.pl2x", "vnd.dolby.pl2x")
        addMapping("application/vnd.syncml.dmtnds+xml", "vnd.syncml.dmtnds+xml")
        addMapping("image/jls", "jls")
        addMapping("application/vnd.arastra.swi", "vnd.arastra.swi - OBSOLETED in favor of application/vnd.aristanetworks.swi")
        addMapping("audio/EVS", "EVS")
        addMapping("audio/vnd.dolby.pl2z", "vnd.dolby.pl2z")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.presentation", "vnd.openxmlformats-officedocument.presentationml.presentation")
        addMapping("application/vnd.fdsn.seed", "vnd.fdsn.seed")
        addMapping("application/vnd.miele+json", "vnd.miele+json")
        addMapping("application/auth-policy+xml", "auth-policy+xml")
        addMapping("application/vnd.oma.cab-pcc+xml", "vnd.oma.cab-pcc+xml")
        addMapping("audio/vnd.presonus.multitrack", "vnd.presonus.multitrack")
        addMapping("application/smil", "smil - OBSOLETED in favor of application/smil+xml")
        addMapping("audio/vnd.cns.anp1", "vnd.cns.anp1")
        addMapping("application/efi", "efi")
        addMapping("audio/ATRAC-ADVANCED-LOSSLESS", "ATRAC-ADVANCED-LOSSLESS")
        addMapping("application/tamp-sequence-adjust", "tamp-sequence-adjust")
        addMapping("application/mxf", "mxf")
        addMapping("application/xhtml+xml", "xhtml+xml")
        addMapping("text/ecmascript", "ecmascript - OBSOLETED in favor of application/ecmascript")
        addMapping("application/H224", "H224")
        addMapping("application/vnd.chipnuts.karaoke-mmd", "vnd.chipnuts.karaoke-mmd")
        addMapping("audio/basic", "basic")
        addMapping("application/vnd.3gpp.pic-bw-small", "vnd.3gpp.pic-bw-small")
        addMapping("audio/dsr-es202050", "dsr-es202050")
        addMapping("message/news", "news - OBSOLETED by RFC5537")
        addMapping("application/LXF", "LXF")
        addMapping("image/jp2", "jp2")
        addMapping("application/mac-binhex40", "mac-binhex40")
        addMapping("video/nv", "nv")
        addMapping("application/ipp", "ipp")
        addMapping("application/vnd.oma.push", "vnd.oma.push")
        addMapping("application/dicom+json", "dicom+json")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.comments+xml", "vnd.openxmlformats-officedocument.wordprocessingml.comments+xml")
        addMapping("application/tamp-community-update-confirm", "tamp-community-update-confirm")
        addMapping("application/3gpp-ims+xml", "3gpp-ims+xml")
        addMapping("application/vnd.etsi.iptvsad-cod+xml", "vnd.etsi.iptvsad-cod+xml")
        addMapping("application/vnd.fujixerox.ddd", "vnd.fujixerox.ddd")
        addMapping("application/fastsoap", "fastsoap")
        addMapping("application/vnd.ah-barcode", "vnd.ah-barcode")
        addMapping("application/vnd.nokia.landmark+xml", "vnd.nokia.landmark+xml")
        addMapping("application/vnd.oma.bcast.sprov+xml", "vnd.oma.bcast.sprov+xml")
        addMapping("application/soap+xml", "soap+xml")
        addMapping("image/dicom-rle", "dicom-rle")
        addMapping("video/vnd.iptvforum.2dparityfec-1010", "vnd.iptvforum.2dparityfec-1010")
        addMapping("application/vnd.gmx", "vnd.gmx - DEPRECATED")
        addMapping("application/its+xml", "its+xml")
        addMapping("application/vnd.dvb.notif-ia-registration-response+xml", "vnd.dvb.notif-ia-registration-response+xml")
        addMapping("application/vnd.oipf.mippvcontrolmessage+xml", "vnd.oipf.mippvcontrolmessage+xml")
        addMapping("application/x-www-form-urlencoded", "x-www-form-urlencoded")
        addMapping("application/vnd.3gpp.mcptt-location-info+xml", "vnd.3gpp.mcptt-location-info+xml")
        addMapping("application/vnd.dpgraph", "vnd.dpgraph")
        addMapping("application/vnd.oma.cab-subs-invite+xml", "vnd.oma.cab-subs-invite+xml")
        addMapping("application/bacnet-xdd+zip", "bacnet-xdd+zip")
        addMapping("application/rls-services+xml", "rls-services+xml")
        addMapping("application/vnd.mynfc", "vnd.mynfc")
        addMapping("application/vnd.sss-ntf", "vnd.sss-ntf")
        addMapping("text/vnd.DMClientScript", "vnd.DMClientScript")
        addMapping("application/vnd.uplanet.alert-wbxml", "vnd.uplanet.alert-wbxml")
        addMapping("application/vnd.oasis.opendocument.chart", "vnd.oasis.opendocument.chart")
        addMapping("message/CPIM", "CPIM")
        addMapping("application/mmt-usd+xml", "mmt-usd+xml")
        addMapping("message/disposition-notification", "disposition-notification")
        addMapping("application/vnd.nokia.iptv.config+xml", "vnd.nokia.iptv.config+xml")
        addMapping("application/vnd.radisys.msml-dialog-transform+xml", "vnd.radisys.msml-dialog-transform+xml")
        addMapping("application/vnd.wv.csp+wbxml", "vnd.wv.csp+wbxml")
        addMapping("image/aces", "aces")
        addMapping("application/vnd.nokia.iSDS-radio-presets", "vnd.nokia.iSDS-radio-presets")
        addMapping("application/vnd.openxmlformats-officedocument.drawing+xml", "vnd.openxmlformats-officedocument.drawing+xml")
        addMapping("application/coap-group+json", "coap-group+json")
        addMapping("message/global-headers", "global-headers")
        addMapping("application/vnd.oasis.opendocument.spreadsheet", "vnd.oasis.opendocument.spreadsheet")
        addMapping("application/vnd.openxmlformats-officedocument.vmlDrawing", "vnd.openxmlformats-officedocument.vmlDrawing")
        addMapping("application/set-payment", "set-payment")
        addMapping("application/poc-settings+xml", "poc-settings+xml")
        addMapping("application/EmergencyCallData.Comment+xml", "EmergencyCallData.Comment+xml")
        addMapping("application/vnd.chemdraw+xml", "vnd.chemdraw+xml")
        addMapping("application/vnd.sealed-tiff", "vnd.sealed.tiff")
        addMapping("application/DCD", "DCD")
        addMapping("application/EDIFACT", "EDIFACT")
        addMapping("application/vnd.crick.clicker.template", "vnd.crick.clicker.template")
        addMapping("application/cdmi-domain", "cdmi-domain")
        addMapping("application/vnd.geo+json", "vnd.geo+json (OBSOLETED by [RFC7946] in favor of application/geo+json)")
        addMapping("application/vnd.ms-powerpoint.template.macroEnabled.12", "vnd.ms-powerpoint.template.macroEnabled.12")
        addMapping("audio/vnd.nuera.ecelp9600", "vnd.nuera.ecelp9600")
        addMapping("application/vnd.3gpp.SRVCC-info+xml", "vnd.3gpp.SRVCC-info+xml")
        addMapping("application/dicom", "dicom")
        addMapping("application/vnd.zzazz.deck+xml", "vnd.zzazz.deck+xml")
        addMapping("application/index.cmd", "index.cmd")
        addMapping("application/vnd.radisys.msml-audit+xml", "vnd.radisys.msml-audit+xml")
        addMapping("application/vnd.preminet", "vnd.preminet")
        addMapping("application/vnd.xacml+json", "vnd.xacml+json")
        addMapping("application/atom+xml", "atom+xml")
        addMapping("application/vnd.3gpp.pic-bw-var", "vnd.3gpp.pic-bw-var")
        addMapping("application/clue_info+xml", "clue_info+xml")
        addMapping("audio/clearmode", "clearmode")
        addMapping("application/vnd.ufdl", "vnd.ufdl")
        addMapping("application/vnd.antix.game-component", "vnd.antix.game-component")
        addMapping("audio/GSM", "GSM")
        addMapping("audio/dls", "dls")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.externalLink+xml", "vnd.openxmlformats-officedocument.spreadsheetml.externalLink+xml")
        addMapping("application/font-tdpfr", "font-tdpfr")
        addMapping("audio/EVRC1", "EVRC1")
        addMapping("video/MPV", "MPV")
        addMapping("audio/EVRC0", "EVRC0")
        addMapping("application/vnd.yamaha.through-ngn", "vnd.yamaha.through-ngn")
        addMapping("message/example", "example")
        addMapping("application/tamp-apex-update", "tamp-apex-update")
        addMapping("application/vnd.geocube+xml", "vnd.geocube+xml - OBSOLETED by request")
        addMapping("application/vnd.kenameaapp", "vnd.kenameaapp")
        addMapping("application/vnd.fujitsu.oasys", "vnd.fujitsu.oasys")
        addMapping("audio/EVRCB", "EVRCB")
        addMapping("application/vnd.visionary", "vnd.visionary")
        addMapping("application/vnd.d2l.coursepackage1p0+zip", "vnd.d2l.coursepackage1p0+zip")
        addMapping("application/xcap-diff+xml", "xcap-diff+xml")
        addMapping("text/html", "html")
        addMapping("application/vnd.lotus-freelance", "vnd.lotus-freelance")
        addMapping("application/vnd.ms-artgalry", "vnd.ms-artgalry")
        addMapping("application/vnd.ecip.rlp", "vnd.ecip.rlp")
        addMapping("application/vnd.geometry-explorer", "vnd.geometry-explorer")
        addMapping("application/vnd.motorola.flexsuite.adsi", "vnd.motorola.flexsuite.adsi")
        addMapping("audio/PCMU-WB", "PCMU-WB")
        addMapping("application/alto-endpointprop+json", "alto-endpointprop+json")
        addMapping("application/zlib", "zlib")
        addMapping("audio/example", "example")
        addMapping("application/vnd.commonspace", "vnd.commonspace")
        addMapping("audio/ip-mr_v2.5", "ip-mr_v2.5")
        addMapping("application/vnd.fujixerox.docuworks.binder", "vnd.fujixerox.docuworks.binder")
        addMapping("application/vnd.amazon.mobi8-ebook", "vnd.amazon.mobi8-ebook")
        addMapping("video/H264-SVC", "H264-SVC")
        addMapping("application/vnd.3gpp.mid-call+xml", "vnd.3gpp.mid-call+xml")
        addMapping("application/vnd.lotus-wordpro", "vnd.lotus-wordpro")
        addMapping("application/vnd.mophun.application", "vnd.mophun.application")
        addMapping("application/vnd.pcos", "vnd.pcos")
        addMapping("application/vnd.ecowin.seriesrequest", "vnd.ecowin.seriesrequest")
        addMapping("application/vnd.sealed-xls", "vnd.sealed.xls")
        addMapping("application/dicom+xml", "dicom+xml")
        addMapping("application/vnd.neurolanguage.nlu", "vnd.neurolanguage.nlu")
        addMapping("application/vnd.gridmp", "vnd.gridmp")
        addMapping("application/urc-uisocketdesc+xml", "urc-uisocketdesc+xml")
        addMapping("application/vnd.powerbuilder6-s", "vnd.powerbuilder6-s")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.dialogsheet+xml", "vnd.openxmlformats-officedocument.spreadsheetml.dialogsheet+xml")
        addMapping("audio/MELP1200", "MELP1200")
        addMapping("application/vnd.sbm.mid2", "vnd.sbm.mid2")
        addMapping("video/example", "example")
        addMapping("application/vnd.ms-cab-compressed", "vnd.ms-cab-compressed")
        addMapping("application/vnd.etsi.iptvsync+xml", "vnd.etsi.iptvsync+xml")
        addMapping("application/vnd.oma.bcast.notification+xml", "vnd.oma.bcast.notification+xml")
        addMapping("video/MP2T", "MP2T")
        addMapping("application/vnd.software602.filler.form+xml", "vnd.software602.filler.form+xml")
        addMapping("application/vnd.rainstor.data", "vnd.rainstor.data")
        addMapping("application/http", "http")
        addMapping("application/vnd.uplanet.bearer-choice", "vnd.uplanet.bearer-choice")
        addMapping("audio/1d-interleaved-parityfec", "1d-interleaved-parityfec")
        addMapping("application/vnd.Mobius.DAF", "vnd.Mobius.DAF")
        addMapping("application/vnd.liberty-request+xml", "vnd.liberty-request+xml")
        addMapping("application/vnd.coreos.ignition+json", "vnd.coreos.ignition+json")
        addMapping("video/MP2P", "MP2P")
        addMapping("application/vnd.apple.mpegurl", "vnd.apple.mpegurl")
        addMapping("application/vnd.rig.cryptonote", "vnd.rig.cryptonote")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.template.main+xml", "vnd.openxmlformats-officedocument.presentationml.template.main+xml")
        addMapping("text/javascript", "javascript - OBSOLETED in favor of application/javascript")
        addMapping("application/vnd.nokia.radio-preset", "vnd.nokia.radio-preset")
        addMapping("application/vnd.hal+json", "vnd.hal+json")
        addMapping("audio/SMV0", "SMV0")
        addMapping("application/pkix-cert", "pkix-cert")
        addMapping("application/vnd.framemaker", "vnd.framemaker")
        addMapping("image/vnd.dece.graphic", "vnd.dece.graphic")
        addMapping("audio/EVRCNW0", "EVRCNW0")
        addMapping("audio/EVRCNW1", "EVRCNW1")
        addMapping("application/vnd.oasis.opendocument.image-template", "vnd.oasis.opendocument.image-template")
        addMapping("application/vnd.wmf.bootstrap", "vnd.wmf.bootstrap")
        addMapping("application/vnd.openxmlformats-officedocument.themeOverride+xml", "vnd.openxmlformats-officedocument.themeOverride+xml")
        addMapping("application/vnd.aristanetworks.swi", "vnd.aristanetworks.swi")
        addMapping("application/vnd.groove-help", "vnd.groove-help")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.commentAuthors+xml", "vnd.openxmlformats-officedocument.presentationml.commentAuthors+xml")
        addMapping("application/vnd.vidsoft.vidconference", "vnd.vidsoft.vidconference")
        addMapping("application/vnd.syncml.dm+wbxml", "vnd.syncml.dm+wbxml")
        addMapping("application/vnd.pg.osasli", "vnd.pg.osasli")
        addMapping("application/vnd.sqlite3", "vnd.sqlite3")
        addMapping("model/vnd.parasolid.transmit-text", "vnd.parasolid.transmit.text")
        addMapping("model/vnd.flatland.3dml", "vnd.flatland.3dml")
        addMapping("application/vnd.evolv.ecig.settings", "vnd.evolv.ecig.settings")
        addMapping("text/vnd.abc", "vnd.abc")
        addMapping("video/vnd.sealedmedia.softseal-mov", "vnd.sealedmedia.softseal.mov")
        addMapping("application/cpl+xml", "cpl+xml")
        addMapping("image/vnd.dvb.subtitle", "vnd.dvb.subtitle")
        addMapping("video/MP1S", "MP1S")
        addMapping("image/png", "png")
        addMapping("application/vnd.eszigno3+xml", "vnd.eszigno3+xml")
        addMapping("application/cccex", "cccex")
        addMapping("application/batch-SMTP", "batch-SMTP")
        addMapping("application/vnd.poc.group-advertisement+xml", "vnd.poc.group-advertisement+xml")
        addMapping("audio/vnd.sealedmedia.softseal-mpeg", "vnd.sealedmedia.softseal.mpeg")
        addMapping("application/vnd.onepagertamx", "vnd.onepagertamx")
        addMapping("application/vnd.musician", "vnd.musician")
        addMapping("application/vnd.msign", "vnd.msign")
        addMapping("application/vnd.ms-xpsdocument", "vnd.ms-xpsdocument")
        addMapping("application/vnd.anser-web-certificate-issue-initiation", "vnd.anser-web-certificate-issue-initiation")
        addMapping("application/vnd.onepagertamp", "vnd.onepagertamp")
        addMapping("application/vnd.sealed.3df", "vnd.sealed.3df")
        addMapping("video/vnd.nokia.mp4vr", "vnd.nokia.mp4vr")
        addMapping("application/vnd.ms-asf", "vnd.ms-asf")
        addMapping("application/vnd.stepmania.package", "vnd.stepmania.package")
        addMapping("video/H263-1998", "H263-1998")
        addMapping("text/vnd.wap.sl", "vnd.wap.sl")
        addMapping("application/vnd.hdt", "vnd.hdt")
        addMapping("text/vnd.wap.si", "vnd.wap.si")
        addMapping("application/vnd.3gpp.GMOP+xml", "vnd.3gpp.GMOP+xml")
        addMapping("application/vnd.radisys.msml-dialog-fax-detect+xml", "vnd.radisys.msml-dialog-fax-detect+xml")
        addMapping("application/vnd.wap-slc", "vnd.wap.slc")
        addMapping("application/vnd.sss-dtf", "vnd.sss-dtf")
        addMapping("application/vnd.youtube.yt", "vnd.youtube.yt")
        addMapping("text/vnd-a", "vnd.a")
        addMapping("application/vnd.oipf.cspg-hexbinary", "vnd.oipf.cspg-hexbinary")
        addMapping("application/vnd.pagerduty+json", "vnd.pagerduty+json")
        addMapping("application/vnd.shana.informed.formdata", "vnd.shana.informed.formdata")
        addMapping("application/vnd.contact.cmsg", "vnd.contact.cmsg")
        addMapping("audio/SMV-QCP", "SMV-QCP")
        addMapping("application/vnd.hp-hps", "vnd.hp-hps")
        addMapping("audio/vnd.rip", "vnd.rip")
        addMapping("application/vnd.dreamfactory", "vnd.dreamfactory")
        addMapping("application/vnd.ntt-local.sip-ta_tcp_stream", "vnd.ntt-local.sip-ta_tcp_stream")
        addMapping("application/vnd.ieee.1905", "vnd.ieee.1905")
        addMapping("text/example", "example")
        addMapping("application/vnd.etsi.iptvservice+xml", "vnd.etsi.iptvservice+xml")
        addMapping("application/whoispp-query", "whoispp-query")
        addMapping("text/vnd.si.uricatalogue", "vnd.si.uricatalogue - OBSOLETED by request")
        addMapping("application/vnd.patentdive", "vnd.patentdive")
        addMapping("application/wordperfect5.1", "wordperfect5.1")
        addMapping("application/vnd.crick.clicker", "vnd.crick.clicker")
        addMapping("video/MP4V-ES", "MP4V-ES")
        addMapping("application/simple-message-summary", "simple-message-summary")
        addMapping("application/marc", "marc")
        addMapping("application/vnd.fujixerox.ART-EX", "vnd.fujixerox.ART-EX")
        addMapping("audio/G726-40", "G726-40")
        addMapping("application/vnd.syncml.dmddf+wbxml", "vnd.syncml.dmddf+wbxml")
        addMapping("application/vnd.dvb.notif-init+xml", "vnd.dvb.notif-init+xml")
        addMapping("model/vnd.parasolid.transmit-binary", "vnd.parasolid.transmit.binary")
        addMapping("application/vnd.smaf", "vnd.smaf")
        addMapping("application/vnd.music-niff", "vnd.music-niff")
        addMapping("application/vnd.openxmlformats-officedocument.drawingml.diagramStyle+xml", "vnd.openxmlformats-officedocument.drawingml.diagramStyle+xml")
        addMapping("application/geoxacml+xml", "geoxacml+xml")
        addMapping("application/vnd.oma.dcdc", "vnd.oma.dcdc")
        addMapping("application/mbms-user-service-description+xml", "mbms-user-service-description+xml")
        addMapping("image/vnd.fujixerox.edmics-mmr", "vnd.fujixerox.edmics-mmr")
        addMapping("application/vnd.data-vision.rdz", "vnd.data-vision.rdz")
        addMapping("application/vnd.oma.bcast.provisioningtrigger", "vnd.oma.bcast.provisioningtrigger")
        addMapping("application/vnd.collection.next+json", "vnd.collection.next+json")
        addMapping("application/vnd.sus-calendar", "vnd.sus-calendar")
        addMapping("application/riscos", "riscos")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.endnotes+xml", "vnd.openxmlformats-officedocument.wordprocessingml.endnotes+xml")
        addMapping("application/vnd.olpc-sugar", "vnd.olpc-sugar")
        addMapping("application/vnd.3gpp.state-and-event-info+xml", "vnd.3gpp.state-and-event-info+xml")
        addMapping("application/merge-patch+json", "merge-patch+json")
        addMapping("application/vnd.Mobius.MBK", "vnd.Mobius.MBK")
        addMapping("video/rtx", "rtx")
        addMapping("audio/vnd.3gpp.iufp", "vnd.3gpp.iufp")
        addMapping("application/vnd.Mobius.DIS", "vnd.Mobius.DIS")
        addMapping("application/vnd.lotus-1-2-3", "vnd.lotus-1-2-3")
        addMapping("multipart/form-data", "form-data")
        addMapping("application/vnd.gov.sk.e-form+zip", "vnd.gov.sk.e-form+zip")
        addMapping("application/vnd.dvb.esgcontainer", "vnd.dvb.esgcontainer")
        addMapping("application/vnd.novadigm.EXT", "vnd.novadigm.EXT")
        addMapping("application/vnd.clonk.c4group", "vnd.clonk.c4group")
        addMapping("application/vnd.oasis.opendocument.graphics-template", "vnd.oasis.opendocument.graphics-template")
        addMapping("application/vnd.oma.xcap-directory+xml", "vnd.oma.xcap-directory+xml")
        addMapping("application/vnd.onepagertatx", "vnd.onepagertatx")
        addMapping("application/vnd.nokia.pcd+wbxml", "vnd.nokia.pcd+wbxml")
        addMapping("audio/G726-16", "G726-16")
        addMapping("application/session-info", "session-info")
        addMapping("application/vnd.noblenet-directory", "vnd.noblenet-directory")
        addMapping("application/vnd.onepagertatp", "vnd.onepagertatp")
        addMapping("font/ttf", "ttf")
        addMapping("application/cms", "cms")
        addMapping("message/delivery-status", "delivery-status")
        addMapping("message/s-http", "s-http")
        addMapping("audio/G726-32", "G726-32")
        addMapping("application/octet-stream", "octet-stream")
        addMapping("application/activemessage", "activemessage")
        addMapping("application/vnd.3gpp.mcptt-mbms-usage-info+xml", "vnd.3gpp.mcptt-mbms-usage-info+xml")
        addMapping("application/rdf+xml", "rdf+xml")
        addMapping("video/3gpp2", "3gpp2")
        addMapping("application/vnd.dvb.ait", "vnd.dvb.ait")
        addMapping("application/davmount+xml", "davmount+xml")
        addMapping("application/patch-ops-error+xml", "patch-ops-error+xml")
        addMapping("application/vnd.windows.devicepairing", "vnd.windows.devicepairing")
        addMapping("application/index", "index")
        addMapping("audio/dsr-es201108", "dsr-es201108")
        addMapping("audio/G726-24", "G726-24")
        addMapping("application/coap-payload", "coap-payload")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.revisionLog+xml", "vnd.openxmlformats-officedocument.spreadsheetml.revisionLog+xml")
        addMapping("application/tamp-community-update", "tamp-community-update")
        addMapping("application/xcap-caps+xml", "xcap-caps+xml")
        addMapping("application/vnd.siren+json", "vnd.siren+json")
        addMapping("application/cose-key", "cose-key")
        addMapping("application/vnd.llamagraphics.life-balance.exchange+xml", "vnd.llamagraphics.life-balance.exchange+xml")
        addMapping("application/tamp-status-response", "tamp-status-response")
        addMapping("application/vnd.osa.netdeploy", "vnd.osa.netdeploy")
        addMapping("image/fits", "fits")
        addMapping("application/vnd.openxmlformats-officedocument.drawingml.diagramData+xml", "vnd.openxmlformats-officedocument.drawingml.diagramData+xml")
        addMapping("audio/ATRAC3", "ATRAC3")
        addMapping("application/set-registration-initiation", "set-registration-initiation")
        addMapping("application/mp21", "mp21")
        addMapping("multipart/related", "related")
        addMapping("application/vnd.software602.filler.form-xml-zip", "vnd.software602.filler.form-xml-zip")
        addMapping("text/strings", "strings")
        addMapping("application/vnd.heroku+json", "vnd.heroku+json")
        addMapping("application/vnd.etsi.tsl+xml", "vnd.etsi.tsl+xml")
        addMapping("application/vnd.motorola.iprm", "vnd.motorola.iprm")
        addMapping("video/DV", "DV")
        addMapping("application/vnd.openxmlformats-officedocument.drawingml.diagramLayout+xml", "vnd.openxmlformats-officedocument.drawingml.diagramLayout+xml")
        addMapping("application/ecmascript", "ecmascript")
        addMapping("application/vnd.swiftview-ics", "vnd.swiftview-ics")
        addMapping("application/vnd.kde.kpresenter", "vnd.kde.kpresenter")
        addMapping("application/vnd.kde.kontour", "vnd.kde.kontour")
        addMapping("application/vnd.uplanet.alert", "vnd.uplanet.alert")
        addMapping("image/vnd.xiff", "vnd.xiff")
        addMapping("application/vnd.collabio.xodocuments.document", "vnd.collabio.xodocuments.document")
        addMapping("application/vnd.oxli.countgraph", "vnd.oxli.countgraph")
        addMapping("application/vnd.syncml.dmddf+xml", "vnd.syncml.dmddf+xml")
        addMapping("application/PDX", "PDX")
        addMapping("text/vnd.IPTC.NITF", "vnd.IPTC.NITF")
        addMapping("application/vnd.comicbook+zip", "vnd.comicbook+zip")
        addMapping("application/dssc+der", "dssc+der")
        addMapping("application/pkcs10", "pkcs10")
        addMapping("application/pkcs12", "pkcs12")
        addMapping("text/troff", "troff")
        addMapping("image/cgm", "cgm")
        addMapping("application/vnd.radisys.msml-dialog-fax-sendrecv+xml", "vnd.radisys.msml-dialog-fax-sendrecv+xml")
        addMapping("application/vnd.cups-pdf", "vnd.cups-pdf")
        addMapping("application/vnd.sealed.net", "vnd.sealed.net")
        addMapping("application/resource-lists+xml", "resource-lists+xml")
        addMapping("application/vnd.cups-raster", "vnd.cups-raster")
        addMapping("application/vnd.etsi.pstn+xml", "vnd.etsi.pstn+xml")
        addMapping("application/atomicmail", "atomicmail")
        addMapping("application/vnd.oipf.pae.gem", "vnd.oipf.pae.gem")
        addMapping("application/vnd.mason+json", "vnd.mason+json")
        addMapping("application/vnd.sbm.cid", "vnd.sbm.cid")
        addMapping("application/EmergencyCallData.ProviderInfo+xml", "EmergencyCallData.ProviderInfo+xml")
        addMapping("application/vnd.mcd", "vnd.mcd")
        addMapping("application/xenc+xml", "xenc+xml")
        addMapping("audio/vnd.everad.plj", "vnd.everad.plj")
        addMapping("application/dns", "dns")
        addMapping("application/vnd.oasis.opendocument.text-master", "vnd.oasis.opendocument.text-master")
        addMapping("application/vnd.bluetooth.ep.oob", "vnd.bluetooth.ep.oob")
        addMapping("application/vnd.rapid", "vnd.rapid")
        addMapping("audio/MP4A-LATM", "MP4A-LATM")
        addMapping("application/vnd.xmpie.dpkg", "vnd.xmpie.dpkg")
        addMapping("text/rtx", "rtx")
        addMapping("application/vnd.dtg.local.flash", "vnd.dtg.local.flash")
        addMapping("text/vnd.esmertec.theme-descriptor", "vnd.esmertec.theme-descriptor")
        addMapping("audio/ATRAC-X", "ATRAC-X")
        addMapping("audio/UEMCLIP", "UEMCLIP")
        addMapping("application/vnd.wrq-hp3000-labelled", "vnd.wrq-hp3000-labelled")
        addMapping("image/vnd.net-fpx", "vnd.net-fpx")
        addMapping("application/vnd.yamaha.openscoreformat.osfpvg+xml", "vnd.yamaha.openscoreformat.osfpvg+xml")
        addMapping("application/srgs+xml", "srgs+xml")
        addMapping("application/vnd.mitsubishi.misty-guard.trustweb", "vnd.mitsubishi.misty-guard.trustweb")
        addMapping("application/applefile", "applefile")
        addMapping("audio/L16", "L16")
        addMapping("text/rtf", "rtf")
        addMapping("application/vnd.pmi.widget", "vnd.pmi.widget")
        addMapping("application/vnd.sun.wadl+xml", "vnd.sun.wadl+xml")
        addMapping("audio/mpeg4-generic", "mpeg4-generic")
        addMapping("audio/vnd.dra", "vnd.dra")
        addMapping("application/news-transmission", "news-transmission")
        addMapping("application/vnd.quobject-quoxdocument", "vnd.quobject-quoxdocument")
        addMapping("application/EmergencyCallData.DeviceInfo+xml", "EmergencyCallData.DeviceInfo+xml")
        addMapping("application/vnd.capasystems-pg+json", "vnd.capasystems-pg+json")
        addMapping("text/cache-manifest", "cache-manifest")
        addMapping("application/vnd.ms-windows.wsd.oob", "vnd.ms-windows.wsd.oob")
        addMapping("application/vnd.nokia.landmarkcollection+xml", "vnd.nokia.landmarkcollection+xml")
        addMapping("application/vnd.ms-windows.nwprinting.oob", "vnd.ms-windows.nwprinting.oob")
        addMapping("application/rpki-publication", "rpki-publication")
        addMapping("application/atomdeleted+xml", "atomdeleted+xml")
        addMapping("application/vnd.adobe.fxp", "vnd.adobe.fxp")
        addMapping("application/vnd.japannet-payment-wakeup", "vnd.japannet-payment-wakeup")
        addMapping("model/vnd.gdl", "vnd.gdl")
        addMapping("application/scvp-cv-response", "scvp-cv-response")
        addMapping("application/iges", "iges")
        addMapping("audio/G729D", "G729D")
        addMapping("audio/G729E", "G729E")
        addMapping("model/vnd.collada+xml", "vnd.collada+xml")
        addMapping("video/1d-interleaved-parityfec", "1d-interleaved-parityfec")
        addMapping("model/vnd.valve.source.compiled-map", "vnd.valve.source.compiled-map")
        addMapping("audio/raptorfec", "raptorfec")
        addMapping("audio/ulpfec", "ulpfec")
        addMapping("application/lostsync+xml", "lostsync+xml")
        addMapping("audio/vnd.dts", "vnd.dts")
        addMapping("video/vnd.objectvideo", "vnd.objectvideo")
        addMapping("application/vnd.3gpp.srvcc-ext+xml", "vnd.3gpp.srvcc-ext+xml")
        addMapping("application/vnd.businessobjects", "vnd.businessobjects")
        addMapping("application/vnd.ms-works", "vnd.ms-works")
        addMapping("application/vnd.xmi+xml", "vnd.xmi+xml")
        addMapping("image/vnd.fujixerox.edmics-rlc", "vnd.fujixerox.edmics-rlc")
        addMapping("application/vnd.dna", "vnd.dna")
        addMapping("application/vnd.lotus-screencam", "vnd.lotus-screencam")
        addMapping("application/vnd.aether.imp", "vnd.aether.imp")
        addMapping("application/vnd.dynageo", "vnd.dynageo")
        addMapping("application/vnd.fluxtime.clip", "vnd.fluxtime.clip")
        addMapping("audio/L24", "L24")
        addMapping("application/vnd.ms-windows.devicepairing", "vnd.ms-windows.devicepairing")
        addMapping("application/vnd.etsi.iptvsad-bc+xml", "vnd.etsi.iptvsad-bc+xml")
        addMapping("application/1d-interleaved-parityfec", "1d-interleaved-parityfec")
        addMapping("audio/L20", "L20")
        addMapping("application/vnd.collabio.xodocuments.spreadsheet-template", "vnd.collabio.xodocuments.spreadsheet-template")
        addMapping("application/mbox", "mbox")
        addMapping("application/prs.xsf+xml", "prs.xsf+xml")
        addMapping("application/reputon+json", "reputon+json")
        addMapping("application/vnd.dvb.iptv.alfec-base", "vnd.dvb.iptv.alfec-base")
        addMapping("application/mbms-register-response+xml", "mbms-register-response+xml")
        addMapping("audio/vnd.CELP", "vnd.CELP")
        addMapping("application/cnrp+xml", "cnrp+xml")
        addMapping("application/vnd.3gpp.mcptt-info+xml", "vnd.3gpp.mcptt-info+xml")
        addMapping("application/vnd.fastcopy-disk-image", "vnd.fastcopy-disk-image")
        addMapping("application/vnd.seemail", "vnd.seemail")
        addMapping("application/scvp-vp-response", "scvp-vp-response")
        addMapping("application/vnd.century-systems.tcp_stream", "vnd.century-systems.tcp_stream")
        addMapping("application/json-seq", "json-seq")
        addMapping("application/kpml-response+xml", "kpml-response+xml")
        addMapping("application/vnd.hp-PCLXL", "vnd.hp-PCLXL")
        addMapping("application/vnd.novadigm.EDM", "vnd.novadigm.EDM")
        addMapping("audio/fwdred", "fwdred")
        addMapping("application/vnd.etsi.iptvcommand+xml", "vnd.etsi.iptvcommand+xml")
        addMapping("application/vnd.novadigm.EDX", "vnd.novadigm.EDX")
        addMapping("application/vnd.muvee.style", "vnd.muvee.style")
        addMapping("application/xml-dtd", "xml-dtd")
        addMapping("application/vnd.collection+json", "vnd.collection+json")
        addMapping("audio/vnd.dolby.heaac.1", "vnd.dolby.heaac.1")
        addMapping("audio/vnd.dolby.heaac.2", "vnd.dolby.heaac.2")
        addMapping("application/tamp-status-query", "tamp-status-query")
        addMapping("application/rdap+json", "rdap+json")
        addMapping("application/vnd.etsi.tsl.der", "vnd.etsi.tsl.der")
        addMapping("application/vnd.infotech.project", "vnd.infotech.project")
        addMapping("application/vnd.radisys.msml-dialog-base+xml", "vnd.radisys.msml-dialog-base+xml")
        addMapping("audio/vnd.vmx.cvsd", "vnd.vmx.cvsd")
        addMapping("application/cea-2018+xml", "cea-2018+xml")
        addMapping("application/vnd.mapbox-vector-tile", "vnd.mapbox-vector-tile")
        addMapping("application/scim+json", "scim+json")
        addMapping("application/vnd.Mobius.MQY", "vnd.Mobius.MQY")
        addMapping("application/vnd.cyan.dean.root+xml", "vnd.cyan.dean.root+xml")
        addMapping("text/prs.fallenstein.rst", "prs.fallenstein.rst")
        addMapping("application/postscript", "postscript")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheDefinition+xml", "vnd.openxmlformats-officedocument.spreadsheetml.pivotCacheDefinition+xml")
        addMapping("application/vnd.ms-ims", "vnd.ms-ims")
        addMapping("application/vnd.ipunplugged.rcprofile", "vnd.ipunplugged.rcprofile")
        addMapping("video/3gpp-tt", "3gpp-tt")
        addMapping("application/shf+xml", "shf+xml")
        addMapping("application/vnd.3gpp-v2x-local-service-information", "vnd.3gpp-v2x-local-service-information")
        addMapping("application/vnd.smart.teacher", "vnd.smart.teacher")
        addMapping("audio/vnd.dolby.mps", "vnd.dolby.mps")
        addMapping("application/vnd.fujitsu.oasysprs", "vnd.fujitsu.oasysprs")
        addMapping("application/vnd.spotfire.sfs", "vnd.spotfire.sfs")
        addMapping("application/vnd.marlin.drm.conftoken+xml", "vnd.marlin.drm.conftoken+xml")
        addMapping("video/encaprtp", "encaprtp")
        addMapping("video/vnd.dvb.file", "vnd.dvb.file")
        addMapping("audio/opus", "opus")
        addMapping("application/vnd.canon-cpdl", "vnd.canon-cpdl")
        addMapping("application/vnd.nintendo.snes.rom", "vnd.nintendo.snes.rom")
        addMapping("video/vnd.sealed.mpeg4", "vnd.sealed.mpeg4")
        addMapping("video/vnd.sealed.mpeg1", "vnd.sealed.mpeg1")
        addMapping("application/CEA", "CEA")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.presProps+xml", "vnd.openxmlformats-officedocument.presentationml.presProps+xml")
        addMapping("application/vnd.firemonkeys.cloudcell", "vnd.firemonkeys.cloudcell")
        addMapping("application/prs.hpub+zip", "prs.hpub+zip")
        addMapping("application/ld+json", "ld+json")
        addMapping("application/vnd.radisys.msml-audit-conf+xml", "vnd.radisys.msml-audit-conf+xml")
        addMapping("application/vnd.yellowriver-custom-menu", "vnd.yellowriver-custom-menu")
        addMapping("application/vnd.dvb.dvbj", "vnd.dvb.dvbj")
        addMapping("application/vnd.uplanet.listcmd", "vnd.uplanet.listcmd")
        addMapping("application/vnd.ecowin.filerequest", "vnd.ecowin.filerequest")
        addMapping("application/csrattrs", "csrattrs")
        addMapping("audio/telephone-event", "telephone-event")
        addMapping("application/vemmi", "vemmi")
        addMapping("application/n-triples", "n-triples")
        addMapping("application/smil+xml", "smil+xml")
        addMapping("application/vnd.cybank", "vnd.cybank")
        addMapping("text/markdown", "markdown")
        addMapping("application/dash+xml", "dash+xml")
        addMapping("application/vnd.micrografx-igx", "vnd.micrografx.igx")
        addMapping("application/vnd.recordare.musicxml+xml", "vnd.recordare.musicxml+xml")
        addMapping("image/vnd-wap-wbmp", "vnd.wap.wbmp")
        addMapping("model/vnd.moml+xml", "vnd.moml+xml")
        addMapping("application/calendar+xml", "calendar+xml")
        addMapping("application/vnd.frogans.fnc", "vnd.frogans.fnc")
        addMapping("application/vnd.globalplatform.card-content-mgt-response", "vnd.globalplatform.card-content-mgt-response")
        addMapping("application/vnd.wap.wmlscriptc", "vnd.wap.wmlscriptc")
        addMapping("model/iges", "iges")
        addMapping("application/mpeg4-iod", "mpeg4-iod")
        addMapping("application/vnd.ntt-local.sip-ta_remote", "vnd.ntt-local.sip-ta_remote")
        addMapping("application/xcap-att+xml", "xcap-att+xml")
        addMapping("application/vnd.cups-ppd", "vnd.cups-ppd")
        addMapping("application/vnd.publishare-delta-tree", "vnd.publishare-delta-tree")
        addMapping("multipart/report", "report")
        addMapping("application/vnd.qualcomm.brew-app-res", "vnd.qualcomm.brew-app-res")
        addMapping("application/vnd.tri.onesource", "vnd.tri.onesource")
        addMapping("application/gzip", "gzip")
        addMapping("video/BMPEG", "BMPEG")
        addMapping("audio/sp-midi", "sp-midi")
        addMapping("text/vnd.net2phone.commcenter.command", "vnd.net2phone.commcenter.command")
        addMapping("application/vnd.motorola.flexsuite.gotap", "vnd.motorola.flexsuite.gotap")
        addMapping("application/cybercash", "cybercash")
        addMapping("video/smpte291", "smpte291")
        addMapping("application/vnd.openxmlformats-officedocument.theme+xml", "vnd.openxmlformats-officedocument.theme+xml")
        addMapping("application/vnd.ms-excel.template.macroEnabled.12", "vnd.ms-excel.template.macroEnabled.12")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.notesSlide+xml", "vnd.openxmlformats-officedocument.presentationml.notesSlide+xml")
        addMapping("image/pwg-raster", "pwg-raster")
        addMapping("application/vnd.onepager", "vnd.onepager")
        addMapping("application/sbml+xml", "sbml+xml")
        addMapping("application/vnd.3gpp.ussd+xml", "vnd.3gpp.ussd+xml")
        addMapping("application/vnd.innopath.wamp.notification", "vnd.innopath.wamp.notification")
        addMapping("application/call-completion", "call-completion")
        addMapping("application/vnd.xara", "vnd.xara")
        addMapping("application/vnd.3gpp.mcptt-signed+xml", "vnd.3gpp.mcptt-signed+xml")
        addMapping("application/vnd.hal+xml", "vnd.hal+xml")
        addMapping("audio/G719", "G719")
        addMapping("application/simpleSymbolContainer", "simpleSymbolContainer")
        addMapping("application/vnd.wfa.p2p", "vnd.wfa.p2p")
        addMapping("application/mbms-associated-procedure-description+xml", "mbms-associated-procedure-description+xml")
        addMapping("application/vnd.ibm.electronic-media", "vnd.ibm.electronic-media")
        addMapping("application/vnd.pawaafile", "vnd.pawaafile")
        addMapping("application/wsdl+xml", "wsdl+xml")
        addMapping("application/vnd.epson.msf", "vnd.epson.msf")
        addMapping("application/vnd.oasis.opendocument.spreadsheet-template", "vnd.oasis.opendocument.spreadsheet-template")
        addMapping("application/trig", "trig")
        addMapping("application/vnd.3lightssoftware.imagescal", "vnd.3lightssoftware.imagescal")
        addMapping("application/vnd.xfdl.webform", "vnd.xfdl.webform")
        addMapping("audio/aptx", "aptx")
        addMapping("application/vnd.ms-office.activeX+xml", "vnd.ms-office.activeX+xml")
        addMapping("audio/G711-0", "G711-0")
        addMapping("application/vnd.FloGraphIt", "vnd.FloGraphIt")
        addMapping("application/vnd.collabio.xodocuments.document-template", "vnd.collabio.xodocuments.document-template")
        addMapping("application/vnd.HandHeld-Entertainment+xml", "vnd.HandHeld-Entertainment+xml")
        addMapping("application/vnd.intercon.formnet", "vnd.intercon.formnet")
        addMapping("multipart/encrypted", "encrypted")
        addMapping("audio/G729", "G729")
        addMapping("audio/G728", "G728")
        addMapping("application/vnd-acucobol", "vnd.acucobol")
        addMapping("multipart/appledouble", "appledouble")
        addMapping("audio/G723", "G723")
        addMapping("audio/G722", "G722")
        addMapping("application/xop+xml", "xop+xml")
        addMapping("application/vnd.openxmlformats-officedocument.drawingml.diagramColors+xml", "vnd.openxmlformats-officedocument.drawingml.diagramColors+xml")
        addMapping("video/quicktime", "quicktime")
        addMapping("video/mp4", "mp4")
        addMapping("application/vnd.collabio.xodocuments.presentation", "vnd.collabio.xodocuments.presentation")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.fontTable+xml", "vnd.openxmlformats-officedocument.wordprocessingml.fontTable+xml")
        addMapping("application/vnd.3gpp.sms", "vnd.3gpp.sms")
        addMapping("audio/vnd.nortel.vbk", "vnd.nortel.vbk")
        addMapping("image/vnd.valve.source.texture", "vnd.valve.source.texture")
        addMapping("application/mads+xml", "mads+xml")
        addMapping("application/vnd.japannet-directory-service", "vnd.japannet-directory-service")
        addMapping("application/vnd.oma.scidm.messages+xml", "vnd.oma.scidm.messages+xml")
        addMapping("application/vnd.intertrust.digibox", "vnd.intertrust.digibox")
        addMapping("text/vnd.wap-wml", "vnd.wap.wml")
        addMapping("application/p2p-overlay+xml", "p2p-overlay+xml")
        addMapping("application/vnd.paos+xml", "vnd.paos.xml")
        addMapping("application/vnd.hhe.lesson-player", "vnd.hhe.lesson-player")
        addMapping("video/3gpp", "3gpp")
        addMapping("application/media_control+xml", "media_control+xml")
        addMapping("model/vnd-dwf", "vnd.dwf")
        addMapping("audio/L8", "L8")
        addMapping("application/soap+fastinfoset", "soap+fastinfoset")
        addMapping("application/vnd.claymore", "vnd.claymore")
        addMapping("text/1d-interleaved-parityfec", "1d-interleaved-parityfec")
        addMapping("application/vnd.acucorp", "vnd.acucorp")
        addMapping("application/vnd.chess-pgn", "vnd.chess-pgn")
        addMapping("application/vnd.nervana", "vnd.nervana")
        addMapping("application/vnd.Mobius.MSL", "vnd.Mobius.MSL")
        addMapping("application/vnd.artsquare", "vnd.artsquare")
        addMapping("application/vnd.openxmlformats-officedocument.drawingml.chart+xml", "vnd.openxmlformats-officedocument.drawingml.chart+xml")
        addMapping("application/vnd.hydrostatix.sof-data", "vnd.hydrostatix.sof-data")
        addMapping("application/pidf-diff+xml", "pidf-diff+xml")
        addMapping("application/scaip+xml", "scaip+xml")
        addMapping("application/vnd.ms-officetheme", "vnd.ms-officetheme")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml", "vnd.openxmlformats-officedocument.spreadsheetml.sharedStrings+xml")
        addMapping("application/vnd.ericsson.quickcall", "vnd.ericsson.quickcall")
        addMapping("application/vnd.marlin.drm.mdcf", "vnd.marlin.drm.mdcf")
        addMapping("application/problem+json", "problem+json")
        addMapping("application/vnd.vcx", "vnd.vcx")
        addMapping("application/dca-rft", "dca-rft")
        addMapping("application/ibe-pkg-reply+xml", "ibe-pkg-reply+xml")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml", "vnd.openxmlformats-officedocument.spreadsheetml.worksheet+xml")
        addMapping("text/vnd.fly", "vnd.fly")
        addMapping("application/ATF", "ATF")
        addMapping("audio/EVRC", "EVRC")
        addMapping("application/vnd.dvb.notif-aggregate-root+xml", "vnd.dvb.notif-aggregate-root+xml")
        addMapping("application/vnd.uplanet.listcmd-wbxml", "vnd.uplanet.listcmd-wbxml")
        addMapping("application/vnd.3gpp2.bcmcsinfo+xml", "vnd.3gpp2.bcmcsinfo+xml")
        addMapping("application/inkml+xml", "inkml+xml")
        addMapping("model/vnd.gtw", "vnd.gtw")
        addMapping("application/vnd.etsi.iptvdiscovery+xml", "vnd.etsi.iptvdiscovery+xml")
        addMapping("application/vnd.motorola.flexsuite.kmr", "vnd.motorola.flexsuite.kmr")
        addMapping("application/vnd.etsi.mheg5", "vnd.etsi.mheg5")
        addMapping("application/vnd.cluetrust.cartomobile-config-pkg", "vnd.cluetrust.cartomobile-config-pkg")
        addMapping("application/vnd.oipf.spdlist+xml", "vnd.oipf.spdlist+xml")
        addMapping("video/vnd.radgamettools.bink", "vnd.radgamettools.bink")
        addMapping("application/vnd.emclient.accessrequest+xml", "vnd.emclient.accessrequest+xml")
        addMapping("application/vnd.3gpp.pic-bw-large", "vnd.3gpp.pic-bw-large")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.userNames+xml", "vnd.openxmlformats-officedocument.spreadsheetml.userNames+xml")
        addMapping("application/msc-mixer+xml", "msc-mixer+xml")
        addMapping("application/vnd.oma.poc.invocation-descriptor+xml", "vnd.oma.poc.invocation-descriptor+xml")
        addMapping("video/mpeg4-generic", "mpeg4-generic")
        addMapping("text/vnd.latex-z", "vnd.latex-z")
        addMapping("font/otf", "otf")
        addMapping("video/BT656", "BT656")
        addMapping("application/vnd.oma.cab-address-book+xml", "vnd.oma.cab-address-book+xml")
        addMapping("application/vnd.dvb.notif-ia-registration-request+xml", "vnd.dvb.notif-ia-registration-request+xml")
        addMapping("application/ibe-pp-data", "ibe-pp-data")
        addMapping("application/vnd.dvb.ipdcesgpdd", "vnd.dvb.ipdcesgpdd")
        addMapping("application/vnd.oma.poc.final-report+xml", "vnd.oma.poc.final-report+xml")
        addMapping("application/vnd.hyper-item+json", "vnd.hyper-item+json")
        addMapping("text/fwdred", "fwdred")
        addMapping("application/pkix-crl", "pkix-crl")
        addMapping("image/vnd.ms-modi", "vnd.ms-modi")
        addMapping("application/vnd.oma.bcast.imd+xml", "vnd.oma.bcast.imd+xml")
        addMapping("video/vc1", "vc1")
        addMapping("application/vnd.oasis.opendocument.presentation-template", "vnd.oasis.opendocument.presentation-template")
        addMapping("application/thraud+xml", "thraud+xml")
        addMapping("application/vnd.gov.sk.xmldatacontainer+xml", "vnd.gov.sk.xmldatacontainer+xml")
        addMapping("application/vnd-fdf", "vnd.fdf")
        addMapping("application/jose+json", "jose+json")
        addMapping("application/pkixcmp", "pkixcmp")
        addMapping("application/vnd.ibm.secure-container", "vnd.ibm.secure-container")
        addMapping("text/vnd.debian.copyright", "vnd.debian.copyright")
        addMapping("application/samlassertion+xml", "samlassertion+xml")
        addMapping("application/xcon-conference-info+xml", "xcon-conference-info+xml")
        addMapping("application/vnd.wap.sic", "vnd.wap.sic")
        addMapping("application/mbms-protection-description+xml", "mbms-protection-description+xml")
        addMapping("application/vnd.oma.cab-feature-handler+xml", "vnd.oma.cab-feature-handler+xml")
        addMapping("application/vnd.jisp", "vnd.jisp")
        addMapping("application/vnd.collabio.xodocuments.spreadsheet", "vnd.collabio.xodocuments.spreadsheet")
        addMapping("video/vnd.dlna.mpeg-tts", "vnd.dlna.mpeg-tts")
        addMapping("application/sql", "sql")
        addMapping("application/vnd.motorola.flexsuite", "vnd.motorola.flexsuite")
        addMapping("application/vnd.intergeo", "vnd.intergeo")
        addMapping("application/vnd.avalon+json", "vnd.avalon+json")
        addMapping("application/vnd.dtg.local", "vnd.dtg.local")
        addMapping("application/vnd.msa-disk-image", "vnd.msa-disk-image")
        addMapping("application/vnd.sealedmedia.softseal-html", "vnd.sealedmedia.softseal.html")
        addMapping("application/CSTAdata+xml", "CSTAdata+xml")
        addMapping("application/vnd.ms-project", "vnd.ms-project")
        addMapping("application/vnd.blink-idb-value-wrapper", "vnd.blink-idb-value-wrapper")
        addMapping("application/vnd.oasis.opendocument.text-template", "vnd.oasis.opendocument.text-template")
        addMapping("application/mbms-register+xml", "mbms-register+xml")
        addMapping("application/webpush-options+json", "webpush-options+json")
        addMapping("application/vnd.amadeus+json", "vnd.amadeus+json")
        addMapping("application/vnd.uplanet.cacheop-wbxml", "vnd.uplanet.cacheop-wbxml")
        addMapping("application/route-usd+xml", "route-usd+xml")
        addMapping("application/vnd-dart", "vnd.dart")
        addMapping("application/sieve", "sieve")
        addMapping("application/QSIG", "qsig")
        addMapping("audio/mobile-xmf", "mobile-xmf")
        addMapping("application/mods+xml", "mods+xml")
        addMapping("application/EDI-X12", "EDI-X12")
        addMapping("application/vnd.fut-misnet", "vnd.fut-misnet")
        addMapping("text/dns", "dns")
        addMapping("application/vnd.ms-powerpoint.slideshow.macroEnabled.12", "vnd.ms-powerpoint.slideshow.macroEnabled.12")
        addMapping("application/vnd.3gpp.mcptt-floor-request+xml", "vnd.3gpp.mcptt-floor-request+xml")
        addMapping("application/vnd.oasis.opendocument.formula-template", "vnd.oasis.opendocument.formula-template")
        addMapping("application/vnd-sema", "vnd.sema")
        addMapping("application/vnd.xfdl", "vnd.xfdl")
        addMapping("application/vnd.yaoweme", "vnd.yaoweme")
        addMapping("application/vnd.groove-account", "vnd.groove-account")
        addMapping("application/vnd.ms-powerpoint", "vnd.ms-powerpoint")
        addMapping("application/vnd.dzr", "vnd.dzr")
        addMapping("image/tiff", "tiff")
        addMapping("application/vnd.dece.unspecified", "vnd.dece.unspecified")
        addMapping("application/vnd.uplanet.channel", "vnd.uplanet.channel")
        addMapping("application/vnd.wt.stf", "vnd.wt.stf")
        addMapping("application/vnd.etsi.asic-e+zip", "vnd.etsi.asic-e+zip")
        addMapping("application/vnd.SimTech-MindMapper", "vnd.SimTech-MindMapper")
        addMapping("application/vnd.oipf.dae.xhtml+xml", "vnd.oipf.dae.xhtml+xml")
        addMapping("application/widget", "widget")
        addMapping("audio/MELP2400", "MELP2400")
        addMapping("application/vnd.dataresource+json", "vnd.dataresource+json")
        addMapping("audio/BV16", "BV16")
        addMapping("application/vnd.oma.dd2+xml", "vnd.oma.dd2+xml")
        addMapping("application/vnd.hp-hpid", "vnd.hp-hpid")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.viewProps+xml", "vnd.openxmlformats-officedocument.presentationml.viewProps+xml")
        addMapping("application/vnd.oma.dcd", "vnd.oma.dcd")
        addMapping("image/example", "example")
        addMapping("application/vnd.uplanet.list", "vnd.uplanet.list")
        addMapping("application/vnd.las.las+xml", "vnd.las.las+xml")
        addMapping("application/vnd.sss-cod", "vnd.sss-cod")
        addMapping("text/vnd.dvb.subtitle", "vnd.dvb.subtitle")
        addMapping("application/vnd.cups-postscript", "vnd.cups-postscript")
        addMapping("application/vnd.isac.fcs", "vnd.isac.fcs")
        addMapping("application/calendar+json", "calendar+json")
        addMapping("application/vnd.hyperdrive+json", "vnd.hyperdrive+json")
        addMapping("application/vnd.stepmania.stepchart", "vnd.stepmania.stepchart")
        addMapping("application/vnd.uoml+xml", "vnd.uoml+xml")
        addMapping("application/vnd.irepository.package+xml", "vnd.irepository.package+xml")
        addMapping("application/vnd.oma.lwm2m+json", "vnd.oma.lwm2m+json")
        addMapping("audio/ac3", "ac3")
        addMapping("audio/tone", "tone")
        addMapping("application/vnd.ctct.ws+xml", "vnd.ctct.ws+xml")
        addMapping("video/mj2", "mj2")
        addMapping("application/vnd.doremir.scorecloud-binary-document", "vnd.doremir.scorecloud-binary-document")
        addMapping("application/ogg", "ogg")
        addMapping("application/vnd.amiga.ami", "vnd.amiga.ami")
        addMapping("video/vnd.fvt", "vnd.fvt")
        addMapping("text/vnd.ascii-art", "vnd.ascii-art")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slideUpdateInfo+xml", "vnd.openxmlformats-officedocument.presentationml.slideUpdateInfo+xml")
        addMapping("audio/BV32", "BV32")
        addMapping("audio/vnd.digital-winds", "vnd.digital-winds")
        addMapping("application/vnd.ibm.rights-management", "vnd.ibm.rights-management")
        addMapping("audio/vorbis-config", "vorbis-config")
        addMapping("video/vnd.dece.video", "vnd.dece.video")
        addMapping("video/vnd.motorola.videop", "vnd.motorola.videop")
        addMapping("image/prs.btif", "prs.btif")
        addMapping("application/vnd.solent.sdkm+xml", "vnd.solent.sdkm+xml")
        addMapping("application/vnd.canon-lips", "vnd.canon-lips")
        addMapping("application/atomcat+xml", "atomcat+xml")
        addMapping("application/alto-directory+json", "alto-directory+json")
        addMapping("application/vnd.dtg.local-html", "vnd.dtg.local.html")
        addMapping("application/vnd.dvb.ipdcroaming", "vnd.dvb.ipdcroaming")
        addMapping("application/vnd.oma.group-usage-list+xml", "vnd.oma.group-usage-list+xml")
        addMapping("application/AML", "AML")
        addMapping("application/vnd.oma.cab-user-prefs+xml", "vnd.oma.cab-user-prefs+xml")
        addMapping("application/tve-trigger", "tve-trigger")
        addMapping("application/yang-patch+xml", "yang-patch+xml")
        addMapping("application/vnd.Kinar", "vnd.Kinar")
        addMapping("application/vnd.openblox.game-binary", "vnd.openblox.game-binary")
        addMapping("image/vnd.dxf", "vnd.dxf")
        addMapping("application/vnd.oma-scws-config", "vnd.oma-scws-config")
        addMapping("application/vnd.etsi.simservs+xml", "vnd.etsi.simservs+xml")
        addMapping("application/vnd.insors.igm", "vnd.insors.igm")
        addMapping("application/vnd.epson.esf", "vnd.epson.esf")
        addMapping("application/vnd.fuzzysheet", "vnd.fuzzysheet")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.table+xml", "vnd.openxmlformats-officedocument.spreadsheetml.table+xml")
        addMapping("application/hyperstudio", "hyperstudio")
        addMapping("application/vnd.americandynamics.acc", "vnd.americandynamics.acc")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slide", "vnd.openxmlformats-officedocument.presentationml.slide")
        addMapping("audio/MELP", "MELP")
        addMapping("application/vcard+json", "vcard+json")
        addMapping("audio/mp4", "mp4")
        addMapping("application/dec-dx", "dec-dx")
        addMapping("application/vnd.mseq", "vnd.mseq")
        addMapping("application/framework-attributes+xml", "framework-attributes+xml")
        addMapping("application/pdf", "pdf")
        addMapping("multipart/voice-message", "voice-message")
        addMapping("application/vnd.oma.bcast.sgboot", "vnd.oma.bcast.sgboot")
        addMapping("application/vnd.cirpack.isdn-ext", "vnd.cirpack.isdn-ext")
        addMapping("video/vnd.iptvforum.ttsmpeg2", "vnd.iptvforum.ttsmpeg2")
        addMapping("application/EmergencyCallData.eCall.MSD", "EmergencyCallData.eCall.MSD")
        addMapping("audio/vnd.rhetorex.32kadpcm", "vnd.rhetorex.32kadpcm")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.footer+xml", "vnd.openxmlformats-officedocument.wordprocessingml.footer+xml")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml-template", "vnd.openxmlformats-officedocument.presentationml.template")
        addMapping("application/vnd.openstreetmap.data+xml", "vnd.openstreetmap.data+xml")
        addMapping("image/vnd.mix", "vnd.mix")
        addMapping("application/vnd.sealedmedia.softseal-pdf", "vnd.sealedmedia.softseal.pdf")
        addMapping("multipart/byteranges", "byteranges")
        addMapping("audio/MELP600", "MELP600")
        addMapping("application/vnd.verimatrix.vcas", "vnd.verimatrix.vcas")
        addMapping("video/rtp-enc-aescm128", "rtp-enc-aescm128")
        addMapping("application/vnd.umajin", "vnd.umajin")
        addMapping("video/H264-RCDO", "H264-RCDO")
        addMapping("application/vnd.dir-bi.plate-dl-nosuffix", "vnd.dir-bi.plate-dl-nosuffix")
        addMapping("application/vnd.minisoft-hp3000-save", "vnd.minisoft-hp3000-save")
        addMapping("application/vnd.nearst.inv+json", "vnd.nearst.inv+json")
        addMapping("application/vnd.jsk.isdn-ngn", "vnd.jsk.isdn-ngn")
        addMapping("application/vnd.yamaha.smaf-phrase", "vnd.yamaha.smaf-phrase")
        addMapping("application/vnd.motorola.flexsuite.ttc", "vnd.motorola.flexsuite.ttc")
        addMapping("application/xml-patch+xml", "xml-patch+xml")
        addMapping("application/ODA", "oda")
        addMapping("image/g3fax", "g3fax")
        addMapping("application/vnd.meridian-slingshot", "vnd.meridian-slingshot")
        addMapping("message/feedback-report", "feedback-report")
        addMapping("application/cdni", "cdni")
        addMapping("application/resource-lists-diff+xml", "resource-lists-diff+xml")
        addMapping("application/vnd.collabio.xodocuments.presentation-template", "vnd.collabio.xodocuments.presentation-template")
        addMapping("image/vnd.dwg", "vnd.dwg")
        addMapping("audio/dsr-es202211", "dsr-es202211")
        addMapping("application/vnd.groove-tool-message", "vnd.groove-tool-message")
        addMapping("audio/DAT12", "DAT12")
        addMapping("text/rfc822-headers", "rfc822-headers")
        addMapping("application/oxps", "oxps")
        addMapping("application/tnauthlist", "tnauthlist")
        addMapping("application/vnd.mfmp", "vnd.mfmp")
        addMapping("audio/vnd.dvb.file", "vnd.dvb.file")
        addMapping("application/ODX", "ODX")
        addMapping("application/vnd.oasis.opendocument.chart-template", "vnd.oasis.opendocument.chart-template")
        addMapping("video/H263-2000", "H263-2000")
        addMapping("audio/vnd.cisco.nse", "vnd.cisco.nse")
        addMapping("audio/dsr-es202212", "dsr-es202212")
        addMapping("application/vnd.shana.informed.package", "vnd.shana.informed.package")
        addMapping("application/fits", "fits")
        addMapping("application/vnd.osgeo.mapguide.package", "vnd.osgeo.mapguide.package")
        addMapping("application/route-s-tsid+xml", "route-s-tsid+xml")
        addMapping("application/vnd.openxmlformats-officedocument.customXmlProperties+xml", "vnd.openxmlformats-officedocument.customXmlProperties+xml")
        addMapping("application/vnd.adobe.partial-upload", "vnd.adobe.partial-upload")
        addMapping("application/vnd.etsi.cug+xml", "vnd.etsi.cug+xml")
        addMapping("application/vnd.fints", "vnd.fints")
        addMapping("audio/LPC", "LPC")
        addMapping("application/vnd.efi-iso", "vnd.efi.iso")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml-template", "vnd.openxmlformats-officedocument.spreadsheetml.template")
        addMapping("application/cfw", "cfw")
        addMapping("application/vnd.ms-3mfdocument", "vnd.ms-3mfdocument")
        addMapping("application/vnd.oma-scws-http-response", "vnd.oma-scws-http-response")
        addMapping("application/commonground", "commonground")
        addMapping("text/vnd.graphviz", "vnd.graphviz")
        addMapping("text/vnd.in3d.spot", "vnd.in3d.spot")
        addMapping("application/vnd.yamaha.remote-setup", "vnd.yamaha.remote-setup")
        addMapping("application/vnd.openxmlformats-officedocument.custom-properties+xml", "vnd.openxmlformats-officedocument.custom-properties+xml")
        addMapping("application/vnd.powerbuilder7", "vnd.powerbuilder7")
        addMapping("application/vnd.omads-file+xml", "vnd.omads-file+xml")
        addMapping("application/vnd.ims.lti.v2.toolsettings+json", "vnd.ims.lti.v2.toolsettings+json")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.sheetMetadata+xml", "vnd.openxmlformats-officedocument.spreadsheetml.sheetMetadata+xml")
        addMapping("application/vnd.vsf", "vnd.vsf")
        addMapping("application/jwk-set+json", "jwk-set+json")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.comments+xml", "vnd.openxmlformats-officedocument.presentationml.comments+xml")
        addMapping("application/vnd.ctc-posml", "vnd.ctc-posml")
        addMapping("application/vnd.bint.med-content", "vnd.bint.med-content")
        addMapping("application/cdmi-container", "cdmi-container")
        addMapping("application/mrb-consumer+xml", "mrb-consumer+xml")
        addMapping("application/vnd.dvb.notif-ia-msglist+xml", "vnd.dvb.notif-ia-msglist+xml")
        addMapping("application/vnd.powerbuilder6", "vnd.powerbuilder6")
        addMapping("application/vnd.ntt-local.content-share", "vnd.ntt-local.content-share")
        addMapping("application/EDI-consent", "EDI-consent")
        addMapping("application/vnd.sealed-ppt", "vnd.sealed.ppt")
        addMapping("message/global-disposition-notification", "global-disposition-notification")
        addMapping("application/vnd.oma.bcast.sgdu", "vnd.oma.bcast.sgdu")
        addMapping("application/fastinfoset", "fastinfoset")
        addMapping("application/vnd.oipf.spdiscovery+xml", "vnd.oipf.spdiscovery+xml")
        addMapping("application/mpeg4-iod-xmt", "mpeg4-iod-xmt")
        addMapping("application/vnd.gerber", "vnd.gerber")
        addMapping("application/vnd.yamaha.hv-dic", "vnd.yamaha.hv-dic")
        addMapping("application/encaprtp", "encaprtp")
        addMapping("text/vnd.wap.wmlscript", "vnd.wap.wmlscript")
        addMapping("model/example", "example")
        addMapping("application/vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml", "vnd.openxmlformats-officedocument.spreadsheetml.sheet.main+xml")
        addMapping("video/vnd.ms-playready.media.pyv", "vnd.ms-playready.media.pyv")
        addMapping("application/vnd.openxmlformats-officedocument.presentationml.slideLayout+xml", "vnd.openxmlformats-officedocument.presentationml.slideLayout+xml")
        addMapping("application/remote-printing", "remote-printing")
        addMapping("application/vnd.onepagertat", "vnd.onepagertat")
        addMapping("application/vnd.webturbo", "vnd.webturbo")
        addMapping("application/vnd.radisys.msml+xml", "vnd.radisys.msml+xml")
        addMapping("image/vnd.cns.inf2", "vnd.cns.inf2")
        addMapping("application/vnd.api+json", "vnd.api+json")
        addMapping("application/vnd.wolfram.mathematica", "vnd.wolfram.mathematica")
        addMapping("application/lost+xml", "lost+xml")
        addMapping("application/urc-targetdesc+xml", "urc-targetdesc+xml")
        addMapping("application/vnd.ims.lti.v2.toolproxy+json", "vnd.ims.lti.v2.toolproxy+json")
        addMapping("application/vnd.radisys.msml-audit-conn+xml", "vnd.radisys.msml-audit-conn+xml")
        addMapping("application/vnd.piaccess.application-licence", "vnd.piaccess.application-licence")
        addMapping("audio/vnd.lucent.voice", "vnd.lucent.voice")
        addMapping("application/mpeg4-generic", "mpeg4-generic")
        addMapping("text/encaprtp", "encaprtp")
        addMapping("text/parameters", "parameters")
        addMapping("application/vnd.openxmlformats-officedocument.wordprocessingml.document.glossary+xml", "vnd.openxmlformats-officedocument.wordprocessingml.document.glossary+xml")
        addMapping("application/beep+xml", "beep+xml")
        addMapping("application/vnd.fdsn.mseed", "vnd.fdsn.mseed")
        addMapping("application/vnd.pwg-multiplexed", "vnd.pwg-multiplexed")
        addMapping("application/vnd.oma.bcast.stkm", "vnd.oma.bcast.stkm")
        addMapping("application/geo+json-seq", "geo+json-seq")
        addMapping("application/vnd.ms-lrm", "vnd.ms-lrm")
        addMapping("application/csta+xml", "csta+xml")
        addMapping("text/calendar", "calendar")
        addMapping("application/vnd.kodak-descriptor", "vnd.kodak-descriptor")
        addMapping("application/xml", "xml")
        addMapping("audio/DV", "DV")
        addMapping("application/vnd.macports.portpkg", "vnd.macports.portpkg")
        addMapping("video/vnd.dece.mobile", "vnd.dece.mobile")
        addMapping("application/vnd.evolv.ecig.profile", "vnd.evolv.ecig.profile")
        addMapping("application/vnd.marlin.drm.license+xml", "vnd.marlin.drm.license+xml")
        addMapping("application/vnd.oma.bcast.smartcard-trigger+xml", "vnd.oma.bcast.smartcard-trigger+xml")
        addMapping("application/pkcs7-signature", "pkcs7-signature")
        addMapping("application/vnd.enliven", "vnd.enliven")
        addMapping("application/vnd.immervision-ivu", "vnd.immervision-ivu")
        addMapping("multipart/example", "example")
        addMapping("application/vnd.bluetooth.le.oob", "vnd.bluetooth.le.oob")
        addMapping("application/vnd.ecowin.series", "vnd.ecowin.series")
        addMapping("application/vnd.immervision-ivp", "vnd.immervision-ivp")
        addMapping("application/pkix-pkipath", "pkix-pkipath")
        addMapping("application/slate", "slate")
        addMapping("application/vnd.zul", "vnd.zul")
        addMapping("application/vnd.xmpie.cpkg", "vnd.xmpie.cpkg")
        addMapping("audio/EVRCWB0", "EVRCWB0")
        addMapping("audio/EVRCWB1", "EVRCWB1")
        addMapping("audio/CN", "CN")
        addMapping("application/vnd.netfpx", "vnd.netfpx")
        addMapping("application/voucher-cms+json", "voucher-cms+json")
        addMapping("application/vnd.llamagraphics.life-balance.desktop", "vnd.llamagraphics.life-balance.desktop")
        addMapping("application/vnd.cendio.thinlinc.clientconf", "vnd.cendio.thinlinc.clientconf")
        addMapping("application/vnd.openxmlformats-officedocument.extended-properties+xml", "vnd.openxmlformats-officedocument.extended-properties+xml")
        addMapping("application/rpki-ghostbusters", "rpki-ghostbusters")
        addMapping("application/vnd.wolfram.player", "vnd.wolfram.player")
        addMapping("application/xcap-error+xml", "xcap-error+xml")
        addMapping("application/vnd.apache.thrift.compact", "vnd.apache.thrift.compact")
        addMapping("video/vnd.nokia.interleaved-multimedia", "vnd.nokia.interleaved-multimedia")
        addMapping("application/simple-filter+xml", "simple-filter+xml")
        addMapping("application/vnd.oasis.opendocument.graphics", "vnd.oasis.opendocument.graphics")
        addMapping("application/pskc+xml", "pskc+xml")
        addMapping("audio/rtp-enc-aescm128", "rtp-enc-aescm128")
        addMapping("application/vnd.trueapp", "vnd.trueapp")
        addMapping("multipart/header-set", "header-set")
        addMapping("audio/EVRC-QCP", "EVRC-QCP")
        addMapping("application/vnd.vel+json", "vnd.vel+json")
        addMapping("application/vnd.radisys.moml+xml", "vnd.radisys.moml+xml")
        addMapping("application/vnd.openxmlformats-officedocument.drawingml.chartshapes+xml", "vnd.openxmlformats-officedocument.drawingml.chartshapes+xml")
        addMapping("application/vnd.syncml.dm.notification", "vnd.syncml.dm.notification")
        addMapping("application/vnd.oipf.userprofile+xml", "vnd.oipf.userprofile+xml")
        addMapping("font/collection", "collection")
        addMapping("application/vnd.coffeescript", "vnd.coffeescript")
        addMapping("application/alto-costmap+json", "alto-costmap+json")
        addMapping("image/tiff-fx", "tiff-fx")
        addMapping("multipart/signed", "signed")
        addMapping("audio/rtp-midi", "rtp-midi")
        addMapping("application/vnd.etsi.overload-control-policy-dataset+xml", "vnd.etsi.overload-control-policy-dataset+xml")
        addMapping("application/vnd.rar", "vnd.rar")
        addMapping("application/mrb-publish+xml", "mrb-publish+xml")
        addMapping("application/rfc+xml", "rfc+xml")
        addMapping("application/vnd.japannet-verification", "vnd.japannet-verification")
        addMapping("application/vnd.bbf.usp.msg+json", "vnd.bbf.usp.msg+json")
        addMapping("application/vnd.efi-img", "vnd.efi.img")
        addMapping("application/vnd.adobe.xdp+xml", "vnd.adobe.xdp+xml")
        addMapping("application/vnd.nitf", "vnd.nitf")
        addMapping("application/vnd.openeye.oeb", "vnd.openeye.oeb")
        addMapping("audio/amr-wb+", "amr-wb+")
        addMapping("application/vnd.apothekende.reservation+json", "vnd.apothekende.reservation+json")
        addMapping("application/vnd.3gpp.sms+xml", "vnd.3gpp.sms+xml")
        addMapping("text/xml-external-parsed-entity", "xml-external-parsed-entity")
        addMapping("application/vnd.wfa.wsc", "vnd.wfa.wsc")
        addMapping("application/vnd.yamaha.tunnel-udpencap", "vnd.yamaha.tunnel-udpencap")
        addMapping("application/marcxml+xml", "marcxml+xml")
        addMapping("application/mbms-msk-response+xml", "mbms-msk-response+xml")
        addMapping("application/vnd.ms-fontobject", "vnd.ms-fontobject")
    }

    open protected fun addMapping(mimeType: String, fileExtension: String) {
        val fileExtensionLowerCased = fileExtension.toLowerCase()

        if(fileExtensionsToMimeTypeMap.containsKey(fileExtensionLowerCased) == false) {
            fileExtensionsToMimeTypeMap.put(fileExtensionLowerCased, LinkedHashSet())
        }

        fileExtensionsToMimeTypeMap[fileExtensionLowerCased]?.add(mimeType)


        val mimeTypeLowerCased = mimeType.toLowerCase()

        if(mimeTypesMap.containsKey(mimeTypeLowerCased) == false) {
            mimeTypesMap.put(mimeTypeLowerCased, LinkedHashSet())
        }

        mimeTypesMap[mimeTypeLowerCased]?.add(fileExtension)
    }

}
