package net.dankito.utils.filesystem

import org.slf4j.LoggerFactory
import java.io.IOException
import java.nio.file.FileVisitResult
import java.nio.file.FileVisitor
import java.nio.file.Files
import java.nio.file.Path
import java.nio.file.attribute.BasicFileAttributes


open class FilesystemWalker : IFilesystemWalker {

	companion object {
		private val log = LoggerFactory.getLogger(FilesystemWalker::class.java)
	}


	override fun listFiles(startDir: Path): List<Path> {
		val discoveredFiles = mutableListOf<Path>()

		walk(startDir) { discoveredFile: Path ->
			discoveredFiles.add(discoveredFile)
		}

		return discoveredFiles
	}


	override fun walk(startDir: Path, discoveredFileCallback: (Path) -> Unit) {

		detailedWalk(startDir) { visitedFile: VisitedFile ->
			visitedFile.path?.let { discoveredFile ->
				discoveredFileCallback(discoveredFile)

				FileVisitResult.CONTINUE
			}
		}
	}


	override fun detailedWalk(startDir: Path, abortOnError: Boolean,
							  preVisitDirectory: ((directory: Path?) -> FileVisitResult)?,
							  postVisitDirectory: ((directory: Path?) -> FileVisitResult)?,
							  visitedFileCallback: (VisitedFile) -> FileVisitResult?) {

		Files.walkFileTree(startDir, object : FileVisitor<Path> {

			// files:

			override fun visitFile(file: Path?, attributes: BasicFileAttributes?): FileVisitResult {
				return visitedFileCallback(VisitedFile(file, attributes)) ?: FileVisitResult.CONTINUE
			}

			override fun visitFileFailed(file: Path?, exception: IOException?): FileVisitResult {
				log.error("Could not visit file '$file'", exception)

				visitedFileCallback(VisitedFile(file, null, exception))

				return if (abortOnError) FileVisitResult.TERMINATE else  FileVisitResult.CONTINUE
			}


			// directories:

			override fun preVisitDirectory(directory: Path?, attributes: BasicFileAttributes?): FileVisitResult {
				return preVisitDirectory?.invoke(directory)
						?: FileVisitResult.CONTINUE
			}

			override fun postVisitDirectory(directory: Path?, exception: IOException?): FileVisitResult {
				return postVisitDirectory?.invoke(directory)
						?: FileVisitResult.CONTINUE
			}

		})
	}

}