package net.dankito.utils.io

import java.io.*


open class FilesUtils {


    @Throws(Exception::class)
    open fun writeToTextFile(fileContent: String, file: File) {
        val outputStreamWriter = OutputStreamWriter(createFileOutputStream(file))

        outputStreamWriter.write(fileContent)

        outputStreamWriter.flush()
        outputStreamWriter.close()
    }

    @Throws(Exception::class)
    open fun writeToBinaryFile(fileContent: ByteArray, file: File) {
        val outputStream = createFileOutputStream(file)

        outputStream.write(fileContent)

        outputStream.flush()
        outputStream.close()
    }

    @Throws(FileNotFoundException::class)
    open fun createFileOutputStream(file: File): OutputStream {
        return FileOutputStream(file)
    }


    @Throws(Exception::class)
    open fun readFromTextFile(file: File): String? {
        val inputStream = createFileInputStream(file)

        val inputStreamReader = InputStreamReader(inputStream)
        val bufferedReader = BufferedReader(inputStreamReader)

        val fileContent = bufferedReader.use { it.readLines() }.joinToString(separator = "") { it }

        bufferedReader.close()
        inputStream.close()

        return fileContent
    }

    @Throws(Exception::class)
    open fun readFromBinaryFile(file: File): ByteArray? {
        val inputStream = createFileInputStream(file)

        val buffer = ByteArrayOutputStream()

        inputStream.copyTo(buffer, 16384)

        buffer.flush()
        inputStream.close()

        return buffer.toByteArray()
    }

    @Throws(FileNotFoundException::class)
    open fun createFileInputStream(file: File): InputStream {
        return FileInputStream(file)
    }


    open fun deleteFolderRecursively(path: File) {
        deleteRecursively(path)
    }

    protected open fun deleteRecursively(file: File) {
        if (file.isDirectory) {
            for (containingFile in file.listFiles()!!) {
                deleteRecursively(containingFile)
            }
        }

        file.delete()
    }


    open fun ensureFileInFolderExists(folder: File, filename: String, subFolderName: String?): File {
        var folderVar = folder
        subFolderName?.let { folderVar = File(folderVar, subFolderName) }

        folderVar.mkdirs()

        return File(folderVar, filename)
    }


    fun getFilesOfDirectorySorted(directory: File, returnOnlyDirectories: Boolean = false, extensionsFilters: List<String> = emptyList()): List<File>? {
        getFilesOfDirectory(directory, returnOnlyDirectories, extensionsFilters)?.let { files ->
            return files.sortedWith(fileComparator)
        }

        return null
    }

    fun getFilesOfDirectory(directory: File, returnOnlyDirectories: Boolean = false, extensionsFilters: List<String> = emptyList()): List<File>? {
        val alsoReturnFiles = ! returnOnlyDirectories

        if(extensionsFilters.isEmpty() && alsoReturnFiles) {
            return directory.listFiles()?.toList() // listFiles() can return null, e.g when not having rights to read this directory
        }

        return directory.listFiles { file ->
            val normalizedExtensionsFilters = normalizeExtensionFilters(extensionsFilters)

            file?.let {
                return@listFiles file.isDirectory ||
                        (alsoReturnFiles && normalizedExtensionsFilters.contains(it.extension.toLowerCase()))
            }

            false
        }?.toList()
    }

    /**
     * Removes '*.' at start of extension filter on lower cases extension
     */
    fun normalizeExtensionFilters(extensionsFilters: List<String>): List<String> {
        return extensionsFilters.map {
            var normalizedFilter = it

            if(normalizedFilter.startsWith('*')) {
                normalizedFilter = normalizedFilter.substring(1)
            }

            if(normalizedFilter.startsWith('.')) {
                normalizedFilter = normalizedFilter.substring(1)
            }

            normalizedFilter.toLowerCase()
        }
    }


    private val fileComparator = Comparator<File> { file0, file1 ->
        if(file0 != null && file1 == null) {
            return@Comparator -1
        }
        else if(file0 == null && file1 != null) {
            return@Comparator 1
        }
        else if(file0 == null && file1 == null) {
            return@Comparator 0
        }

        if(file0.isDirectory && file1.isDirectory == false) { // list directories before files
            return@Comparator -1
        }
        else if(file0.isDirectory == false && file1.isDirectory) {
            return@Comparator 1
        }

        return@Comparator file0.name.compareTo(file1.name, true)
    }

}