package net.customware.license.confluence;

import com.atlassian.confluence.core.ConfluenceActionSupport;
import com.opensymphony.webwork.ServletActionContext;
import com.opensymphony.webwork.dispatcher.multipart.MultiPartRequestWrapper;
import de.schlichtherle.license.LicenseContent;
import de.schlichtherle.license.LicenseManager;
import net.customware.license.support.util.LicenseUtils;
import org.apache.commons.codec.binary.Base64;
import org.apache.commons.lang.StringUtils;

import java.io.File;
import java.io.FileOutputStream;
import java.util.MissingResourceException;

/**
 * This class provides the basline for allowing plugins to install their own
 * license key.
 *
 * @author David Peterson
 */
public abstract class AbstractLicenseAction extends ConfluenceActionSupport {
    private static final long MILLIS_PER_DAY = 24 * 60 * 60 * 1000;

    private LicenseManager licenseManager = null;

    private MultiPartRequestWrapper multiPartRequest;

    private String install;

    private String uninstall;

    private String licenseText;

    protected MultiPartRequestWrapper getMultiPartRequest() {
        if (multiPartRequest == null && ServletActionContext.getRequest() instanceof MultiPartRequestWrapper)
            multiPartRequest = (MultiPartRequestWrapper) ServletActionContext.getRequest();
        return multiPartRequest;
    }

    @Override
    public String execute() {
        if (uninstall != null)
            return doUninstall();
        if (install != null)
            return doInstall();

        return INPUT;
    }

    private String doUninstall() {
        try {
            getLicenseManager().uninstall();
            return SUCCESS;
        } catch (Exception e) {
            e.printStackTrace();
            addActionError(getText(e.getLocalizedMessage()));
            return ERROR;
        }
    }

    private String doInstall() {
        try {
            File file = null;

            File[] licenseFiles = getMultiPartRequest().getFiles("licenseFile");
            if (licenseFiles == null || (licenseFiles.length >= 1 && licenseFiles[0] == null)) {
                if (StringUtils.isNotBlank(licenseText)) {
                    file = File.createTempFile("plugin_", ".key");
                    FileOutputStream out = new FileOutputStream(file);
                    out.write(Base64.decodeBase64(licenseText.getBytes("ascii")));
                    out.close();
                } else {
                    addActionError(getText("license.error.licenseRequired"));
                    return ERROR;
                }
            } else {
                file = licenseFiles[0];
            }

            getLicenseManager().install(file);

            file.delete();
        } catch (Exception e) {
            try {
                addActionError(getText(e.getLocalizedMessage()));
                e.printStackTrace();
            } catch (MissingResourceException e2) {
                addActionError(getText(e.getMessage()));
            }
            return ERROR;
        }

        return SUCCESS;
    }

    public LicenseContent getLicenseContent() {
        try {
            return getLicenseManager().verify();
        } catch (Exception e) {
            return null;
        }
    }

    public String getLicenseHolder() {
        LicenseContent content = getLicenseContent();
        if (content != null) {
            return LicenseUtils.formatX500Principal(content.getHolder());
        }
        return null;
    }

    public String getLicenseIssuer() {
        LicenseContent content = getLicenseContent();
        if (content != null) {
            return LicenseUtils.formatX500Principal(content.getIssuer());
        }
        return null;
    }

    public int getLicenseExpiring() {
        LicenseContent lc = getLicenseContent();
        if (lc != null && lc.getNotAfter() != null) {
            long now = System.currentTimeMillis();
            long expires = lc.getNotAfter().getTime();

            return (int) ((expires - now) / MILLIS_PER_DAY);
        }
        return Integer.MAX_VALUE;
    }

    public String getInstall() {
        return install;
    }

    public void setInstall(String upload) {
        this.install = upload;
    }

    protected LicenseManager getLicenseManager() {
        if (licenseManager == null)
            licenseManager = createLicenseManager();

        return licenseManager;
    }

    public String getUninstall() {
        return uninstall;
    }

    public void setUninstall(String uninstall) {
        this.uninstall = uninstall;
    }

    public String getAbsoluteEulaPath() {
        String eulaPath = getEulaPath();
        if (eulaPath != null && eulaPath.startsWith("/")) {
            eulaPath = getGlobalSettings().getBaseUrl() + eulaPath;
        }
        return eulaPath;
    }

    /**
     * Called to create the license manager for this plugin.
     *
     * @return The license manager implementation.
     */
    protected abstract LicenseManager createLicenseManager();

    /**
     * Returns the name of the plugin, for use in the user interface.
     * Implementors may wish to use the {@link #getText(String)} method(s) to
     * internationalise the title.
     *
     * @return The plugin name.
     */
    public abstract String getPluginName();

    /**
     * Returns the Confluence-relative path to the EULA text.
     *
     * @return The EULA path.
     */
    public abstract String getEulaPath();

    public String getLicenseText() {
        return licenseText;
    }

    public void setLicenseText(String licenseText) {
        this.licenseText = licenseText;
    }
}
