package net.customware.license.confluence;

import org.apache.log4j.Logger;
import org.randombits.support.confluence.ConfluenceMacro;
import org.randombits.support.confluence.MacroInfo;
import org.randombits.support.confluence.ServletAssistant;

import com.atlassian.confluence.macro.MacroExecutionException;
import com.atlassian.confluence.xhtml.api.XhtmlContent;
import com.atlassian.renderer.v2.macro.MacroException;

import de.schlichtherle.license.LicenseManager;
import de.schlichtherle.license.NoLicenseInstalledException;
import org.randombits.support.core.env.EnvironmentAssistant;
import org.randombits.support.core.text.I18NAssistant;

/**
 * This is an extension of the {@link ConfluenceMacro} which checks for a valid
 * license being installed before continuing with normal execution of the macro.
 * 
 * <p>
 * Implementors of this class should put their usual macro code in the
 * {@link #executeVerified(MacroInfo)} method instead of the usual
 * {@link #execute(MacroInfo)} method. This is called if the license is verified
 * to be valid.
 * 
 * @author David Peterson
 */
public abstract class LicensedConfluenceMacro extends ConfluenceMacro {
    private static final Logger LOG = Logger.getLogger( LicensedConfluenceMacro.class );

    private static LicenseManager licenseManager;
    private I18NAssistant i18nAssistant;

    @Override protected final String execute( MacroInfo info ) throws MacroExecutionException {
        try {
            getLicenseManager().verify();
        } catch ( NoLicenseInstalledException e ) {
            LOG.info( e.getLocalizedMessage() );
            return executeUnverified( info, e );
        } catch ( Exception e ) {
            LOG.error( e );
            return executeUnverified( info, e );
        }

        return executeVerified( info );
    }
	
	public LicensedConfluenceMacro(EnvironmentAssistant environmentAssistant, XhtmlContent xhtmlContent, I18NAssistant i18nAssistant){
		    super(environmentAssistant, xhtmlContent);
		    this.i18nAssistant = i18nAssistant;
	}

    /**
     * This method is executed when verification fails. It can be overridden if
     * macro subclasses wish to change the default error handling behaviour.
     * 
     * <p>
     * The default behaviour is to simply display the exception error message in
     * a highlighted box.
     * 
     * @param info
     *            The macro info.
     * @param e
     *            The exception thrown when verifying.
     * @return The XHTML code for the error message.
     * @throws MacroException
     *             if there is a problem while rendering the macro.
     */
    protected String executeUnverified( MacroInfo info, Exception e ) throws MacroExecutionException {
        return "<p><span class='error'>"
                + i18nAssistant.getText( "license.error.licenseUnverified",
                        new String[]{e.getLocalizedMessage()} ) + "</span></p>";
    }

    /**
     * This method is called when the plugin has its license successfully
     * verified - that is, the macro can be used.
     * 
     * @param info
     *            The macro info.
     * @return The executed macro content.
     * @throws MacroException
     *             if there is a problem while executing the macro.
     */
    protected abstract String executeVerified( MacroInfo info ) throws MacroExecutionException;

    /**
     * Provides the license manager for this macro. If the macro requires the
     * LicenseManager for some reason, this method should be called rather than
     * the {@link #createLicenseManager()} method, purely for performance
     * reasons.
     * 
     * @return the current license manager.
     */
    protected LicenseManager getLicenseManager() {
        if ( licenseManager == null )
            licenseManager = createLicenseManager();
        return licenseManager;
    }

    /**
     * This method is called to create the license manager. If the macro needs
     * access to the license manager for any reason it should call
     * {@link #getLicenseManager()} instead of this method, for performance
     * reasons.
     * 
     * @return the new license manager.
     */
    protected abstract LicenseManager createLicenseManager();

}
