package net.customware.license.confluence;

import java.util.Map;

import org.apache.log4j.Logger;

import com.atlassian.renderer.RenderContext;
import com.atlassian.renderer.v2.macro.BaseMacro;
import com.atlassian.renderer.v2.macro.MacroException;

import de.schlichtherle.license.LicenseManager;
import de.schlichtherle.license.NoLicenseInstalledException;

/**
 * This is an extension of the {@link BaseMacro} which checks for a valid
 * license being installed before continuing with normal execution of the macro.
 * 
 * <p>
 * Implementors of this class should put their usual macro code in the
 * {@link #executeVerified(Map, String, RenderContext)} method instead of the
 * usual {@link #execute(Map, String, RenderContext)} method. This is called if
 * the license is verified to be valid.
 * 
 * @author David Peterson
 */
public abstract class LicensedBaseMacro extends BaseMacro {
    private static final Logger LOG = Logger.getLogger( LicensedBaseMacro.class );

    private static LicenseManager licenseManager;

    @SuppressWarnings("unchecked") 
    public final String execute( Map params, String body, RenderContext ctx ) throws MacroException {
        try {
            getLicenseManager().verify();
        } catch ( NoLicenseInstalledException e ) {
            LOG.info( e.getLocalizedMessage() );
            return executeUnverified( params, body, ctx, e );
        } catch ( Exception e ) {
            LOG.error( e );
            return executeUnverified( params, body, ctx, e );
        }

        return executeVerified( params, body, ctx );
    }

    /**
     * This method is executed when verification fails. It can be overridden if
     * macro subclasses wish to change the default error handling behaviour.
     * 
     * <p>
     * The default behaviour is to simply display the exception error message in
     * a highlighted box.
     * 
     * @param params
     *            The macro parameters.
     * @param body
     *            The macro body text.
     * @param ctx
     *            The render context.
     * @param e
     *            The exception thrown when verifying.
     * @return The XHTML code for the error message.
     * @throws MacroException
     *             if there is a problem while rendering the macro.
     */
    protected String executeUnverified( Map<String, String> params, String body, RenderContext ctx, Exception e )
            throws MacroException {
        return "<p><span class='error'>The license could not be verified: " + e.getLocalizedMessage() + "</span></p>";
    }

    /**
     * Executed if the current license is verified - that is, the macro can
     * execute successfully.
     * 
     * @param params
     *            The macro parameters.
     * @param body
     *            The macro body.
     * @param ctx
     *            The render context.
     * @return The rendered macro.
     * @throws MacroException
     *             if there is a problem while executing the macro.
     */
    protected abstract String executeVerified( Map<String, String> params, String body, RenderContext ctx ) throws MacroException;

    protected LicenseManager getLicenseManager() {
        if ( licenseManager == null )
            licenseManager = createLicenseManager();
        return licenseManager;
    }

    /**
     * This method is called to initially create the license manager. If the
     * macro needs access to the license manager for any reason it should call
     * {@link #getLicenseManager()} instead of this method, for performance
     * reasons.
     * 
     * @return the new license manager.
     */
    protected abstract LicenseManager createLicenseManager();
}
