package net.customware.license.confluence;

import java.util.Collection;
import java.util.List;

import net.customware.license.atlassian.AtlassianLicenseParam;
import net.customware.license.atlassian.DefaultLicenseContext;
import net.customware.license.atlassian.LicenseContext;
import net.customware.license.confluence.prefs.ConfluenceSystemPreferences;
import net.customware.license.confluence.prefs.ConfluenceUserPreferences;

import com.atlassian.confluence.setup.ConfluenceBootstrapConstants;
import com.atlassian.license.LicenseType;
import com.atlassian.license.applications.confluence.ConfluenceLicenseTypeStore;

/**
 * An implementation of the {@link SimpleLicenseManager} with defaults
 * appropriate for most Confluence plugins. Specifically:
 * 
 * <ul>
 * <li>Require one license for the whole server (including clusters if
 * available).</li>
 * <li>Store the license key in Bandana.</li>
 * </ul>
 * 
 * @author David Peterson
 */
public abstract class ConfluenceLicenseParam extends AtlassianLicenseParam {

    private static final List<LicenseType> LICENSE_TYPES;

    static {
        LICENSE_TYPES = new java.util.ArrayList<LicenseType>();
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.ACADEMIC );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.COMMUNITY );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.DEMONSTRATION );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.DEVELOPER );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.EVALUATION );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.FULL_LICENSE );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.HOSTED );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.HOSTED_EVALUATION );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.NON_PROFIT );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.OPEN_SOURCE );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.PERSONAL );
        LICENSE_TYPES.add( ConfluenceLicenseTypeStore.TESTING );
    }

    public static LicenseType findLicenseType( int id ) {
        for ( LicenseType type : LICENSE_TYPES ) {
            if ( type.getType() == id )
                return type;
        }
        return null;
    }

    public static Collection<LicenseType> findLicenseTypes() {
        return LICENSE_TYPES;
    }

    private static LicenseContext LICENSE_CONTEXT = new DefaultLicenseContext(
            ConfluenceBootstrapConstants.DEFAULT_LICENSE_REGISTRY_KEY );

    /**
     * Returns the LicenseContext for
     * 
     * @return
     */
    public static LicenseContext getLicenseContext() {
        return LICENSE_CONTEXT;
    }

    @Override public LicenseType getLicenseType( int id ) {
        return findLicenseType( id );
    }

    /**
     * Initialises the parameters to use the system preferences by default. May
     * be overridden in the {@link #init()} method by calling
     * {@link #requireLicensePerUser()} or
     * {@link #setPreferences(java.util.prefs.Preferences)}.
     */
    @Override protected void preInit() {
        requireLicensePerServer();
    }

    protected void requireLicensePerServer() {
        setPreferences( new ConfluenceSystemPreferences().nodeForPackage( getClass() ) );
    }

    protected void requireLicensePerUser() {
        setPreferences( new ConfluenceUserPreferences().nodeForPackage( getClass() ) );
    }

    /**
     * Returns the Confluence {@link LicenseContext}.
     * 
     * @see #getLicenseContext()
     */
    @Override public Object getContext() {
        return getLicenseContext();
    }
}
