package net.croz.nrich.registry.data.controller;

import lombok.RequiredArgsConstructor;
import lombok.SneakyThrows;
import net.croz.nrich.registry.api.data.request.ListBulkRegistryRequest;
import net.croz.nrich.registry.api.data.request.ListRegistryRequest;
import net.croz.nrich.registry.api.data.service.RegistryDataService;
import net.croz.nrich.registry.data.request.CreateRegistryRequest;
import net.croz.nrich.registry.data.request.DeleteRegistryRequest;
import net.croz.nrich.registry.data.request.UpdateRegistryRequest;
import net.croz.nrich.registry.data.service.RegistryDataRequestConversionService;
import org.springframework.data.domain.Page;
import org.springframework.validation.BeanPropertyBindingResult;
import org.springframework.validation.BindException;
import org.springframework.validation.BindingResult;
import org.springframework.validation.Validator;
import org.springframework.web.bind.annotation.PostMapping;
import org.springframework.web.bind.annotation.RequestBody;
import org.springframework.web.bind.annotation.RequestMapping;
import org.springframework.web.bind.annotation.ResponseBody;

import javax.validation.Valid;
import java.util.Map;

@RequiredArgsConstructor
@RequestMapping("${croz.nrich.registry.domain:}/nrich/registry/data")
@ResponseBody
public class RegistryDataController {

    private final RegistryDataService registryDataService;

    private final RegistryDataRequestConversionService registryDataRequestConversionService;

    private final Validator validator;

    @PostMapping("list-bulk")
    public Map<String, Page<?>> listBulk(@RequestBody @Valid final ListBulkRegistryRequest request) {
        return registryDataService.listBulk(request);
    }

    @PostMapping("list")
    public <P> Page<P> list(@RequestBody @Valid final ListRegistryRequest request) {
        return registryDataService.list(request);
    }

    @PostMapping("delete")
    public <T> T delete(@RequestBody @Valid final DeleteRegistryRequest request) {
        return registryDataService.delete(request.getClassFullName(), request.getId());
    }

    @PostMapping("create")
    public <T> T create(@RequestBody @Valid final CreateRegistryRequest request) {
        final Object entityData = registryDataRequestConversionService.convertEntityDataToTyped(request);

        validateEntityData(entityData);

        return registryDataService.create(request.getClassFullName(), entityData);
    }

    @PostMapping("update")
    public <T> T update(@RequestBody @Valid final UpdateRegistryRequest request) {
        final Object entityData = registryDataRequestConversionService.convertEntityDataToTyped(request);

        validateEntityData(entityData);

        return registryDataService.update(request.getClassFullName(), request.getId(), entityData);
    }

    @SneakyThrows
    private void validateEntityData(final Object entityData) {
        final BindingResult errors = new BeanPropertyBindingResult(entityData, "entityData");

        validator.validate(entityData, errors);

        if (errors.hasErrors()) {
            throw new BindException(errors);
        }
    }
}
