/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.testsuite;

import net.craftforge.essential.client.Client;
import net.craftforge.essential.client.ClientRequest;
import net.craftforge.essential.client.HttpClient;
import net.craftforge.essential.client.LocalClient;
import net.craftforge.essential.controller.Configuration;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.essential.controller.constants.HttpMethod;
import net.craftforge.essential.testsuite.environment.mocks.SimpleAuth;
import net.craftforge.essential.testsuite.environment.mocks.SimpleLog;
import org.junit.Before;
import org.junit.Test;

import java.util.LinkedList;
import java.util.List;

/**
 * Provides a predefined client test with test suite using a local and an HTTP client.
 *
 * @author Christian Bick
 * @since 07.07.11
 */
public abstract class ClientTest {

    private String address = "localhost";
    private int port = 8080;
    private String context = "/";
    private String packagePath = "net.craftforge.essential.testsuite.environment.services";
    private RequestTestSuite testSuite;

    /**
     * Configures the HTTP client.
     *
     * @param address The server address.
     * @param port The server port.
     * @param context The server context
     */
    protected void configureHttpClient(String address, int port, String context) {
        this.address = address;
        this.port = port;
        this.context = context;
    }

    /**
     * Configures the local client.
     *
     * @param packagePath The controller's package path.
     */
    protected void configureLocalClient(String packagePath) {
        this.packagePath = packagePath;
    }

    /**
     * Gets the server address.
     *
     * @return The server address
     */
    protected String getAddress() {
        return address;
    }

    /**
     * Gets the server port.
     *
     * @return The server port
     */
    protected int getPort() {
        return port;
    }

    /**
     * Gets the server context.
     *
     * @return The server context
     */
    protected String getContext() {
        return context;
    }

    /**
     * Gets the controller's package path.
     *
     * @return The controller's package path
     */
    protected String getPackagePath() {
        return packagePath;
    }

    /**
     * Gets the controller's logging provider.
     *
     * @return The controller's logging provider
     */
    protected String getLoggingProvider() {
        return SimpleLog.class.getName();
    }

    /**
     * Gets the controller's authentication provider.
     *
     * @return The controller's authentication provider
     */
    protected String getAuthenticationProvider() {
        return SimpleAuth.class.getName();
    }

    /**
     * Gets the test suite.
     *
     * @return The test suite
     */
    protected RequestTestSuite getTestSuite() {
        return testSuite;
    }

    /**
     * <p>Initializes a test suite, using a local client and an HTTP client</p>
     *
     * @throws Exception if an unexpected exception occurs
     */
    @Before
    public void setUp() throws Exception {
        List<Client> clients = new LinkedList<Client>();
        Configuration config = new Configuration();
        config.setLoggingProvider(getLoggingProvider());
        config.setAuthenticationProvider(getAuthenticationProvider());
        clients.add(new LocalClient(packagePath, config));
        clients.add(new HttpClient("http://" + address + ":" + port + context));
        testSuite = new RequestTestSuite(clients);
    }

    /**
     * Tests a simple request.
     *
     * @throws Exception if an unexpected exception occurs
     */
    @Test
    public void testBasics() throws Exception {
        ClientRequest request = new ClientRequest(HttpMethod.GET, "/integration/simple/public");
        testSuite.runTest(request, HttpStatusCode.OK.getCode());
    }

    /**
     * Tests a request with authorization.
     *
     * @throws Exception if an unexpected exception occurs
     */
    @Test
    public void testAuthentication() throws Exception {
        ClientRequest request = new ClientRequest(HttpMethod.GET, "/integration/simple/mocks");
        request.setAuthorization("admin", "admin");
        testSuite.runTest(request, HttpStatusCode.OK.getCode());
    }

}
