/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.example.persistence;

import org.hibernate.cfg.DefaultNamingStrategy;

import java.util.regex.Pattern;

/**
 * The naming strategy for database tables depending on
 * class and property names.
 *
 * @author Christian Bick
 * @since 05.08.11
 */
public class DbNamingStrategy extends DefaultNamingStrategy {

    /**
     * Converts a class name into a table name.
     *
     *
     * @param className The class name
     * @return The table name
     */
    @Override
    public String classToTableName (String className) {
        if (className.isEmpty()) {
            return className;
        }
        String head = className.substring(0, 1);
        String tail = className.substring(1, className.length());
        className = head.toLowerCase() + tail;
        return camelCaseToUnderScore(className);
    }

    /**
     * Converts a property name into a column name
     *
     * @param propertyName The property name
     * @return The column name
     */
    @Override
    public String propertyToColumnName (String propertyName) {
        return camelCaseToUnderScore(propertyName);
    }

    /**
     * Gets the foreign key column name of a property by only regarding
     * the referenced column name and applying
     * {@link #propertyToColumnName(String)} on it.
     *
     * @param propertyName The property name (not used)
     * @param propertyEntityName The property entity name (not used)
     * @param propertyTableName The property table name (not used)
     * @param referencedColumnName The referenced column name (used)
     * @return The foreign key column name
     */
    @Override
    public String foreignKeyColumnName(String propertyName, String propertyEntityName, String propertyTableName, String referencedColumnName) {
        return propertyToColumnName(referencedColumnName);
    }

    /**
     * <p>Converts a camel cased name into an underscored name.</p>
     * <p><b>Example:</b> myPropertyName -> my_property_name</p>
     *
     * @param input The camel cased input name
     * @return The underscored output name
     */
    protected String camelCaseToUnderScore(String input) {
        return Pattern.compile("[A-Z]").matcher(input).replaceAll("_$0").toLowerCase();
    }

}
