/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.essential.example.persistence;

import net.craftforge.essential.controller.ControllerException;
import net.craftforge.essential.controller.constants.HttpStatusCode;
import net.craftforge.reflection.managers.ReferenceManager;

import javax.persistence.EntityManager;

/**
 * <p>A CRUD DAO generalizes the simple create, read, update
 * and delete methods for entities of a given type. Each
 * operations opens, commits and closes transactions itself.</p>
 *
 * <p>Do not use this DAO for a sequence of operations if they
 * shall share one transaction.</p>
 *
 * @author Christian Bick
 * @since 09.08.11
 *
 * @param <T> The entity type
 */
public class CrudDao<T> {

    private EntityManager em;

    /**
     * Initializes the CRUD DAO with the entity manager to
     * use for its operations.
     *
     * @param entityManager The entity manager
     */
    public CrudDao(EntityManager entityManager) {
        this.em = entityManager;
    }

    /**
     * Creates an entity by making it persistent.
     *
     * @param entity The entity to persist
     * @return The now persistent entity
     * @throws ControllerException if the entity has already been persistent
     */
    @SuppressWarnings("unchecked")
    public T createEntity(T entity) throws ControllerException {
        em.getTransaction().begin();
        em.persist(entity);
        em.getTransaction().commit();
        em.detach(entity);
        return ReferenceManager.getInstance((Class<T>) entity.getClass()).nullifyActualReferences(entity);
    }

    /**
     * Updates an entity by merging it into the persistence context.
     *
     * @param id The entity id
     * @param entity The entity to merge
     * @return The merged entity
     * @throws ControllerException if no entity with the given id has already been persistent
     */
    @SuppressWarnings("unchecked")
    public T updateEntity(Integer id, T entity) throws ControllerException {
        T existingEntity = em.find((Class<T>)entity.getClass(), id);
        requireExistence(existingEntity);
        em.getTransaction().begin();
        T result = em.merge(entity);
        em.getTransaction().commit();
        em.detach(result);
        return ReferenceManager.getInstance((Class<T>) entity.getClass()).nullifyActualReferences(result);
    }

    /**
     * Deletes an entity of the given class with the specified id
     * by removing it from the persistence context.
     *
     * @param clazz The entity class
     * @param id The entity id
     * @throws ControllerException if no entity with the given id has already been persistent
     */
    public void deleteEntity(Class<T> clazz, Integer id) throws ControllerException {
        T entity = em.find(clazz, id);
        requireExistence(entity);
        em.getTransaction().begin();
        em.remove(entity);
        em.getTransaction().commit();
    }

    /**
     * Reads an entity of the given class with the specified id by
     * finding it in the the persistence context.
     *
     * @param clazz The entity class
     * @param id The entity id
     * @return The found entity
     * @throws ControllerException if no entity with the given id has already been persistent
     */
    public T readEntity(Class<T> clazz, Integer id) throws ControllerException {
        T entity = em.find(clazz, id);
        requireExistence(entity);
        em.detach(entity);
        return ReferenceManager.getInstance(clazz).nullifyActualReferences(entity);
    }

    /**
     * Requires the existence of an entity.
     *
     * @param entity The entity
     * @throws ControllerException if the entity is null
     */
    protected void requireExistence(T entity) throws ControllerException {
        if (entity == null) {
            throw new ControllerException("Entity not found", HttpStatusCode.NotFound);
        }
    }

}
