/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.commons.database.memory;

import org.hibernate.ejb.Ejb3Configuration;

import java.io.IOException;
import java.util.concurrent.ConcurrentHashMap;

/**
 * <p>An in-memory database manager manages several in-memory database instances
 * of the same database setup. This allows test- and demo-databases which can be
 * booted from the scratch with the same scheme and data but are exclusive for users
 * or processes although running at the same time.</p>
 *
 * <p>The database in-memory manager is unique in respect an SQL setup-script. All
 * databases are setup using this setup-script. An in-memory database itself is
 * unique in respect to its name within an in-memory manager.</p>
 *
 * <p>It is to the user of this class to find an appropriate way of naming the
 * databases in a way which makes them exclusive in the context of their purpose.
 * Examples: thread id, user id, remote ip address (Base64 encoded).</p>
 *
 * @author Christian Bick
 * @since 02.08.11
 */
public class InMemoryDbManager {

    /**
     * The in-memory database manager instances
     */
    private static ConcurrentHashMap<String, InMemoryDbManager> instances = new ConcurrentHashMap<String, InMemoryDbManager>();

    /**
     * Gets or creates and gets The in-memory database manager associated to the given database
     * setup script resource.
     *
     * @param dbSetupScriptResource The database setup script resource
     * @param configuration The EJB3 configuration for the entity-manager factory creation
     * @return The in-memory database manager
     */
    public static InMemoryDbManager getInstance(String dbSetupScriptResource, Ejb3Configuration configuration) {
        instances.putIfAbsent(dbSetupScriptResource, new InMemoryDbManager(dbSetupScriptResource, configuration));
        return instances.get(dbSetupScriptResource);
    }

    /**
     * The EJB configuration used for entity manager factories associated to in-memory databases
     * managed.
     */
    private Ejb3Configuration configuration;

    /**
     * The in-memory databases already created
     */
    private ConcurrentHashMap<String, InMemoryDb> inMemoryDatabasesCreated = new ConcurrentHashMap<String, InMemoryDb>();

    /**
     * The in-memory databases in creation
     */
    private ConcurrentHashMap<String, InMemoryDb> inMemoryDatabasesInCreation = new ConcurrentHashMap<String, InMemoryDb>();

    /**
     * The database creation monitor which makes sure that only one process
     * at the same time is handling the database creation organization
     */
    private final Object dbCreationMonitor = new Object();

    /**
     * The database setup statements
     */
    private String[] dbSetupStatements;

    /**
     * Initializes a in-memory database manager using the given database setup script
     * resource for setting up the newly created managed databases.
     *
     * @param dbSetupScriptResource The database setup script resource
     * @param configuration The EJB3 configuration for the entity-manager factory creation
     */
    private InMemoryDbManager(String dbSetupScriptResource, Ejb3Configuration configuration) {
        this.configuration = configuration;
        try {
            dbSetupStatements = InMemoryUtils.getSqlStatementsFromResource(dbSetupScriptResource);
        } catch (IOException e) {
            throw new RuntimeException(e);
        }
    }

    /**
     * Gets an in-memory database with the given name. The database instance
     * will be created if it has not already been existing.
     *
     * @param databaseName The database name
     * @return The in-memory database
     */
    public InMemoryDb getInMemoryDb(String databaseName) {
        return getInMemoryDb(databaseName, InMemoryDb.DEFAULT_INITIAL_TIMEOUT, InMemoryDb.DEFAULT_TIMEOUT_CHECK_INTERVAL);
    }

    /**
     * Gets an in-memory database with the given name. The database instance
     * will be created if it has not already been existing. The instance will be destroyed
     * after the given timeout and if the timeout has already been reaches is checked
     * in the given interval.
     *
     * @param databaseName The database name
     * @param timeout The timeout (in milliseconds)
     * @param timeoutCheckInterval (in milliseconds)
     * @return The in-memory database
     */
    public InMemoryDb getInMemoryDb(String databaseName, long timeout, long timeoutCheckInterval) {
        if (! inMemoryDatabasesCreated.containsKey(databaseName)) {
            InMemoryDb inMemoryDbInCreation;
            // If the db is not contained within created one, a new database is created in a synchronized block
            synchronized (dbCreationMonitor) {
                if (inMemoryDatabasesCreated.containsKey(databaseName)) {
                    // The very unlikely but possible case: the database was creation has finished in the meantime
                    return inMemoryDatabasesCreated.get(databaseName);
                } else if (inMemoryDatabasesInCreation.containsKey(databaseName)) {
                    // The more likely case: the database creation process has already been started but not finished
                    inMemoryDbInCreation = inMemoryDatabasesInCreation.get(databaseName);
                } else {
                    // The most common case: the database creation process has not started yet - so it is started here
                    inMemoryDbInCreation = new InMemoryDb(this, databaseName, timeout, timeoutCheckInterval);
                    inMemoryDbInCreation.start();
                    inMemoryDatabasesInCreation.put(databaseName, inMemoryDbInCreation);
                }
            }
            // Waiting for the database creation process to finish outside the monitor
            inMemoryDbInCreation.waitForCreationCompleted();
            // After creation is completed a synchronized block using the same monitor as before is entered to avoid
            // confusion between entering database requests and exiting one
            synchronized (dbCreationMonitor) {
                // After the creation process has finished, the database is added to the created map
                inMemoryDatabasesCreated.putIfAbsent(databaseName, inMemoryDbInCreation);
                // The database is no more in creation process, so it must be removed
                inMemoryDatabasesInCreation.remove(databaseName);
            }
        }
        return inMemoryDatabasesCreated.get(databaseName);
    }

    /**
     * Whether an in-memory database with the given database name is already
     * managed by this manager or not.
     *
     * @param databaseName The database name
     * @return Whether it is already managed or not
     */
    public boolean existsInMemoryDb(String databaseName) {
        return inMemoryDatabasesCreated.containsKey(databaseName)
                || inMemoryDatabasesInCreation.containsKey(databaseName);
    }

    /**
     * Gets the EJB3 configuration used by this in-memory manager to
     * create entity-manager factories.
     *
     * @return The EJB3 configuration
     */
    public Ejb3Configuration getConfiguration() {
        return configuration;
    }

    /**
     * Gets the database setup statements used to set up in-memory
     * databases instances.
     *
     * @return The SQL statements
     */
    protected String[] getDbSetupStatements() {
        return dbSetupStatements;
    }

    /**
     * Removes the in-memory database with the given database name
     * from the entity manager.
     *
     * @param databaseName The database name
     */
    protected void removeInMemoryDb(String databaseName) {
        inMemoryDatabasesCreated.remove(databaseName);
    }
}
