/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.commons.database.memory;

import org.hibernate.Session;
import org.hibernate.SessionFactory;
import org.hibernate.cfg.Configuration;
import org.hibernate.ejb.Ejb3Configuration;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import javax.persistence.EntityManagerFactory;
import java.sql.DriverManager;
import java.sql.SQLException;

/**
 * An in-memory db is a wrapper for holding derby in-memory database
 * instances. An in-memory db will be destroyed after a certain timeout
 * has been reached, so it acts as temporary database which is only
 * used for a certain time.
 *
 * The timeout is reset to its initial value with entity manager factory
 * call. So an in-memory database keeps alive as long as it is frequently
 * used.
 *
 * @author Christian Bick
 * @since 08.08.11
 */
public class InMemoryDb extends Thread {

    private static Logger logger = LoggerFactory.getLogger(InMemoryDb.class);

    /**
     * Default maximum idle time before a database is dropped: 1 hour.
     */
    public static final long DEFAULT_INITIAL_TIMEOUT = 3600000;

    /**
     * Default update interval to check if the database should be dropped: 10 seconds.
     */
    public static final long DEFAULT_TIMEOUT_CHECK_INTERVAL = 10000;

    /**
     * The monitor for the database creation. Assures that the first
     * process attempting to use this database triggers its creation
     * and that all following processes wait until the creation process
     * is finished.
     */
    private final Object creationCompleted = new Object();

    /**
     * Whether the database has already been created ot is still
     * in the creation process
     */
    private boolean databaseCreated = false;

    /**
     * The name of this database
     */
    private String databaseName;

    /**
     * The interval between two timeout checks
     */
    private long timeoutCheckInterval;

    /**
     * The remaining time till timeout
     */
    private long timeToTimeout;

    /**
     * The initial time till timeout
     */
    private long timeout;

    /**
     * The in-memory database manager for this database
     */
    private InMemoryDbManager manager;

    /**
     * The entity-manager factory creating entity-managers
     * for entities persisted in this database
     */
    private EntityManagerFactory entityManagerFactory;

    /**
     * Create a new in memory-db with the given database name. Intended to be called by
     * the managing in-memory manager only. The database will be destroyed after
     * the no new entity manager was called within the given timeout. If
     * the timeout has already been reached is checked in the given interval.
     *
     * @param manager The in-memory database manager for this database
     * @param databaseName The database name
     * @param timeout The timeout in milliseconds
     * @param timeoutCheckInterval The interval to check for timeout in milliseconds
     */
    protected InMemoryDb(InMemoryDbManager manager, String databaseName, long timeout, long timeoutCheckInterval) {
        this.manager = manager;
        this.databaseName = databaseName;
        this.timeout = timeout;
        this.timeToTimeout = timeout;
        this.timeoutCheckInterval = timeoutCheckInterval;
    }

    /**
     * Sets up the database and then waits until the timeout is reached
     */
    @Override
    public void run() {
        setupDatabase();
        waitForTimeout();
    }

    /**
     * Gets the entity-manager factory creating entity-managers
     * for entities persisted in this database. Calling this
     * method resets the remaining timeout its initial value.
     *
     * @return The entity manager factory
     */
    public EntityManagerFactory getEntityManagerFactory() {
        timeToTimeout = timeout;
        return entityManagerFactory;
    }

    /**
     * The first process calling this method will trigger the creation
     * of a Derby in-memory database. All following database will
     * be queued for waiting until the creation process completed.
     *
     * @return Whether the database has already been created or not
     */
    protected boolean waitForCreationCompleted() {
        synchronized (creationCompleted) {
            if (databaseCreated) {
                return true;
            }
            try {
                if (getState().equals(State.NEW)) {
                    start();
                    creationCompleted.wait();
                    return false;
                } else {
                    creationCompleted.wait();
                    return false;
                }
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
        }
    }

    /**
     * Sets up an associated Derby in-memory database instance and executes the SQL
     * setup script of the in-memory database manager.
     */
    protected void setupDatabase() {
        logger.info("Setting up database {}", databaseName);

        // Builds and opens the database session
        SessionFactory sessionFactory = buildSessionFactory();
        Session session = sessionFactory.openSession();

        // Sets up the initial database
        session.getTransaction().begin();
        executeStatements(session, manager.getDbSetupStatements());
        session.getTransaction().commit();
        logger.info("Database {} set up", databaseName);
        session.close();

        // Builds the entity manager factory responsible for this database
        entityManagerFactory = buildEntityManagerFactory();
        synchronized (creationCompleted) {
            databaseCreated = true;
            creationCompleted.notifyAll();
        }
    }

    /**
     * Waits for the timeout to be reaches by frequently
     * checking if the timeout has already been reached an
     * decreasing the remaining time.
     */
    protected void waitForTimeout() {
        while (timeToTimeout > 0) {
            try {
                Thread.sleep(timeoutCheckInterval);
            } catch (InterruptedException e) {
                throw new RuntimeException(e);
            }
            timeToTimeout -= timeoutCheckInterval;
        }
        destroyDatabase();
    }

    /**
     * Destroys an associated Derby in-memory database instance.
     */
    protected void destroyDatabase() {
        logger.info("Destroying database {}", databaseName);
        try {
            DriverManager.getConnection("jdbc:derby:memory:" + databaseName + ";drop=true");
        } catch (SQLException e) {
            SQLException nextException = e;
            boolean isExpectedException = false;
            while (nextException != null && ! isExpectedException) {
                isExpectedException = e.getMessage().contains("dropped");
                nextException = e.getNextException();
            }
            if (! isExpectedException) {
                throw new RuntimeException("Failed to destroy database", e);
            }
        }
        logger.info("Database {} destroyed", databaseName);
        manager.removeInMemoryDb(databaseName);
    }

    /**
     * Builds an entity manager factory associated to this in-memory database using
     * the EJB configuration of the in-memory database manager.
     *
     * @return The entity manager factory
     */
    protected EntityManagerFactory buildEntityManagerFactory() {
        Ejb3Configuration cfg = manager.getConfiguration();
        cfg.setProperty("hibernate.connection.url", "jdbc:derby:memory:" + databaseName +";create=true");
        return cfg.buildEntityManagerFactory();
    }

    /**
     * Builds a hibernate session factory associated to this in-memory database
     * for the database setup.
     *
     * @return The hibernate session factory
     */
    protected SessionFactory buildSessionFactory() {
        Configuration cfg = new Configuration();
        cfg.setProperty("hibernate.connection.url", "jdbc:derby:memory:" + databaseName +";create=true");
        return cfg.buildSessionFactory();
    }

    /**
     * Executes all given SQL statements using the given hibernate session.
     *
     * @param session The hibernate session
     * @param statements The SQL statements
     */
    protected void executeStatements(Session session, String[] statements) {
        for (String statement : statements) {
            if (statement.isEmpty()) {
                continue;
            }
            session.createSQLQuery(statement).executeUpdate();
        }
    }
}
