/*
 * This file is part of essential (http://essential.craftforge.net).
 *
 *     Essential is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU Lesser Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     Essential is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with Foobar.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Copyright (c) 2011 Christian Bick.
 */

package net.craftforge.reflection.utils;

import java.lang.reflect.Field;
import java.sql.Date;
import java.util.*;

/**
 *
 *
 * @author Christian Bick
 * @since 02.12.2010
 */
public class ReflUtils {

    /**
     * <p>Converts a primitive class to its equivalent object class.
     * Will return the original class if it is no primitive.</p>
     * Examples: int -> Integer, char -> Character, Float -> Float
     *
     * @param primitiveClass The primitive class
     * @return The equivalent object class
     */
    public static Class<?> primitiveToObject(Class<?> primitiveClass) {
        Class<?> objectClass;
        if (primitiveClass.equals(boolean.class)) {
            objectClass = Boolean.class;
        } else if (primitiveClass.equals(byte.class)) {
            objectClass = Byte.class;
        } else if (primitiveClass.equals(short.class)) {
            objectClass = Short.class;
        } else if (primitiveClass.equals(int.class)) {
            objectClass = Integer.class;
        } else if (primitiveClass.equals(long.class)) {
            objectClass = Long.class;
        } else if (primitiveClass.equals(double.class)) {
            objectClass = Double.class;
        } else if (primitiveClass.equals(float.class)) {
            objectClass = Float.class;
        } else if (primitiveClass.equals(char.class)) {
            objectClass = Character.class;
        } else {
            objectClass = primitiveClass;
        }
        return objectClass;
    }

    /**
     * Checks if the class a primitive, including primitive objects
     * (Integer, Long etc.).
     *
     * @param c The class
     * @return true, if the class is a primitive, false otherwise
     */
    public static boolean isPrimitive(Class<?> c) {
        return c.isPrimitive()
                || c.equals(Boolean.class)
                || c.equals(Character.class)
                || c.equals(Short.class)
                || c.equals(Integer.class)
                || c.equals(Byte.class)
                || c.equals(Long.class)
                || c.equals(Double.class)
                || c.equals(Float.class);    
    }

    /**
     * Checks if the class represents a virtual primitive. Virtual primitives are a
     * all real primitives, their object representation (Integer, Long etc.)
     * as well as {@link String} and {@link Date}.
     *
     * @param c The class
     * @return true, if the class is a simple data type, false otherwise
     */
    public static boolean isVirtualPrimitive(Class<?> c) {
        return isPrimitive(c)
                || String.class.isAssignableFrom(c)
                || Date.class.isAssignableFrom(c);
    }

    /**
     * Checks if the class represents an actual reference. Actual references are
     * represented by all classes that are not virtual primitives. So this method
     * ignores references to primitive objects (Integer, Long, etc.) as well as
     * to String and Date.
     *
     * @param c The class
     * @return true, if the class is not a simple data type, false otherwise
     */
    public static boolean isActualReference(Class<?> c) {
        return ! isVirtualPrimitive(c);
    }

    /**
     * Checks if the class is a collection (implements interface collection).
     *
     * @param c The class
     * @return true, if the field type is a collection, false otherwise
     */
    public static boolean isCollection(Class<?> c) {
        return Collection.class.isAssignableFrom(c);
    }

    /**
     * Checks if the field is a data field.
     *
     * @param field The field
     * @return true, if the field is a data field but no collection, false otherwise
     */
    public static boolean isSingleVirtualPrimitive(Field field) {
        return ! isCollection(field.getType()) && ! isActualReference(field.getType());
    }

    /**
     * Checks if the field is a reference field.
     *
     * @param field The field
     * @return true, if the field is a reference field but no collection, false otherwise
     */
    public static boolean isSingleActualReference(Field field) {
        return ! isCollection(field.getType()) && isActualReference(field.getType());
    }

    /**
     * Checks if the reference is a collection of data fields.
     *
     * @param field The field
     * @return true, if the field is a collection of data fields, false otherwise
     */
    public static boolean isCollectionData(Field field) {
        /* return isCollection(field.getType()) && ! isReference(field.getType().getComponentType()); */
        return false;
    }

    /**
     * Checks if the collection is a collection of references
     *
     * @param field The field
     * @return true, if the field is a collection of reference fields, false otherwise
     */
    public static boolean isCollectionReference(Field field) {
        return isCollection(field.getType()) /* && isReference(field.getType().getComponentType())*/;
    }

    /**
	 * Builds the name of the get-method for the given field name.
	 * 
	 * @param fieldName The field name
	 * @return The get-method's name
	 */
	public static String getterName(String fieldName) {
		String first_lc = fieldName.substring(0, 1);
		String first_uc = first_lc.toUpperCase();
		fieldName = first_uc + fieldName.substring(1, fieldName.length());
		return "get" + fieldName;
	}

	/**
	 * Builds the name of the set-method for the given field name.
	 * 
	 * @param fieldName The field name
	 * @return The set-method's name
	 */
	public static String setterName(String fieldName) {
		String first_lc = fieldName.substring(0, 1);
		String first_uc = first_lc.toUpperCase();
		fieldName = first_uc + fieldName.substring(1, fieldName.length());
		return "set" + fieldName;
	}

    /**
     * Creates a parametrized collection of the given collection type (e.g. List) using
     * the given generic type as generic parameter.<br>
     * <br>
     * Example: createCollection(List.class, String.class) will return a collection
     * of type List&lt;String&gt;
     *
     * @param collectionType The collection type
     * @param <T> The class parameter
     * @return The created collection
     */
    @SuppressWarnings("unchecked")
    public static <T> Collection<T> createCollection(Class<?> collectionType) {
        if (collectionType.equals(Collection.class)) {
            return new ArrayList<T>();
        } else if(collectionType.equals(List.class)) {
            return new ArrayList<T>();
        } else if (collectionType.equals(Set.class)) {
            return new HashSet<T>();
        } else if (collectionType.equals(Queue.class)) {
            return new LinkedList<T>();
        }
        return null;
    }
}
